package edu.ucc.utils;

import edu.ucc.network.devices.BaseStation;
import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.Location;
import edu.ucc.utils.kmeans.Centroid;
import edu.ucc.utils.kmeans.EuclideanDistance;
import edu.ucc.utils.kmeans.KMeans;
import edu.ucc.utils.kmeans.Record;

import java.util.*;

public class HierarchyUtils {
    public static BaseStation getBaseStationForLocation(Location location, List<BaseStation> baseStations, double bsRange) {
        List<BaseStation> closestBaseStationsWithinRange = getClosestBaseStationsWithinRange(location, baseStations, bsRange);
        return closestBaseStationsWithinRange.get(0);
    }

    private static List<BaseStation> getClosestBaseStationsWithinRange(Location location, List<BaseStation> baseStations, double bsRadio) {
        class SortingHelper {
            double distance;
            BaseStation baseStation;

            SortingHelper(double distance, BaseStation baseStation) {
                this.distance = distance;
                this.baseStation = baseStation;
            }
        }

        List<SortingHelper> possibleBss = new ArrayList<>();
        for (BaseStation bs : baseStations) {
            double distance = location.distanceTo(bs.getLocation());
            if (distance <= bsRadio) {
                possibleBss.add(new SortingHelper(distance, bs));
            }
        }

        possibleBss.sort(Comparator.comparingDouble(o -> o.distance));
        List<BaseStation> closestBaseStations = new ArrayList<>();
        possibleBss.forEach(o -> closestBaseStations.add(o.baseStation));
        return closestBaseStations;
    }

    public static Map<Location, List<BaseStation>> launchKMeans(List<BaseStation> baseStations, int numEdgeServers) {
        Map<Location, List<BaseStation>> newOutcome = new LinkedHashMap<>();
        List<Record> records = new ArrayList<>();
        for (Host bs : baseStations) {
            Map<String, Double> features = new LinkedHashMap<>();
            features.put("latitude", bs.getLocation().getLatitude());
            features.put("longitude", bs.getLocation().getLongitude());
            Record record = new Record("A base station", features);
            records.add(record);
        }

        Map<Centroid, List<Record>> fit = KMeans.fit(records, numEdgeServers, new EuclideanDistance(), 1000);
        fit.forEach((edgeLocation, bssLocations) -> {
            final Map<String, Double> coordinates = edgeLocation.getCoordinates();
            final double latitude = coordinates.get("latitude");
            final double longitude = coordinates.get("longitude");
            Location locationEdge = new Location(latitude, longitude);

            List<BaseStation> bssThisCentroid = new ArrayList<>();
            for (Record record : bssLocations) {
                final Map<String, Double> recordFeatures = record.getFeatures();
                double bsLatitude = recordFeatures.get("latitude");
                double bsLongitude = recordFeatures.get("longitude");
                BaseStation bs = getBaseStationFromLocation(bsLatitude, bsLongitude, baseStations);
                bssThisCentroid.add(bs);
            }
            newOutcome.put(locationEdge, bssThisCentroid);

        });
        return newOutcome;
    }

    public static BaseStation getBaseStationFromLocation(double bsLatitude, double bsLongitude, List<BaseStation> baseStations) {
        Location bsLocation = new Location(bsLatitude, bsLongitude);
        for (BaseStation bs : baseStations) {
            Location thisBsLocation = bs.getLocation();
            if (bsLocation.getLatitude() == thisBsLocation.getLatitude() && bsLocation.getLongitude() == thisBsLocation.getLongitude()) {
                return bs;
            }
        }
        // If this fails because == on doubles, use the getCurrentBaseStation() from Utils instead. Should work
        throw new RuntimeException("NO corresponding BS. Maybe doing == on doubles is not a good idea");
    }
}
