package edu.ucc.testbedinterface;

import edu.ucc.core.events.simulationevents.*;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.network.devices.AdvancedHost;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;
import edu.ucc.statisticscollection.Request;
import edu.ucc.statisticscollection.StatisticsCollector;

import java.util.List;

/**
 * This class plugs the edu.ucc.network testbed to the statistics collector and to our solution to collect input events that are
 * the raw material for analytics.
 */
public class TestbedBroker implements ITestbedEvents {
    private final StatisticsCollector statisticsCollector;
    private final ObservableTestbedEventListener observableTestbedEventListener;

    public TestbedBroker(StatisticsCollector statisticsCollector, ObservableTestbedEventListener observableTestbedEventListener) {
        this.statisticsCollector = statisticsCollector;
        this.observableTestbedEventListener = observableTestbedEventListener;
    }

    @Override
    public void onPullRequestReceived(Host host, PullRequestEvent pullRequestEvent) {
        statisticsCollector.onPullRequestReceived(host, pullRequestEvent);
    }

    @Override
    public void onAckReceived(Host host, AckReceivedEvent ackReceivedEvent) {
        statisticsCollector.onAckReceived(host, ackReceivedEvent);
    }

    @Override
    public void onChunkReceived(Host host, ChunkReceivedEvent chunkReceivedEvent) {
        final RequestEvent requestEvent = chunkReceivedEvent.getChunk().getRequestEvent();
        if (requestEvent instanceof PushDataRequestEvent || requestEvent instanceof PushFileRequestEvent
                || requestEvent instanceof PullDataRequestEvent || requestEvent instanceof PullFileRequestEvent) {
            statisticsCollector.onChunkReceived(host, chunkReceivedEvent);
        }
        if (isAdvancedHost(host)) {
            final ContentChunk chunk = chunkReceivedEvent.getChunk();
            final Request requestInfo = statisticsCollector.findRequest(host, chunk.getRequestEvent().getRequestId());
            if (chunk.isLastChunk() && requestInfo.getReceivedChunks() == chunk.getTotalChunks()) {
                ContentReceivedEvent contentReceivedEvent = new ContentReceivedEvent(
                        chunkReceivedEvent.getEventOriginHost(),
                        chunkReceivedEvent.getEventDestinationHost(),
                        chunkReceivedEvent.getTimestamp(),
                        chunk.getRequestEvent().getAppId(),
                        chunk.getRequestEvent().getRequestId(),
                        chunk.getRequestEvent().getReferredContent(),
                        chunk.getRequestEvent()
                );
                observableTestbedEventListener.onContentReceptionCompleted(host, contentReceivedEvent);
            }
        }
    }

    @Override
    public void onPushRequestReceived(Host host, PushRequestEvent pushRequestEvent) {
        if (pushRequestEvent instanceof PushDataRequestEvent || pushRequestEvent instanceof PushFileRequestEvent) {
            statisticsCollector.onPushRequestReceived(host, pushRequestEvent);
        }
    }


    @Override
    public void onUserEquipmentHandover(Host host, HandoverEvent handoverEvent) {
        statisticsCollector.onUserEquipmentHandover(host, handoverEvent);
        if (isAdvancedHost(host)) {
            observableTestbedEventListener.onUserEquipmentHandover(host, handoverEvent);

            if (handoverEvent.isIntraHandover()) {
                host.getFileSystem().updateURIForMovingUEData(handoverEvent.getMovingUE());
            }
        }
    }

    private boolean isAdvancedHost(Host host) {
        // return host instanceof EdgeServer || host instanceof CloudUnit;
        return host instanceof AdvancedHost;
    }

    @Override
    public void onChunkTransmittedUpwards(Host host, ContentChunk chunk) {
        statisticsCollector.onChunkTransmittedUpwards(host, chunk);
    }

    @Override
    public void onChunkTransmittedDownwards(Host host, ContentChunk chunk, double currentTime) {
        statisticsCollector.onChunkTransmittedDownwards(host, chunk, currentTime);
    }

    @Override
    public void onChunkTransmittedToSibling(Host host, ContentChunk chunk) {
        statisticsCollector.onChunkTransmittedToSibling(host, chunk);
    }

    @Override
    public void onPullRequestServedLocally(EdgeServer edgeServer, PullRequestEvent pullRequestEvent) {
        statisticsCollector.onPullRequestServedLocally(edgeServer, pullRequestEvent);
        observableTestbedEventListener.onPullRequestServedLocally(edgeServer, pullRequestEvent);
    }

    @Override
    public boolean isAppDeployedHere(Host host, int appId) {
        return observableTestbedEventListener.isAppDeployedHere(host, appId);
    }

    @Override
    public Host findDestinationForChunkAtServerWithoutApp(ContentChunk chunk, EdgeServer edgeServer, int appId) {
        return observableTestbedEventListener.findDestinationForChunkAtServerWithoutApp(chunk, edgeServer, appId);
    }

    @Override
    public void onPrefetchRequestReceived(Host host, PrefetchRequestEvent prefetchRequestEvent) {
        if (host instanceof CloudUnit) {
            observableTestbedEventListener.onPrefetchRequestReceived(host, prefetchRequestEvent);
        } else {
            throw new RuntimeException("A prefetch event has been received by a host that is not the cloud");
        }
    }

    @Override
    public void registerInitialMobilityEvent(List<HandoverEvent> initialMobilityEvents) {
        observableTestbedEventListener.registerInitialMobilityEvent(initialMobilityEvents);
    }
}
