package edu.ucc.statisticscollection;

import edu.ucc.core.events.simulationevents.*;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.entities.Content;
import edu.ucc.entities.Data;
import edu.ucc.entities.File;
import edu.ucc.network.devices.*;
import edu.ucc.testbedinterface.ITestbedEvents;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static edu.ucc.utils.Logging.*;
import static edu.ucc.utils.TransmissionUtils.getNumberOfChunksForContent;

public class StatisticsCollector implements ITestbedEvents {
    @Override
    public void onPullRequestServedLocally(EdgeServer edgeServer, PullRequestEvent pullRequestEvent) {
        // if (edgeServer == pullRequestEvent.getPullInitiatorHost()) {
        //     onPullRequestReceivedAtPullInitiatorHost(edgeServer, pullRequestEvent);
        // } else if (edgeServer == pullRequestEvent.getPullDestinationHost()) {
        //    onPullRequestReceivedAtPullDestinationHost(edgeServer, pullRequestEvent);
        // } else {
        final int chunkSize = edgeServer.getMtu();
        int totalChunks = getNumberOfChunksForContent(pullRequestEvent.getReferredContent(), chunkSize);
        contentScoreboard.createRequestIfNeeded(edgeServer, pullRequestEvent, totalChunks, pullRequestEvent.getTimestamp());
        logRequest(pullRequestEvent);
        // }
    }

    @Override
    public boolean isAppDeployedHere(Host host, int appId) {
        throw new RuntimeException("Of no use here");
    }

    @Override
    public Host findDestinationForChunkAtServerWithoutApp(ContentChunk chunk, EdgeServer edgeServer, int appId) {
        throw new RuntimeException("Of no use here");
    }

    @Override
    public void onPullRequestReceived(Host host, PullRequestEvent pullRequestEvent) {
        if (host == pullRequestEvent.getPullInitiatorHost()) {
            onPullRequestReceivedAtPullInitiatorHost(host, pullRequestEvent);
        } else if (host == pullRequestEvent.getPullDestinationHost()) {
            onPullRequestReceivedAtPullDestinationHost(host, pullRequestEvent);
        } else {
            // Lets create the request now. This means the object will be created the first time it passes through
            // the host
            final int chunkSize = host.getMtu();
            int totalChunks = getNumberOfChunksForContent(pullRequestEvent.getReferredContent(), chunkSize);
            contentScoreboard.createRequestIfNeeded(host, pullRequestEvent, totalChunks, pullRequestEvent.getTimestamp());
            //            logRequest(pullRequestEvent);
            // at the moment, the edge will process the request, but it is not the pull destination host (cloud) so,
            // how to handle this case? Run it, you will see how it does not fit in previous if's branches
        }
        logRequest(pullRequestEvent);
    }

    @Override
    public void onAckReceived(Host host, AckReceivedEvent ackReceivedEvent) {
        logAck(ackReceivedEvent);
    }

    @Override
    public void onChunkReceived(Host host, ChunkReceivedEvent chunkReceivedEvent) {
        final ContentChunk chunk = chunkReceivedEvent.getChunk();
        final double timestamp = chunkReceivedEvent.getTimestamp();
        final Request request = contentScoreboard.createRequestIfNeeded(host, chunk.getRequestEvent(),
                chunk.getTotalChunks(), timestamp);
        request.increaseReceivedChunks();
        if (chunk.isFirstChunk()) {
            request.setFirstChunkReceivedAt(timestamp);
        }
        if (chunk.isLastChunk()) {
            request.setCompletedAt(timestamp);
        }
        logChunk(chunkReceivedEvent);
    }

    @Override
    public void onPushRequestReceived(Host host, PushRequestEvent pushRequestEvent) {
        if (host == pushRequestEvent.getPushInitiatorHost()) {
            final Content content = pushRequestEvent.getReferredContent();
            final int chunkSize = cloud.getMtu();
            int totalChunks = getNumberOfChunksForContent(content, chunkSize);
            PushRequest request = createRequestAtPushInitiatorHost(host, pushRequestEvent, totalChunks);
            contentScoreboard.addContentRequest(host, request);

            logRequest(pushRequestEvent);
        } else printWarning("StatsCollector, A push request received not in push initiator host");
    }

    @Override
    public void onUserEquipmentHandover(Host host, HandoverEvent handoverEvent) {
        // NO statistics for handover events are currently collected
    }

    @Override
    public void onChunkTransmittedUpwards(Host host, ContentChunk chunk) {
        final Request request = contentScoreboard.findRequest(host, chunk.getRequestEvent().getRequestId());
        request.increaseTransmittedChunksUpwards();
    }

    @Override
    public void onChunkTransmittedDownwards(Host host, ContentChunk chunk, double currentTime) {
        // try to fix it quick... final Request request = scoreboard.findRequest(host, chunk.getRequestEvent().getRequestId());
        Request request = contentScoreboard.createRequestIfNeeded(host, chunk.getRequestEvent(), chunk.getTotalChunks(), currentTime);
        request.increaseTransmittedChunksDownwards();
    }

    @Override
    public void onChunkTransmittedToSibling(Host host, ContentChunk chunk) {
        final Request request = contentScoreboard.findRequest(host, chunk.getRequestEvent().getRequestId());
        request.increaseTransmittedChunksSiblings();
    }

    public Request findRequest(Host host, int requestId) {
        return contentScoreboard.findRequest(host, requestId);
    }

    @Override
    public void onPrefetchRequestReceived(Host host, PrefetchRequestEvent prefetchRequestEvent) {
        // No statistics for prefetch events are requested.
    }

    @Override
    public void registerInitialMobilityEvent(List<HandoverEvent> initialMobilityEvents) {
        // no functionality
    }

    static class ContentRequestScoreboard {
        HashMap<Host, Map<Integer, Request>> scoreboard;

        ContentRequestScoreboard() {
            this.scoreboard = new HashMap<>();
        }

        Request createRequestIfNeeded(Host host, RequestEvent requestEvent, int totalChunks, double timestamp) {
            Request request;
            Map<Integer, Request> requestsInHost = scoreboard.computeIfAbsent(host, v -> new HashMap<>());
            request = findRequest(host, requestEvent.getRequestId());
            if (request == null) {
                request = createRequest(host, requestEvent, totalChunks, timestamp);
            }
            requestsInHost.put(request.getRequestId(), request);
            return request;
        }

        private Request createRequest(Host host, RequestEvent requestEvent, int totalChunks, double timestamp) {
            Request newRequest = null;
            final int requestId = requestEvent.getRequestId();
            final Content referredContent = requestEvent.getReferredContent();

            if (requestEvent instanceof PullRequestEvent) {
                final PullRequestEvent asPullEvent = (PullRequestEvent) requestEvent;
                final Host requestingHost = asPullEvent.getPullInitiatorHost();
                final Host contentLocationHost = asPullEvent.getPullDestinationHost();
                if (asPullEvent instanceof PullFileRequestEvent) {
                    newRequest = new PullFileRequest(requestId, host, (File) referredContent, timestamp,
                            (UserEquipment) requestingHost, contentLocationHost, totalChunks);
                } else if (asPullEvent instanceof PullDataRequestEvent) {
                    newRequest = new PullDataRequest(requestId, host, (Data) referredContent, timestamp, requestingHost,
                            (UserEquipment) contentLocationHost, totalChunks);
                } else throw new RuntimeException("Unrecognized PullRequestEvent type");
            } else if (requestEvent instanceof PushRequestEvent) {
                final PushRequestEvent asPushEvent = (PushRequestEvent) requestEvent;
                final Host pushInitiatorHost = asPushEvent.getPushInitiatorHost();
                final Host pushDestinationHost = asPushEvent.getPushDestinationHost();
                if (asPushEvent instanceof PushFileRequestEvent) {
                    newRequest = new PushFileRequest(requestId, host, (File) referredContent, timestamp,
                            totalChunks, pushInitiatorHost, pushDestinationHost);
                } else if (asPushEvent instanceof PushDataRequestEvent) {
                    newRequest = new PushDataRequest(requestId, host, (Data) referredContent, timestamp, totalChunks,
                            pushInitiatorHost, pushDestinationHost);
                } else throw new RuntimeException("Unrecognized PushRequestEvent type");
            }
            return newRequest;
        }

        public Request findRequest(Host host, int requestId) {
            var requestsThisHost = scoreboard.get(host);
            return requestsThisHost.get(requestId);
        }

        void addContentRequest(Host host, Request request) {
            Map<Integer, Request> requestsInHost = scoreboard.computeIfAbsent(host, k -> new HashMap<>());
            requestsInHost.put(request.getRequestId(), request);
        }

        public List<Request> getRequestsInHost(Host host) {
            final Map<Integer, Request> requestsInHost = scoreboard.get(host);
            if (requestsInHost == null) {
                return new ArrayList<>();
            }
            return new ArrayList<>(requestsInHost.values());
        }

    }

    private final ContentRequestScoreboard contentScoreboard;
    private final CloudUnit cloud;
    private final List<EdgeServer> edgeServers;
    private final List<BaseStation> baseStations;
    private final List<UserEquipment> userEquipments;

    public StatisticsCollector(CloudUnit cloud, List<EdgeServer> edgeServers, List<BaseStation> baseStations,
                               List<UserEquipment> userEquipments) {
        contentScoreboard = new ContentRequestScoreboard();
        this.cloud = cloud;
        this.edgeServers = edgeServers;
        this.baseStations = baseStations;
        this.userEquipments = userEquipments;
    }


    private void onPullRequestReceivedAtPullDestinationHost(Host contentLocationHost, PullRequestEvent pullRequestEvent) {
        final Content content = pullRequestEvent.getReferredContent();
        final int chunkSize = contentLocationHost.getMtu();
        int totalChunks = getNumberOfChunksForContent(content, chunkSize);
        PullRequest request = createPullRequestAtDestinationHost(contentLocationHost, pullRequestEvent, totalChunks);

        final double timestamp = pullRequestEvent.getTimestamp();
        request.setFirstChunkReceivedAt(timestamp);
        request.setCompletedAt(timestamp);
        contentScoreboard.addContentRequest(contentLocationHost, request);
    }

    private void onPullRequestReceivedAtPullInitiatorHost(Host requestingHost, PullRequestEvent pullRequestEvent) {
        final Content content = pullRequestEvent.getReferredContent();
        final int chunkSize = cloud.getMtu();
        int totalChunks = getNumberOfChunksForContent(content, chunkSize);
        PullRequest request = createRequestAtPullInitiatorHost(requestingHost, pullRequestEvent, totalChunks);
        contentScoreboard.addContentRequest(requestingHost, request);
    }

    private PullRequest createPullRequestAtDestinationHost(Host contentLocationHost, PullRequestEvent requestEvent, int totalChunks) {
        PullRequest request;
        final Content content = requestEvent.getReferredContent();
        final double requestTimestamp = requestEvent.getTimestamp();
        Host requestingHost = requestEvent.getPullInitiatorHost();

        if (requestEvent instanceof PullFileRequestEvent) {
            // then it means that contentLocationHost is the cloud
            request = new PullFileRequest(requestEvent.getRequestId(), contentLocationHost,
                    (File) content, requestTimestamp,
                    (UserEquipment) requestingHost, contentLocationHost, totalChunks);
        } else if (requestEvent instanceof PullDataRequestEvent) {
            // Then contentLocationHost is a UE
            request = new PullDataRequest(requestEvent.getRequestId(), contentLocationHost, (Data) content,
                    requestTimestamp, requestingHost, (UserEquipment) contentLocationHost, totalChunks);
        } else throw new RuntimeException("Unknown request type");

        return request;
    }

    private PullRequest createRequestAtPullInitiatorHost(Host requestingHost, PullRequestEvent requestEvent, int totalChunks) {
        PullRequest request;
        final Content content = requestEvent.getReferredContent();
        final double requestTimestamp = requestEvent.getTimestamp();
        final Host contentLocationHost = requestEvent.getPullDestinationHost();

        if (requestEvent instanceof PullFileRequestEvent) {
            request = new PullFileRequest(requestEvent.getRequestId(), requestingHost,
                    (File) content, requestTimestamp,
                    (UserEquipment) requestingHost, contentLocationHost, totalChunks);
        } else if (requestEvent instanceof PullDataRequestEvent) {
            // Then the destination request is a UE
            request = new PullDataRequest(requestEvent.getRequestId(), requestingHost, (Data) content,
                    requestTimestamp, requestingHost, (UserEquipment) contentLocationHost, totalChunks);
        } else throw new RuntimeException("Unknown request type");

        return request;
    }

    private PushRequest createRequestAtPushInitiatorHost(Host pushInitiatorHost, PushRequestEvent pushEvent, int totalChunks) {
        PushRequest request;
        final Content content = pushEvent.getReferredContent();
        final double requestTimestamp = pushEvent.getTimestamp();
        final Host pushDestinationHost = pushEvent.getPushDestinationHost();
        final int requestId = pushEvent.getRequestId();
        if (pushEvent instanceof PushFileRequestEvent) {
            request = new PushFileRequest(requestId, pushInitiatorHost, (File) content, requestTimestamp, totalChunks,
                    pushInitiatorHost, pushDestinationHost);
        } else if (pushEvent instanceof PushDataRequestEvent) {
            request = new PushDataRequest(requestId, pushInitiatorHost, (Data) content, requestTimestamp, totalChunks,
                    pushInitiatorHost, pushDestinationHost);
        } else throw new RuntimeException("Unknown request type");

        final double timestamp = pushEvent.getTimestamp();
        request.setFirstChunkReceivedAt(timestamp);
        request.setCompletedAt(timestamp);
        return request;
    }

    public CloudUnit getCloud() {
        return cloud;
    }

    public List<EdgeServer> getEdgeServers() {
        return edgeServers;
    }

    public List<UserEquipment> getUserEquipments() {
        return userEquipments;
    }

    public ContentRequestScoreboard getContentScoreboard() {
        return contentScoreboard;
    }
}
