package edu.ucc.statisticscollection;

import edu.ucc.core.events.simulationevents.PrefetchRequestEvent;
import edu.ucc.core.events.simulationevents.PullDataRequestEvent;
import edu.ucc.core.events.simulationevents.PullFileRequestEvent;
import edu.ucc.core.events.simulationevents.RequestEvent;
import edu.ucc.entities.Content;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.UserEquipment;
import edu.ucc.workload.Workload;

import java.io.*;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static edu.ucc.utils.TransmissionConstants.CHUNK_SIZE;
import static edu.ucc.utils.TransmissionUtils.getNumberOfChunksForContent;

/**
 * Simple class to calculate statistics from the records collected by edge servers and the cloud.
 * This class can be modified to calculate further statistics. For now it collects speed and traffic information.
 */
public class StatisticsCalculator {
    private final StatisticsCollector statisticsCollector;
    private final double simulationTime;
    private final Workload workload;
    private final List<FileDownloadLatencyRecord> fileDownloadLatencyRecords;
    private final List<DataUploadLatencyRecord> dataUploadLatencyRecords;
    private final List<DataRequestFromUELatencyRecord> dataRequestFromUELatencyRecords;
    private final List<DataRequestFromCloudLatencyRecord> dataRequestFromCloudLatencyRecords;

    private TrafficRecord trafficRecord;


    public StatisticsCalculator(StatisticsCollector statisticsCollector, Workload workload, double simulationTime) {
        this.statisticsCollector = statisticsCollector;
        this.workload = workload;
        this.simulationTime = simulationTime;
        this.fileDownloadLatencyRecords = new ArrayList<>();
        this.dataUploadLatencyRecords = new ArrayList<>();
        dataRequestFromUELatencyRecords = new ArrayList<>();
        dataRequestFromCloudLatencyRecords = new ArrayList<>();

        this.calculateLatencies();
        this.calculateTraffic();
    }

    /**
     * Prints the statistics in the console
     */
    public void presentStatistics() {
        this.showLatencies();
        this.showTraffic();
    }

    /**
     * Writes the calculated statistics into a file. Parameters are just for naming the file
     * @param outputDirPath The full path+file name to write
     * @param numUes The number of UEs (just for naming the file)
     * @param numOfFileRequests The number of file requests (just for naming the file)
     * @param numOfGeneratedFiles The number of generated files (just for naming the file)
     * @param numOfUploads The number of uploads from UEs (just for naming the file)
     */
    public void writeStatistics(String outputDirPath, int numUes, int numOfFileRequests, int numOfGeneratedFiles, int numOfUploads) {
        // for downlink latencies - speeds
        File outputDir = new File(outputDirPath);
        outputDir.mkdirs();
        String filename = outputDirPath + String.format("downlink-latencies-%s-ues-%s-requests-%s-generated-files.csv", numUes, numOfFileRequests, numOfGeneratedFiles);
        File downlinkLatenciesFile = new File(filename);
        try (PrintWriter pw = new PrintWriter(new BufferedOutputStream(new FileOutputStream(downlinkLatenciesFile, false)))) {
            pw.println(FileDownloadLatencyRecord.getCsvHeader());
            for (FileDownloadLatencyRecord record : fileDownloadLatencyRecords) {
                pw.println(record.toCsv());
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }

        // for uplink latencies-speed
        filename = outputDirPath + String.format("uplink-latencies-%s-ues-%s-generated-uploads.csv", numUes, numOfUploads);
        File uplinkLatenciesFile = new File(filename);
        try (PrintWriter pw = new PrintWriter(new BufferedOutputStream(new FileOutputStream(uplinkLatenciesFile, false)))) {
            pw.println(DataUploadLatencyRecord.getCsvHeader());
            for (DataUploadLatencyRecord record : dataUploadLatencyRecords) {
                pw.println(record.toCsv());
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }

        // for downlink traffic
        filename = outputDirPath + String.format("downlink-traffic-%s-ues-%s-requests-%s-generated-files.txt", numUes, numOfFileRequests, numOfGeneratedFiles);
        File trafficRecordFile = new File(filename);
        try (PrintWriter pw = new PrintWriter(new BufferedOutputStream(new FileOutputStream(trafficRecordFile, false)))) {
            pw.println("Pull file requests to cloud: " + trafficRecord.getPullFileRequestsToCloud());
            pw.println("Chunks sent by cloud for pull file requests: " + trafficRecord.getChunksSentByCloudForPullRequests());
            pw.println("Pull file requests in workload: " + trafficRecord.getPullFileRequestsInWorkload());
            pw.println("Theoretical max chunks in pull file requests in workload: " + trafficRecord.getTheoreticalMaxChunksInPullFileRequests());
            pw.println("Pull file requests to ESs: " + trafficRecord.getPullFileRequestsToEdgeServers());
            pw.println("Chunks sent by ESs: " + trafficRecord.getChunksSentByEdgeServers());
            pw.println("Prefetch file requests in workload: " + trafficRecord.getPrefetchFileRequestsInWorkload());
            pw.println("Chunks in prefetch file requests in workload: " + trafficRecord.getChunksSentByCloudForPrefetchRequests());
            pw.println("Percentage of chunks sent by ESs (and not by cloud): " + trafficRecord.getPcChunksByESs());
            pw.println("Percentage of requests handled by ESs (and not by cloud): " + trafficRecord.getPcRequestsByESs());
            pw.println("Percentage of theoretical max traffic in the cloud - es segment: " + trafficRecord.getPcOfTheoreticalMaxTraffic());
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }

        // for simulation times
        try {
            final String timesFileName = outputDirPath +
                    String.format("simulation-time-%s-ues-%s-requests-%s-generated-files.txt", numUes, numOfFileRequests, numOfGeneratedFiles);
            PrintWriter pw = new PrintWriter(new BufferedWriter(new FileWriter(timesFileName, false)));
            pw.println(String.format("simulation time for %s requests: %s", numOfFileRequests, simulationTime));
            pw.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void calculateTraffic() {
        final List<PullRequest> requestedContents = getListOfPullFileRequestsInCloud();
        int fileRequestsToCloud = requestedContents.size();
        long chunksSentByCloudForPullRequests = requestedContents.stream().mapToLong(PullRequest::getTransmittedChunksDownwards).sum();
        int pullFileRequestsInWorkload = getListOfPullFileRequestsInWorkload().size();
        int theoreticalMaxChunksInPullFileRequests = calculateChunksInListOfPullRequestEvents();

        final List<PushFileRequest> prefetchFileRequests = getListOfPrefetchFileRequestsInCloud();
        long chunksSentByCloudForPrefetchRequests = prefetchFileRequests.stream().mapToLong(PushFileRequest::getTransmittedChunksDownwards).sum();
        int prefetchFileRequestsInWorkload = getListOfPrefetchFileRequestsInWorkload().size();
        int theoreticalMaxChunksInPrefetchRequests = calculateChunksInListOfPrefetchRequestEvents();

        // Get chunks and workload.requests in the edge servers.
        int fileRequestsToEdgeServers = getNumberOfPullFileRequestsToEdgeServers();
        long chunksSentByEdgeServers = getNumberOfChunksSentForPullFileRequestsByEdgeServers();

        final double pcChunksByESs = chunksSentByEdgeServers == 0 ? 0 : 100 * (double) chunksSentByEdgeServers / theoreticalMaxChunksInPullFileRequests;
        final double pcRequestsByESs = fileRequestsToEdgeServers == 0 ? 0 : 100 * (double) fileRequestsToEdgeServers / pullFileRequestsInWorkload;

        // Now, we know that the traffic edge-bs-ue is unavoidable. A difference can be made in the traffic cloud-edge.
        // We obtain the theoretical maximum number of chunks.
        double pcOfTheoreticalMaxTraffic = 100 * (double) chunksSentByCloudForPullRequests / theoreticalMaxChunksInPullFileRequests;

        trafficRecord = new TrafficRecord(fileRequestsToCloud, chunksSentByCloudForPullRequests,
                chunksSentByCloudForPrefetchRequests,
                pullFileRequestsInWorkload,
                prefetchFileRequestsInWorkload,
                theoreticalMaxChunksInPullFileRequests,
                theoreticalMaxChunksInPrefetchRequests,
                fileRequestsToEdgeServers, chunksSentByEdgeServers,
                pcChunksByESs, pcRequestsByESs, pcOfTheoreticalMaxTraffic);
    }

    private void calculateLatencies() {
        List<UserEquipment> userEquipments = statisticsCollector.getUserEquipments();
        List<EdgeServer> edgeServers = statisticsCollector.getEdgeServers();

        // ue to cloud pull file requests (i.e., normal downloads)
        for (UserEquipment userEquipment : userEquipments) {
            final int idUe = userEquipment.getId();
            final List<PullFileRequest> fileRequests = getPullFileRequestsByUserEquipment(userEquipment);

            if (!fileRequests.isEmpty()) {
                for (PullRequest request : fileRequests) {
                    final Content file = request.getReferredContent();
                    final int requestId = request.getRequestId();
                    final double requestTime = request.getRequestTimestamp();
                    final double firstChunkReceivedAt = request.getFirstChunkReceivedAt();
                    final double completedAt = request.getCompletedAt();
                    final double latencyFirstChunk = firstChunkReceivedAt - requestTime;
                    final double latencyComplete = completedAt - requestTime;
                    final long fileSizeBytes = file.getSize();
                    final double speed = ((fileSizeBytes * 8) / latencyComplete) / 1000.0;
                    FileDownloadLatencyRecord fileDownloadLatencyRecord = new FileDownloadLatencyRecord(
                            requestId, idUe, file.getDescription(), fileSizeBytes, requestTime, completedAt, latencyFirstChunk, latencyComplete, speed
                    );
                    this.fileDownloadLatencyRecords.add(fileDownloadLatencyRecord);
                }
            }
        }

        // for push data requests from UEs (like for IoT processing scenarios) AT EDGE SERVERS
        for (EdgeServer edgeServer : edgeServers) {
            final int idEdgeServer = edgeServer.getId();
            // final int idUe = edgeServer.getId();
            final List<PushDataRequest> dataRequests = getPushDataRequestsFromUEsAtEdgeServer(edgeServer);
            if (!dataRequests.isEmpty()) {
                for (PushDataRequest request : dataRequests) {
                    Content data = request.getReferredContent();
                    int idUe = request.getPushInitiatorHost().getId();
                    final int requestId = request.getRequestId();
                    final double requestTime = request.getRequestTimestamp();
                    final double firstChunkReceivedAt = request.getFirstChunkReceivedAt();
                    final double completedAt = request.getCompletedAt();
                    final double latencyFirstChunk = firstChunkReceivedAt - requestTime;
                    final double latencyComplete = completedAt - requestTime;
                    final long dataSizeBytes = data.getSize();
                    final double speed = ((dataSizeBytes * 8) / latencyComplete) / 1000.0;
                    DataUploadLatencyRecord dataUploadLatencyRecord = new DataUploadLatencyRecord(
                            requestId, idUe, idEdgeServer, data.getDescription(), dataSizeBytes, requestTime, completedAt, latencyFirstChunk, latencyComplete, speed
                    );
                    this.dataUploadLatencyRecords.add(dataUploadLatencyRecord);
                }
            }
        }

        // ue to ue (downloads of ue data requested from another ue)
        for (UserEquipment userEquipment : userEquipments) {
            final int reqUEId = userEquipment.getId();
            final List<PullDataRequest> dataRequests = getPullDataRequestsByUEs(userEquipment);

            if (!dataRequests.isEmpty()) {
                for (PullRequest request : dataRequests) {
                    final Content content = request.getReferredContent();
                    final int requestId = request.getRequestId();
                    final int sourceUEId = request.getContentLocationHost().getId();
                    final double requestTime = request.getRequestTimestamp();
                    final double firstChunkReceivedAt = request.getFirstChunkReceivedAt();
                    final double completedAt = request.getCompletedAt();
                    final double latencyFirstChunk = firstChunkReceivedAt - requestTime;
                    final double latencyComplete = completedAt - requestTime;
                    final long fileSizeBytes = content.getSize();
                    final double speed = ((content.getSize() * 8) / latencyComplete) / 1000.0;
                    DataRequestFromUELatencyRecord latencyRecord = new DataRequestFromUELatencyRecord(
                            requestId, reqUEId, sourceUEId, content.getDescription(), fileSizeBytes, requestTime, completedAt, latencyFirstChunk, latencyComplete, speed
                    );
                    this.dataRequestFromUELatencyRecords.add(latencyRecord);
                }
            }
        }

        // cloud to ue, requests for data produced by UEs (like IoT data).
        final List<PullRequest> dataRequests = getPullDataRequestsCloudToUE();
        if (!dataRequests.isEmpty()) {
            for (PullRequest request : dataRequests) {
                final Content file = request.getReferredContent();
                final int requestId = request.getRequestId();
                final int dataSourceUEId = request.getContentLocationHost().getId();
                final double requestTime = request.getRequestTimestamp();
                final double firstChunkReceivedAt = request.getFirstChunkReceivedAt();
                final double completedAt = request.getCompletedAt();
                final double latencyFirstChunk = firstChunkReceivedAt - requestTime;
                final double latencyComplete = completedAt - requestTime;
                final long fileSizeBytes = file.getSize();
                final double speed = ((fileSizeBytes * 8) / latencyComplete) / 1000.0;

                DataRequestFromCloudLatencyRecord latencyRecord = new DataRequestFromCloudLatencyRecord(
                        requestId, dataSourceUEId, file.getDescription(), fileSizeBytes, requestTime, completedAt, latencyFirstChunk, latencyComplete, speed
                );
                dataRequestFromCloudLatencyRecords.add(latencyRecord);

            }
        }
    }

    private void showLatencies() {
        System.out.println();
        System.out.println("LATENCY STATS");
        System.out.println("<<File workload.requests>>");
        System.out.println("Req | UE\t| File \t\t\t\t| File size   | Req. time\t\t\t |   Compl. time \t\t  | Latency 1st Chk \t | Elapsed time | Speed (kbps) ");
        System.out.println("─────────────────────────────────────────────────────────────────────────────────────────────────────────────");
        for (FileDownloadLatencyRecord fileDownloadLatencyRecord : fileDownloadLatencyRecords) {
            System.out.println(fileDownloadLatencyRecord.toString());
        }
        final int downlinkFileRecords = fileDownloadLatencyRecords.size();
        double avgLatency = fileDownloadLatencyRecords.stream().mapToDouble(FileDownloadLatencyRecord::getLatencyFirstChunk).sum() / downlinkFileRecords;
        double avgSpeed = fileDownloadLatencyRecords.stream().mapToDouble(FileDownloadLatencyRecord::getSpeed).sum() / downlinkFileRecords;
        System.out.println(String.format("Average latency is %s", avgLatency));
        System.out.println(String.format("Average speed is %s", avgSpeed));

        // ---------------------------------------------------------------------
        System.out.println();
        System.out.println("<<Data workload.requests>>");
        System.out.println("UE-UE data workload.requests");
        System.out.println("Req | Req. UE\t| Src UE | Data \t\t\t\t| Data size   | Req. time\t\t\t |   Compl. time \t\t  | Latency 1st Chk \t | Elapsed time | Speed (kbps) ");
        System.out.println("─────────────────────────────────────────────────────────────────────────────────────────────────────────────");
        for (DataRequestFromUELatencyRecord dataLatencyRecord : dataRequestFromUELatencyRecords) {
            System.out.println(dataLatencyRecord.toString());
        }

        final int dataRequestFromUeRecords = dataRequestFromUELatencyRecords.size();
        avgLatency = dataRequestFromUELatencyRecords.stream().mapToDouble(DataRequestFromUELatencyRecord::getLatencyFirstChunk).sum() / dataRequestFromUeRecords;
        avgSpeed = dataRequestFromUELatencyRecords.stream().mapToDouble(DataRequestFromUELatencyRecord::getSpeed).sum() / dataRequestFromUeRecords;
        System.out.println(String.format("Average latency is %s", avgLatency));
        System.out.println(String.format("Average speed is %s", avgSpeed));

        System.out.println();
        System.out.println("Cloud-UE data workload.requests");
        System.out.println("Req | Src. UE\t| Data \t\t\t\t| Data size   | Req. time\t\t\t |   Compl. time \t\t  | Latency 1st Chk \t | Elapsed time | Speed (kbps) ");
        System.out.println("─────────────────────────────────────────────────────────────────────────────────────────────────────────────");

        for (DataRequestFromCloudLatencyRecord dataLatencyRecord : dataRequestFromCloudLatencyRecords) {
            System.out.println(dataLatencyRecord.toString());
        }

        final int dataRequestsFromCloudRecords = dataRequestFromCloudLatencyRecords.size();
        avgLatency = dataRequestFromCloudLatencyRecords.stream().mapToDouble(DataRequestFromCloudLatencyRecord::getLatencyFirstChunk).sum() / dataRequestsFromCloudRecords;
        avgSpeed = dataRequestFromCloudLatencyRecords.stream().mapToDouble(DataRequestFromCloudLatencyRecord::getSpeed).sum() / dataRequestFromUeRecords;
        System.out.println(String.format("Average latency is %s", avgLatency));
        System.out.println(String.format("Average speed is %s", avgSpeed));
    }

    // This is for the push data requests started by the UEs in IoT pushings
    private List<PushDataRequest> getPushDataRequestsFromUEsAtEdgeServer(EdgeServer edgeServer) {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final List<Request> requestsInHost = scoreboard.getRequestsInHost(edgeServer);
        return requestsInHost.stream()
                .filter(r -> r instanceof PushDataRequest)
                .filter(r -> ((PushDataRequest) r).getPushInitiatorHost() instanceof UserEquipment)
                .map(r -> (PushDataRequest) r).collect(Collectors.toList());
    }

    // Methods to calculate statistics for pull file requests
    private List<PullFileRequest> getPullFileRequestsByUserEquipment(UserEquipment userEquipment) {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final List<Request> requestsInHost = scoreboard.getRequestsInHost(userEquipment);
        return requestsInHost.stream()
                .filter(r -> r instanceof PullFileRequest)
                .map(r -> (PullFileRequest) r).collect(Collectors.toList());
    }

    private List<PushFileRequest> getListOfPrefetchFileRequestsInCloud() {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final List<Request> requestsInHost = scoreboard.getRequestsInHost(statisticsCollector.getCloud());

        return requestsInHost.stream()
                .filter(r -> r instanceof PushFileRequest)
                .map(r -> (PushFileRequest) r)
                .collect(Collectors.toList());
    }

    private List<PullRequest> getListOfPullFileRequestsInCloud() {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();

        final List<Request> requestsInHost = scoreboard.getRequestsInHost(statisticsCollector.getCloud());
        return requestsInHost.stream()
                .filter(r -> r instanceof PullFileRequest)
                .map(r -> (PullFileRequest) r)
                .collect(Collectors.toList());
    }

    private List<PullFileRequest> getPullFileRequestsToEdgeServer(EdgeServer edgeServer) {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final Collection<Request> allRequests = scoreboard.getRequestsInHost(edgeServer);
        return allRequests.stream().filter(r -> r instanceof PullFileRequest)
                .map(r -> (PullFileRequest) r).collect(Collectors.toList());
    }

    private int getNumberOfPullFileRequestsToEdgeServers() {
        final List<EdgeServer> edgeServers = statisticsCollector.getEdgeServers();
        int sum = 0;
        for (EdgeServer edgeServer : edgeServers) {
            final List<PullFileRequest> fileRequests = getPullFileRequestsToEdgeServer(edgeServer);
            sum += fileRequests.size();
        }
        return sum;
    }

    private long getNumberOfChunksSentForPullFileRequestsByEdgeServer(EdgeServer edgeServer) {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final List<Request> requests = scoreboard.getRequestsInHost(edgeServer);
        final Stream<Request> fileRequests = requests.stream().filter(r -> r instanceof PullFileRequest);
        return fileRequests.mapToLong(Request::getTransmittedChunksDownwards).sum();
    }

    private long getNumberOfChunksSentForPullFileRequestsByEdgeServers() {
        long chunksByEdges = 0;
        final List<EdgeServer> edgeServers = statisticsCollector.getEdgeServers();
        for (EdgeServer edgeServer : edgeServers) {
            chunksByEdges += getNumberOfChunksSentForPullFileRequestsByEdgeServer(edgeServer);
        }
        return chunksByEdges;
    }

    private int calculateChunksInListOfPullRequestEvents() {
        final List<RequestEvent> requestEvents = getListOfPullFileRequestsInWorkload();
        return calculateChunksInListOfPullRequestEvents(requestEvents);
    }

    private int calculateChunksInListOfPrefetchRequestEvents() {
        final List<RequestEvent> requestEvents = getListOfPrefetchFileRequestsInWorkload();
        return calculateChunksInListOfPrefetchFileRequestEvents(requestEvents);
    }

    private List<RequestEvent> getListOfPrefetchFileRequestsInWorkload() {
        return workload.getRequestEvents().stream().
                filter(r -> r instanceof PrefetchRequestEvent).
                collect(Collectors.toList());
    }

    private int calculateChunksInListOfPrefetchFileRequestEvents(List<RequestEvent> requestEvents) {
        return requestEvents.stream()
                .map(RequestEvent::getReferredContent)
                .mapToInt(f -> getNumberOfChunksForContent(f, CHUNK_SIZE)).sum();
    }

    private List<RequestEvent> getListOfPullFileRequestsInWorkload() {
        return workload.getRequestEvents().stream().
                filter(r -> r instanceof PullFileRequestEvent).
                collect(Collectors.toList());
    }

    private int calculateChunksInListOfPullRequestEvents(List<RequestEvent> requestEvents) {
        return requestEvents.stream()
                .map(r -> ((PullFileRequestEvent) r).getReferredContent())
                .mapToInt(f -> getNumberOfChunksForContent(f, CHUNK_SIZE)).sum();
    }



    // Methods to calculate statistics for data requests
    private int getPullDataRequestsUEToUEInWorkload() {
        final List<RequestEvent> requestEvents = workload.getRequestEvents();
        final Stream<RequestEvent> dataRequests = requestEvents.stream()
                .filter(r -> r instanceof PullDataRequestEvent);
        return (int) dataRequests.filter(e -> e.getEventOriginHost() instanceof UserEquipment).count();
    }

    private int getPullDataRequestsCloudToUEInWorkload() {
        final List<RequestEvent> requestEvents = workload.getRequestEvents();
        final Stream<RequestEvent> dataRequests = requestEvents.stream()
                .filter(r -> r instanceof PullDataRequestEvent);
        return (int) dataRequests.filter(r -> r.getEventOriginHost() instanceof CloudUnit).count();
    }

    private List<PullDataRequest> getPullDataRequestsByUEs(UserEquipment userEquipment) {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final List<Request> requests = scoreboard.getRequestsInHost(userEquipment);
        final Stream<PullDataRequest> dataRequests = requests.stream()
                .filter(ev -> ev instanceof PullDataRequest)
                .map(ev -> (PullDataRequest) ev);
        // Note that we need only to report the workload.requests made by this ue, not the ones served by the ue
        //  so we need to filter by requesting host
        final Stream<PullDataRequest> requestsByUE = dataRequests.filter(r -> r.getRequestingHost() == userEquipment);
        return requestsByUE.collect(Collectors.toList());
    }

    private List<PullDataRequest> getListOfPullDataRequestsServedByUEs() {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        List<PullDataRequest> dataRequestsServedByUEs = new ArrayList<>();
        final List<UserEquipment> userEquipments = statisticsCollector.getUserEquipments();

        for (UserEquipment ue : userEquipments) {
            final List<Request> requestsInHost = scoreboard.getRequestsInHost(ue);
            final Stream<PullDataRequest> dataRequests = requestsInHost.stream()
                    .filter(r -> r instanceof PullDataRequest)
                    .map(r -> (PullDataRequest) r);
            final List<PullDataRequest> requestsServedByUE = dataRequests
                    .filter(r -> r.getContentLocationHost() == ue).collect(Collectors.toList());
            dataRequestsServedByUEs.addAll(requestsServedByUE);
        }

        return dataRequestsServedByUEs;
    }

    private long getTheoreticalMaxChunksInPullDataRequests() {
        final Stream<RequestEvent> dataRequests = workload.getRequestEvents()
                .stream().filter(r -> r instanceof PullDataRequestEvent);
        return dataRequests.map(r -> (PullDataRequestEvent) r)
                .map(PullDataRequestEvent::getReferredContent)
                .mapToLong(data -> getNumberOfChunksForContent(data, CHUNK_SIZE)).sum();
    }

    private long getTheoreticalMaxChunksInCloudToUEPullDataRequests() {
        CloudUnit cloud = statisticsCollector.getCloud();
        final Stream<RequestEvent> cloudToUEDataRequests = workload.getRequestEvents()
                .stream().filter(r -> r instanceof PullDataRequestEvent && r.getEventOriginHost() == cloud);

        return cloudToUEDataRequests.map(ev -> (PullDataRequestEvent) ev)
                .map(PullDataRequestEvent::getReferredContent)
                .mapToLong(data -> getNumberOfChunksForContent(data, CHUNK_SIZE)).sum();
    }

    private long getTheoreticalMaxChunksInUEToUEPullDataRequests() {
        final Stream<RequestEvent> UEToUEDataRequests = workload.getRequestEvents()
                .stream()
                .filter(r -> r instanceof PullDataRequestEvent && r.getEventOriginHost() instanceof UserEquipment);

        return UEToUEDataRequests.map(r -> (PullDataRequestEvent) r)
                .map(PullDataRequestEvent::getReferredContent)
                .mapToLong(data -> getNumberOfChunksForContent(data, CHUNK_SIZE)).sum();
    }

    private List<PullDataRequest> getCloudToUEsPullDataRequestsByEdgeServer(EdgeServer edgeServer) {
        CloudUnit cloud = statisticsCollector.getCloud();
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();

        final List<Request> requestsInHost = scoreboard.getRequestsInHost(edgeServer);
        final Stream<PullDataRequest> dataRequests = requestsInHost.stream()
                .filter(r -> r instanceof PullDataRequest)
                .map(r -> (PullDataRequest) r);
        final Stream<PullDataRequest> cloudToUERequests = dataRequests.filter(r -> r.getRequestingHost() == cloud);
        return cloudToUERequests.collect(Collectors.toList());
    }

    private List<PullRequest> getUEToUEsPullDataRequestsByEdgeServer(EdgeServer edgeServer) {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();

        final List<Request> requestsInHost = scoreboard.getRequestsInHost(edgeServer);
        final Stream<PullDataRequest> dataRequests = requestsInHost.stream()
                .filter(r -> r instanceof PullDataRequest).map(r -> (PullDataRequest) r);
        final Stream<PullDataRequest> ueToUERequests = dataRequests.filter(r -> r.getRequestingHost() instanceof UserEquipment);
        return ueToUERequests.collect(Collectors.toList());
    }

    private List<PullRequest> getCloudToUEsPullDataRequestsByEdgeServers() {
        CloudUnit cloud = statisticsCollector.getCloud();
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final List<EdgeServer> edgeServers = statisticsCollector.getEdgeServers();
        List<PullRequest> allRequests = new ArrayList<>();
        for (EdgeServer edgeServer : edgeServers) {
            final List<Request> requestsInHost = scoreboard.getRequestsInHost(edgeServer);
            final Stream<PullDataRequest> dataRequests = requestsInHost.stream()
                    .filter(r -> r instanceof PullDataRequest).map(r -> (PullDataRequest) r);
            final Stream<PullDataRequest> cloudToUERequests = dataRequests.filter(r -> r.getRequestingHost() == cloud);
            allRequests.addAll(cloudToUERequests.collect(Collectors.toList()));
        }
        return allRequests;
    }

    private List<PullRequest> getUEToUEsPullDataRequestsByEdgeServers() {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final List<EdgeServer> edgeServers = statisticsCollector.getEdgeServers();
        List<PullRequest> allRequests = new ArrayList<>();
        for (EdgeServer edgeServer : edgeServers) {
            final List<Request> requestsInHost = scoreboard.getRequestsInHost(edgeServer);
            final Stream<PullDataRequest> dataRequests = requestsInHost.stream().filter(r -> r instanceof PullDataRequest)
                    .map(r -> (PullDataRequest) r);
            final Stream<PullDataRequest> UEToUERequests = dataRequests.filter(r -> r.getRequestingHost() instanceof UserEquipment);
            allRequests.addAll(UEToUERequests.collect(Collectors.toList()));
        }
        return allRequests;
    }

    private long getNumberOfChunksSentForCloudToUEsPullDataRequestsByEdgeServer(EdgeServer edgeServer) {
        CloudUnit cloud = statisticsCollector.getCloud();
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();

        final List<Request> requestsInHost = scoreboard.getRequestsInHost(edgeServer);
        final Stream<PullDataRequest> dataRequests = requestsInHost.stream().filter(r -> r instanceof PullDataRequest)
                .map(r -> (PullDataRequest) r);
        final Stream<PullDataRequest> cloudToUERequests = dataRequests.filter(r -> r.getRequestingHost() == cloud);
        return cloudToUERequests.mapToLong(PullRequest::getTransmittedChunksUpwards).sum();
    }

    private long getNumberOfChunksSentForUEToUEPullDataRequestsByEdgeServer(EdgeServer edgeServer) {
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();

        final List<Request> requestsInHost = scoreboard.getRequestsInHost(edgeServer);
        final Stream<PullDataRequest> dataRequests = requestsInHost.stream()
                .filter(r -> r instanceof PullDataRequest).map(r -> (PullDataRequest) r);
        final Stream<PullDataRequest> cloudToUERequests = dataRequests.filter(r -> r.getRequestingHost() instanceof UserEquipment);
        return cloudToUERequests.mapToLong(PullRequest::getTransmittedChunksDownwards).sum();
    }

    private long getNumberOfChunksSentForCloudToUEPullDataRequestsByEdgeServers() {
        CloudUnit cloud = statisticsCollector.getCloud();
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();

        List<EdgeServer> edgeServers = statisticsCollector.getEdgeServers();
        long chunksByEdgeServersInCloudToUEDataRequests = 0;
        for (EdgeServer edgeServer : edgeServers) {
            final Collection<Request> requestsInEdgeServer = scoreboard.getRequestsInHost(edgeServer);
            final Stream<PullRequest> allPullDataRequests = requestsInEdgeServer.stream()
                    .filter(r -> r instanceof PullDataRequest).map(r -> (PullDataRequest) r);
            final Stream<PullRequest> cloudToUERequests = allPullDataRequests.filter(r -> r.getRequestingHost() == cloud);
            chunksByEdgeServersInCloudToUEDataRequests += cloudToUERequests.mapToLong(PullRequest::getTransmittedChunksUpwards).sum();
        }
        return chunksByEdgeServersInCloudToUEDataRequests;
    }

    private long getNumberOfChunksSentForUEToUEPullDataRequestsByEdgeServers() {
        List<EdgeServer> edgeServers = statisticsCollector.getEdgeServers();
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        long chunksByEdgeServersInUEToUEDataRequests = 0;
        for (EdgeServer edgeServer : edgeServers) {
            final Collection<Request> requestsInEdgeServer = scoreboard.getRequestsInHost(edgeServer);
            final Stream<PullDataRequest> allPullDataRequests = requestsInEdgeServer.stream()
                    .filter(r -> r instanceof PullDataRequest).map(r -> (PullDataRequest) r);
            final List<PullRequest> UEToUERequests = allPullDataRequests
                    .filter(r -> r.getRequestingHost() instanceof UserEquipment).collect(Collectors.toList());
            chunksByEdgeServersInUEToUEDataRequests += UEToUERequests.stream().mapToLong(PullRequest::getTransmittedChunksDownwards).sum();
        }

        return chunksByEdgeServersInUEToUEDataRequests;
    }

    private List<PullRequest> getPullDataRequestsUEToUE() {
        List<PullRequest> dataRequestsByUEs = new ArrayList<>();
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final List<UserEquipment> userEquipments = statisticsCollector.getUserEquipments();
        for (UserEquipment ue : userEquipments) {
            final List<Request> requestsInHost = scoreboard.getRequestsInHost(ue);
            final Stream<PullDataRequest> dataRequests = requestsInHost.stream()
                    .filter(r -> r instanceof PullDataRequest).map(r -> (PullDataRequest) r);
            // workload.requests made by UEs
            final List<PullDataRequest> requestsFromAnotherUE = dataRequests
                    .filter(r -> r.getRequestingHost() == ue).collect(Collectors.toList());
            dataRequestsByUEs.addAll(requestsFromAnotherUE);
        }
        return dataRequestsByUEs;
    }

    private List<PullRequest> getPullDataRequestsCloudToUE() {
        final CloudUnit cloud = statisticsCollector.getCloud();
        final StatisticsCollector.ContentRequestScoreboard scoreboard = statisticsCollector.getContentScoreboard();
        final List<Request> requestsInCloud = scoreboard.getRequestsInHost(cloud);
        final Stream<PullDataRequest> pullDataRequests = requestsInCloud.stream()
                .filter(r -> r instanceof PullDataRequest).map(r -> (PullDataRequest) r);
        // only those made by the cloud
        return pullDataRequests.filter(r -> r.getRequestingHost() == cloud).collect(Collectors.toList());
    }

    private void showTraffic() {
        System.out.println();
        System.out.println("TRAFFIC STATS");
        printTrafficForPullFileRequests();

        System.out.println();
        printTrafficForPullDataRequests();
    }

    private void printTrafficForPullDataRequests() {
        System.out.println("<<Data workload.requests>>");

        int cloudToUERequestsInWorkload = getPullDataRequestsCloudToUEInWorkload();
        int UEToUERequestsInWorkload = getPullDataRequestsUEToUEInWorkload();
        System.out.println(String.format("Data workload.requests from Cloud to UE in workload: %s", cloudToUERequestsInWorkload));
        System.out.println(String.format("Data workload.requests from UE to UE in workload: %s", UEToUERequestsInWorkload));

        final List<PullRequest> dataRequestsUEToUE = getPullDataRequestsUEToUE();
        final List<PullRequest> dataRequestsCloudToUE = getPullDataRequestsCloudToUE();


        // Get chunks and workload.requests in UEs
        System.out.println("Traffic in UEs-edge segment");
        final List<PullDataRequest> pullDataRequestsServedByUEs = getListOfPullDataRequestsServedByUEs();
        long chunksByUEs = pullDataRequestsServedByUEs.stream().mapToLong(PullRequest::getTransmittedChunksUpwards).sum();
        long theoreticalMaxChunks = getTheoreticalMaxChunksInPullDataRequests();
        System.out.println(String.format("Theoretical max chunks in all data transmissions: %s", theoreticalMaxChunks));
        System.out.println(String.format("Chunks sent by UEs %s", chunksByUEs));
        System.out.println(String.format("Data reqs. served by UEs %s", pullDataRequestsServedByUEs.size()));
        double proportionOfTraffic = 100.0 * chunksByUEs / theoreticalMaxChunks;
        System.out.println(String.format("Pc of max hypothetical traffic %.3f%% (saved traffic: %.3f%%)", proportionOfTraffic, 100 - proportionOfTraffic));

        System.out.println();
        System.out.println("Traffic in ES-Cloud segment (only for Cloud to UE data workload.requests)");
        System.out.println(String.format("Effective data workload.requests from Cloud to UE: %s", dataRequestsCloudToUE.size()));
        long chunksByEdgeServersInCloudToUEDataRequests = getNumberOfChunksSentForCloudToUEPullDataRequestsByEdgeServers();
        List<PullRequest> cloudToUEDataRequestsByEdgeServer = getCloudToUEsPullDataRequestsByEdgeServers();
        final List<EdgeServer> edgeServers = statisticsCollector.getEdgeServers();
        for (EdgeServer edgeServer : edgeServers) {
            final int dataRequests = getCloudToUEsPullDataRequestsByEdgeServer(edgeServer).size();
            final long chunksThisEdge = getNumberOfChunksSentForCloudToUEsPullDataRequestsByEdgeServer(edgeServer);
            final double pcDataRequests = dataRequests == 0 ? 0 : 100 * (double) dataRequests / cloudToUEDataRequestsByEdgeServer.size();
            final double pcChunks = chunksThisEdge == 0 ? 0 : 100 * (double) chunksThisEdge / chunksByEdgeServersInCloudToUEDataRequests;
            System.out.println(String.format("Edge %s %s. Served data workload.requests: %s (%.3f%%), Sent chunks %s (%.3f%%)",
                    edgeServer.getId(), edgeServer.getLocation(), dataRequests, pcDataRequests, chunksThisEdge, pcChunks));
        }
        theoreticalMaxChunks = getTheoreticalMaxChunksInCloudToUEPullDataRequests();
        System.out.println(String.format("Theoretical max chunks in Cloud-UE transmissions: %s", theoreticalMaxChunks));
        System.out.println(String.format("Chunks sent by ES: %s", chunksByEdgeServersInCloudToUEDataRequests));
        proportionOfTraffic = chunksByEdgeServersInCloudToUEDataRequests == 0 ? 0 : 100.0 * chunksByEdgeServersInCloudToUEDataRequests / theoreticalMaxChunks;
        System.out.println(String.format("Pc of max hypothetical traffic %.3f%% (saved traffic: %.3f%%)", proportionOfTraffic, 100 - proportionOfTraffic));


        System.out.println();
        System.out.println("Traffic in ES-UEs segment (only for UE to UE data workload.requests)");
        System.out.println(String.format("Effective data workload.requests from UE to UE: %s", dataRequestsUEToUE.size()));

        long chunksByEdgeServersInUEToUEDataRequests = getNumberOfChunksSentForUEToUEPullDataRequestsByEdgeServers();
        List<PullRequest> UEToUEDataRequestsByEdgeServer = getUEToUEsPullDataRequestsByEdgeServers();
        for (EdgeServer edgeServer : edgeServers) {
            final int dataRequests = getUEToUEsPullDataRequestsByEdgeServer(edgeServer).size();
            final long chunksThisEdge = getNumberOfChunksSentForUEToUEPullDataRequestsByEdgeServer(edgeServer);
            final double pcDataRequests = dataRequests == 0 ? 0 : 100 * (double) dataRequests / UEToUEDataRequestsByEdgeServer.size();
            final double pcChunks = chunksThisEdge == 0 ? 0 : 100 * (double) chunksThisEdge / chunksByEdgeServersInUEToUEDataRequests;
            System.out.println(String.format("Edge %s %s. Served data workload.requests: %s (%.3f%%), Sent chunks %s (%.3f%%)",
                    edgeServer.getId(), edgeServer.getLocation(), dataRequests, pcDataRequests, chunksThisEdge, pcChunks));
        }
        theoreticalMaxChunks = getTheoreticalMaxChunksInUEToUEPullDataRequests();
        System.out.println(String.format("Theoretical max chunks in UE-UE transmissions: %s", theoreticalMaxChunks));
        System.out.println(String.format("Chunks sent by ES: %s", chunksByEdgeServersInUEToUEDataRequests));
        proportionOfTraffic = chunksByEdgeServersInUEToUEDataRequests == 0 ? 0 : 100.0 * chunksByEdgeServersInUEToUEDataRequests / theoreticalMaxChunks;
        System.out.println(String.format("Pc of max hypothetical traffic %.3f%% (saved traffic: %.3f%%)", proportionOfTraffic, 100 - proportionOfTraffic));
    }

    private void printTrafficForPullFileRequests() {
        System.out.println("<<File workload.requests>>");
        System.out.println("Traffic in cloud-edge segment");

        System.out.println(String.format("PullFileRequests (to the cloud) in workload: %s (%s chunks)",
                trafficRecord.getPullFileRequestsInWorkload(), trafficRecord.getTheoreticalMaxChunksInPullFileRequests()));
        System.out.println(String.format("Chunks sent by cloud %s", trafficRecord.getChunksSentByCloudForPullRequests()));
        System.out.println(String.format("Files served from cloud %s", trafficRecord.getPullFileRequestsToCloud()));

        System.out.println("Traffic in edge-bs-ue segment");
        // Get chunks and workload.requests in the edge servers.
        int fileRequestsToEdgeServers = getNumberOfPullFileRequestsToEdgeServers();
        long chunksSentByEdgeServers = getNumberOfChunksSentForPullFileRequestsByEdgeServers();

        final List<EdgeServer> edgeServers = statisticsCollector.getEdgeServers();
        for (EdgeServer edgeServer : edgeServers) {
            final List<PullFileRequest> filesForES = getPullFileRequestsToEdgeServer(edgeServer);
            final int filesThisEdge = filesForES.size();
            final long chunksThisEdge = getNumberOfChunksSentForPullFileRequestsByEdgeServer(edgeServer);
            final double pcFiles = filesThisEdge == 0 ? 0 : 100 * (double) filesThisEdge / fileRequestsToEdgeServers;
            final double pcChunks = chunksThisEdge == 0 ? 0 : 100 * (double) chunksThisEdge / chunksSentByEdgeServers;
            System.out.println(String.format("Edge %s %s. Served file workload.requests: %s (%.3f%%), Sent chunks %s (%.3f%%)",
                    edgeServer.getId(), edgeServer.getLocation(), filesThisEdge, pcFiles, chunksThisEdge, pcChunks));
        }

        //        final double pcChunks = chunksSentByEdgeServers == 0 ? 0 : 100 * (double) chunksSentByEdgeServers / trafficRecord.getTheoreticalMaxChunksInPullFileRequests();
        //        final double pcRequests = fileRequestsToEdgeServers == 0 ? 0 : 100 * (double) fileRequestsToEdgeServers / trafficRecord.getPullFileRequestsInWorkload();
        //        System.out.println(String.format("Total Chunks sent by edges %s (%s%%)",  chunksSentByEdgeServers, pcChunks));
        System.out.println(String.format("Total Chunks sent by edges %s (%s%%)", trafficRecord.getChunksSentByEdgeServers(), trafficRecord.getPcChunksByESs()));
        //        System.out.println(String.format("Total Files served from edges %s (%s%%)", fileRequestsToEdgeServers, pcRequests));
        System.out.println(String.format("Total Files served from edges %s (%s%%)", trafficRecord.getPullFileRequestsToEdgeServers(), trafficRecord.getPcRequestsByESs()));

        // Now, we know that the traffic edge-bs-ue is unavoidable. A difference can be made in the traffic cloud-edge.
        // We obtain the theoretical maximum number of chunks.
        //        double pcTraffic = 100 * (double) trafficRecord.getChunksSentByCloud() / trafficRecord.getTheoreticalMaxChunksInPullFileRequests();
        //        System.out.println(String.format("Pc of max hypothetical traffic from cloud %.3f%% (saved traffic: %.3f%%)", pcTraffic, 100 - pcTraffic));
        System.out.println(String.format("Pc of max hypothetical traffic from cloud %.3f%% (saved traffic: %.3f%%)",
                trafficRecord.getPcOfTheoreticalMaxTraffic(), 100 - trafficRecord.getPcOfTheoreticalMaxTraffic()));
        System.out.println("Max Hypothetical traffic refers to requested files * chunk size. Previous %% could be " +
                "> 100 due to handover and retransmitted chunks.");
    }
}
