package edu.ucc.network.devices;

import edu.ucc.core.ContentTransmitter;
import edu.ucc.core.HandoverManager;
import edu.ucc.core.PrefetchManager;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.core.transport.chunks.ChunksBuffer;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.core.transport.links.GroupedLink;
import edu.ucc.utils.TransmissionConstants;
import edu.ucc.utils.TransmissionUtils;

import java.io.Serializable;
import java.util.List;

import static edu.ucc.utils.Logging.printWarning;
import static edu.ucc.utils.TransmissionConstants.CHUNK_SIZE_FOR_ACK;
import static edu.ucc.utils.TransmissionConstants.PROCESSING_DELAY;

/**
 * Basic class representing a Host in the architecture (a {@link CloudUnit}, {@link EdgeServer}, {@link BaseStation},
 * or {@link UserEquipment}. Notice they are typified on {@link RelayingHost} and {@link AdvancedHost}
 * This class manages the assignation of children, sibling and parent units. For siblings and children units, this class
 * controls their bandwidth assignment. The content transmission is administered using a {@link ContentTransmitter}
 * implementation.
 */
public abstract class Host implements Serializable {
    /**
     * The identifier id of the host.
     */
    protected int id;

    /**
     * The storage capacity of this host.
     */
    private final FileSystem fileSystem;

    /**
     * The location coordinates of the host.
     * This is exploited to adjust connectivity to other hosts (i.e., UEs and BSs).
     */
    protected Location location;

    /**
     * The Maximum Transmission Unit (sort of) in the DownLink for this host.
     * Content is split according to this size.
     */
    private int mtu;

    /**
     * The parent unit, to create a hierarchy.
     * The host can refer to the parent unit for collecting content not available locally.
     */
    protected Host parentUnit;

    /**
     * The link to parent.
     * Notice that a {@link GroupedLink} is used since it comprises both UpLink and DownLinks.
     */
    protected GroupedLink groupedLinkToParent;

    /**
     * The list of chunks awaiting for ACK from the parent host.
     */
    protected final ChunksBuffer uplinkChunksAwaitingAck;

    /**
     * The buffer of chunks to deliver to parent unit
     */
    private final ChunksBuffer uplinkChunksBuffer;

    /**
     * An implementation of the {@link ContentTransmitter} interface to control how the host moves content across the
     * hierarchy.
     */
    private ContentTransmitter contentTransmitter;

    /**
     * An implementation of the {@link HandoverManager} interface to control how the architecture must react when a UE
     * connects or disconnects from base stations.
     */
    private HandoverManager handoverManager;

    private PrefetchManager prefetchManager;

    /**
     * Basic constructor.
     *
     * @param id       The host's id.
     * @param storage  The available storage for the host.
     * @param location The host's location coordinates.
     * @param mtu      The host's Maximum Transmission Unit
     */
    public Host(int id, long storage, Location location, int mtu) {
        this.id = id;
        this.fileSystem = new FileSystem(this, storage);
        this.location = location;
        this.mtu = mtu;
        this.uplinkChunksAwaitingAck = new ChunksBuffer(this);
        this.uplinkChunksBuffer = new ChunksBuffer(this);
    }

    /**
     * Obtains the host's id
     *
     * @return The host's id
     */
    public int getId() {
        return id;
    }

    /**
     * Obtains the host's location coordinates.
     *
     * @return The host's location coordinates.
     */
    public Location getLocation() {
        return location;
    }

    /**
     * Sets the new host's location
     *
     * @param location The {@link Location} to set for this host.
     */
    public void setLocation(Location location) {
        this.location = location;
    }

    /**
     * Gets the size of the chunk transmitted to other hosts for content workload.requests
     *
     * @return The size of the chunks (MTU)
     */
    public int getMtu() {
        return mtu;
    }

    /**
     * Obtains a reference to the host's parent unit.
     *
     * @return The host's parent unit. null if no parent has been defined.
     */
    public Host getParentUnit() {
        return parentUnit;
    }

    /**
     * Sets the host's parent unit.
     *
     * @param newParent The new parent for the host. Notice that type checking is performed (e.g., a MEC is not a valid
     *                  UEs parent.
     */
    public abstract void setParentUnit(Host newParent);

    /**
     * Assigns the host's parent unit.
     *
     * @param newParent The parent unit to assigns.
     */
    protected void setParent(Host newParent) {
        this.parentUnit = newParent;
    }


    public GroupedLink getGroupedLinkToParent() {
        return groupedLinkToParent;
    }

    /**
     * Sets the {@link GroupedLink} for this host's parent.
     *
     * @param link The link to set.
     */
    public void setGroupedLinkToParent(GroupedLink link) {
        this.groupedLinkToParent = link;
    }

    /**
     * Sets the implementation of the {@link ContentTransmitter} interface, for the host to control how to move content
     * across the hierarchy in content workload.requests.
     *
     * @param contentTransmitter An implementation of the {@link ContentTransmitter} interface.
     */
    public void setContentTransmitter(ContentTransmitter contentTransmitter) {
        this.contentTransmitter = contentTransmitter;
    }

    /**
     * Sets the implementation of the {@link HandoverManager} interface, for the host to handle moving user equipments.
     *
     * @param handoverManager An implementation of the {@link HandoverManager} interface.
     */
    public void setHandoverManager(HandoverManager handoverManager) {
        this.handoverManager = handoverManager;
    }

    /**
     * Sets the implementation of the {@link PrefetchManager} interface, for the host to handle content prefetch requests.
     * @param prefetchManager An implementation of the {@link PrefetchManager} interface
     */
    protected void setPrefetchManager(PrefetchManager prefetchManager) {
        this.prefetchManager = prefetchManager;
    }

    public double calculateTxTimeToParent(int chunkSize) {
        // double bandwidthToParent =groupedLinkToParent.getTotalBandwidthProvidedByMe(this);
        double bandwidthToParent = groupedLinkToParent.getAvailableBandwidthProvidedByMe(this);
        double transmissionDelay = chunkSize / bandwidthToParent;
        double distance = this.location.distanceTo(parentUnit.location);
        double propagationDelay;
        //        if (parentUnit instanceof CloudUnit)
        //            propagationDelay = distance / COPPER_PROPAGATION_FACTOR;
        //        else
        propagationDelay = distance / TransmissionConstants.LIGHT_SPEED;

        return transmissionDelay + propagationDelay;
    }

    public void increaseBandwidthFromParent(ContentChunk chunk) {
        int chunkSize = chunk.getChunkSize();
        this.groupedLinkToParent.increaseBandwidthFromMe(this.getParentUnit(), chunkSize);
    }

    /**
     * Processes each incoming event.
     * This is the edu.ucc.core event for hosts during the system simulation.
     * Since the events have a type, this method consists on a switch statement with a case for each event type.
     * Notice that the host has the ability to add more events to the queue of simulation events to advance on the
     * simulation process (see {@link Simulation}
     *
     * @param event The {@link TestbedEvent} to process.
     */
    public void processEvent(TestbedEvent event) {
        if (event instanceof PullRequestEvent) {
            final PullRequestEvent pullRequestEvent = (PullRequestEvent) event;
            contentTransmitter.onPullRequestReceived(this, pullRequestEvent);
        } else if (event instanceof PushRequestEvent) {
            final PushRequestEvent pushRequestEvent = (PushRequestEvent) event;
            contentTransmitter.onPushRequestReceived(this, pushRequestEvent);
        } else if (event instanceof AckReceivedEvent) {
            contentTransmitter.onAckReceived(this, (AckReceivedEvent) event);
        } else if (event instanceof ChunkReceivedEvent) {
            contentTransmitter.onChunkReceived(this, (ChunkReceivedEvent) event);
        } else if (event instanceof HandoverEvent) {
            handoverManager.onUserEquipmentHandover(this, (HandoverEvent) event);
        } else if (event instanceof PrefetchRequestEvent) {
            prefetchManager.onPrefetchRequestReceived(this, (PrefetchRequestEvent) event);
        } else {
            throw new RuntimeException("Event not recognized");
        }
    }


    /**
     * Sends the ACK about received chunk to sender. Here, it can only be the parent. For RelayingHost it can be a child.
     *
     * @param ackedHost  The host to which the ACK must be sent.
     * @param ackedChunk The chunk that has been received and whose ack should be notified
     */
    public boolean sendAck(Host ackedHost, ContentChunk ackedChunk) {
        double txTime = 0.0;
        if (ackedHost == this.parentUnit) {
            txTime = calculateTxTimeToParent(CHUNK_SIZE_FOR_ACK);
        } else {
            return false;
        }
        // Note that technically here we should be reducing the available bandwidth in the uplink for sending the ACK
        double timestamp = Simulation.getInstance().getCurrentTime() + txTime + PROCESSING_DELAY;
        TestbedEvent ackEvent = new AckReceivedEvent(this, ackedHost, timestamp, ackedChunk);
        Simulation.getInstance().postEvent(ackEvent);
        return true;
    }

    public void addChunksForTransmissionToParent(RequestEvent requestEvent, int totalChunks, int chunkSize) {
        for (int chunkNum = 0; chunkNum < totalChunks; chunkNum++) {
            this.addChunkForTransmissionToParent(new ContentChunk(requestEvent, chunkNum, totalChunks, chunkSize));
        }
    }

    public void addChunkForTransmissionToParent(ContentChunk chunk) {
        this.uplinkChunksBuffer.addToBuffer(parentUnit, chunk);
    }

    /**
     * This method is called to try to relay a chunk recently received by the current host and that is addressed to the
     * parent device.
     */
    public void sendChunksToParent() {
        // We will only send if there is nothing pending for the parent
        if (uplinkChunksBuffer.areTherePendingChunks()) {
            if (uplinkChunksAwaitingAck.noChunksForHost(parentUnit)) {
                final List<ContentChunk> chunksForHost = uplinkChunksBuffer.getChunksForNextHopHost(parentUnit);
                sendNextChunkToParent(chunksForHost.get(0));
            }
        }
    }

    private void sendNextChunkToParent(ContentChunk nextChunk) {
        GroupedLink linkToParent = getGroupedLinkToParent();
        if (linkToParent.getAvailableBandwidthProvidedByMe(this) > nextChunk.getChunkSize()) {
            uplinkChunksAwaitingAck.addToBuffer(parentUnit, nextChunk);
            // Simulation.getInstance().getStatisticsCollector().onChunkTransmittedUpwards(this, nextChunk);
            Simulation.getInstance().getEventsBroker().onChunkTransmittedUpwards(this, nextChunk);

            final boolean b = uplinkChunksBuffer.removeChunkFromBuffer(parentUnit, nextChunk);
            if (!b) throw new RuntimeException("I could not delete the chunk from buffer!!");

            double txTime = calculateTxTimeToParent(nextChunk.getChunkSize());
            txTime += PROCESSING_DELAY; // This adds a processing delay to each sent chunk
            linkToParent.decreaseBandwidthFromMe(this, nextChunk.getChunkSize());
            TestbedEvent sendChunkEvent = TransmissionUtils.createEventForChunk(this, parentUnit,
                    txTime, nextChunk);
            Simulation.getInstance().postEvent(sendChunkEvent);
        } else {
            printWarning(String.format("Link bandwidth from %s %s to %s %s is not enough to move %s bits",
                    getTypeAsString(), this.id, parentUnit.getTypeAsString(), parentUnit.id,
                    nextChunk.getChunkSize()));
        }
    }

    /**
     * This method is always called after receiving an ACK.
     * It sends next chunks for parent.
     */
    public void sendChunksToParentAfterAck() {
        sendChunksToParent();
    }

    /**
     * This method is always called after the sending UE is in a handover.
     */
    public void sendChunksToParentAfterHandover() {
        sendChunksToParent();
    }

    public void deleteChunkFromUplinkAckBuffer(ContentChunk chunk) {
        this.uplinkChunksAwaitingAck.removeChunkFromBuffer(parentUnit, chunk);
    }

    public List<ContentChunk> getPendingAckChunks(Host destinationHost) {
        return uplinkChunksAwaitingAck.getChunksForNextHopHost(destinationHost);
    }

    public List<ContentChunk> getPendingChunks(Host destinationHost) {
        return uplinkChunksBuffer.getChunksForNextHopHost(destinationHost);
    }

    public void removeChunksWaitingAckFromPreviousParent(Host previousParent) {
        uplinkChunksAwaitingAck.removeChunksForHost(previousParent);
    }

    public void removePendingChunksFromPreviousParent(Host previousParent) {
        uplinkChunksBuffer.removeChunksForHost(previousParent);
    }

    public boolean isThisMyParent(Host host) {
        return host == getParentUnit();
    }

    public FileSystem getFileSystem() {
        return fileSystem;
    }

    public abstract String getTypeAsString();
}

