package edu.ucc.network.devices;

import edu.ucc.entities.Content;
import edu.ucc.entities.Data;
import edu.ucc.entities.IoTData;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * This class models the file system for a host and associated operations.
 */
public class FileSystem {
    private final Map<String, LocalContent> contents;
    private long freeSpace;
    private final Host host;

    public FileSystem(Host host, long totalStorageSize) {
        this.freeSpace = totalStorageSize;
        this.contents = new HashMap<>();
        this.host = host;
    }

    public boolean contentExists(Content content) {
        return contents.containsKey(content.getURI());
    }

    public boolean contentExists(String URI) {
        return contents.containsKey(URI);
    }

    public Content getTouchContent(String URI, double timestamp) {
        final LocalContent localContent = contents.get(URI);
        localContent.setLastAccessed(timestamp);
        return localContent.content;
    }

    public Content getContent(String URI) {
        final LocalContent localContent = contents.get(URI);
        if (localContent == null) return null;
        return localContent.content;
    }

    public StorageReason getStorageReason(String URI) {
        final LocalContent localContent = contents.get(URI);
        if (localContent == null) return null;
        return localContent.storageReason;
    }

    public boolean storeContent(int appId, Content content, double timestamp, StorageReason reason) {
        if (this.freeSpace > content.getSize()) {
            this.contents.put(content.getURI(), new LocalContent(appId, content, timestamp, reason));
            this.reduceFreeSpace(content);
            return true;
        }
        return false;
    }

    public boolean deleteContent(Content content) {
        LocalContent removedContent = this.contents.remove(content.getURI());
        if (removedContent == null) return false;
        this.increaseFreeSpace(removedContent);
        return true;
    }

    private void reduceFreeSpace(Content addedContent) {
        this.freeSpace -= addedContent.getSize();
    }

    private void increaseFreeSpace(LocalContent removedContent) {
        this.freeSpace += removedContent.content.getSize();
    }

    public void updateURIForMovingUEData(UserEquipment movingUE) {
        final List<Map.Entry<String, LocalContent>> entriesForDataPushedByMovingUE = this.contents.entrySet().stream()
                .filter(contentEntry -> contentEntry.getValue().content instanceof Data)
                .filter(entry -> ((IoTData) (entry.getValue().content)).getDataSourceUE() == movingUE)
                .collect(Collectors.toList());

        for (Map.Entry<String, LocalContent> entry : entriesForDataPushedByMovingUE) {
            String key = entry.getKey();
            final LocalContent localContent = this.deleteLocalContent(key);
            IoTData deletedData = (IoTData) localContent.content;
            deletedData.updateURI();
            storeContent(localContent.appId, deletedData, localContent.created, localContent.storageReason);
        }
    }

    private LocalContent deleteLocalContent(String URI) {
        LocalContent localContent = null;
        if (contentExists(URI)) {
            localContent = this.contents.remove(URI);
            this.increaseFreeSpace(localContent);
        }
        return localContent;
    }

    public void updateLastAccessedTime(Content content, double timestamp) {
        final LocalContent localContent = contents.get(content.getURI());
        localContent.setLastAccessed(timestamp);
    }

    public void updateStorageReason(Content content, StorageReason storageReason) {
        final LocalContent localContent = contents.get(content.getURI());
        localContent.setStorageReason(storageReason);
    }

    public List<Content> getContentForApp(int appId) {
        final Collection<LocalContent> allContents = contents.values();
        return allContents.stream().filter(c -> c.appId == appId).map(c -> c.content).collect(Collectors.toList());
    }

    public List<Content> getContentForAppAndStorageReason(int appId, StorageReason reason) {
        return contents.values().stream()
                .filter(c -> c.appId == appId & c.storageReason == reason)
                .map(c -> c.content)
                .collect(Collectors.toList());
    }

    //    public double getLastAccessedTime(Content content) {
    //        if (contentExists(content)) {
    //            LocalContent localContent = contents.get(content.getURI());
    //            return localContent.lastAccessed;
    //        }
    //        throw new RuntimeException(String.format("Content %s does not exist in file system for %s %s",
    //                content.getDescription(), host.getTypeAsString(), host.getId()));
    //    }

    static class LocalContent {
        private final int appId;
        private final Content content;
        private final double created;
        private double lastAccessed;
        private StorageReason storageReason;

        LocalContent(int appId, Content content, double created, StorageReason storageReason) {
            this.appId = appId;
            this.content = content;
            this.created = created;
            this.storageReason = storageReason;
            this.lastAccessed = created;
        }

        public void setLastAccessed(double lastAccessed) {
            this.lastAccessed = lastAccessed;
        }

        public void setStorageReason(StorageReason storageReason) {
            this.storageReason = storageReason;
        }
    }
}
