package edu.ucc.network.devices;

import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.AckReceivedEvent;
import edu.ucc.core.events.simulationevents.RequestEvent;
import edu.ucc.core.events.simulationevents.TestbedEvent;
import edu.ucc.core.transport.chunks.ChunksBuffer;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.core.transport.links.FullDuplexLink;
import edu.ucc.core.transport.routing.RoutingTable;
import edu.ucc.utils.TransmissionConstants;
import edu.ucc.utils.TransmissionUtils;

import java.util.ArrayList;
import java.util.List;

import static edu.ucc.utils.Logging.printWarning;
import static edu.ucc.utils.TransmissionConstants.CHUNK_SIZE_FOR_ACK;
import static edu.ucc.utils.TransmissionConstants.PROCESSING_DELAY;

public class EdgeServer extends AdvancedHost {
    /***
     * The total bandwidth available for siblings connection.
     * Recall within interconnected siblings, such bandwidth is shared
     */
    private double bandwidthForSiblings;

    /***
     * The link to sibling units.
     * Notice that a {@link FullDuplexLink} is employed for siblings since the same link provides the DownLink and
     * UpLink for a pair of connected siblings (it is simultaneously affected).
     */
    private List<FullDuplexLink> fullDuplexLinksToSiblings;

    /***
     * The list of references to sibling units.
     */
    private List<Host> siblingUnits;

    /**
     * The list of chunks awaiting for ACK from the parent host.
     */
    protected final ChunksBuffer siblingsChunksAwaitingAck;

    /**
     * The buffer of chunks to deliver to sibling units
     */
    protected final ChunksBuffer siblingsChunksBuffer;

    /**
     * The index of the next sibling host to which content will be transmitted
     */
    private int nextSiblingHostIndex;

    public EdgeServer(int id, long storage, Location location, int mtu) {
        super(id, storage, location, mtu);
        this.siblingUnits = new ArrayList<>();
        this.fullDuplexLinksToSiblings = new ArrayList<>();
        this.siblingsChunksAwaitingAck = new ChunksBuffer(this);
        this.siblingsChunksBuffer = new ChunksBuffer(this);
        this.nextSiblingHostIndex = 0;
    }

    @Override
    public void addChunksForTransmissionToChild(Host nextHopHost, RequestEvent requestEvent, int totalChunks, int chunkSize) {
        if (nextHopHost instanceof BaseStation) {
            for (int chunkNum = 0; chunkNum < totalChunks; chunkNum++) {
                this.downlinkChunksBuffer.addToBuffer(nextHopHost,
                        new ContentChunk(requestEvent, chunkNum, totalChunks, chunkSize));
            }
        } else throw new RuntimeException(String.format("NextHop for downlink should be a BS (An %s was specified)",
                nextHopHost.getTypeAsString()));
    }

    @Override
    public String toString() {
        return String.format("EdgeServer %s with Cloud Unit %s as parent, and %s children BSs",
                id, parentUnit.getId(), childrenUnits.size());
    }

    @Override
    public void setParentUnit(Host newParent) {
        if (newParent instanceof CloudUnit) {
            this.setParent(newParent);
        } else {
            throw new RuntimeException("An EdgeServer's parent must be a Cloud Unit object");
        }
    }

    /***
     * Sets the host's sibling units.
     * @param newSiblings The list of Sibling Units for the host. Notice that type checking is performed (e.g., a BS is
     *                    not a valid sibling unit for an EdgeServer. All the siblings must be of the same type. Some
     *                    units do not support siblings in our implementation (CloudUnit, UEs).
     */
    public void setSiblingUnits(List<Host> newSiblings) {
        for (Host h : newSiblings) {
            if (!(h instanceof EdgeServer)) {
                throw new IllegalArgumentException("One of the new siblings is not of type EdgeServer");
            }
        }

        this.assignSiblings(newSiblings);
    }

    @Override
    public void setChildrenUnits(List<Host> newChildren) {
        for (Host h : newChildren) {
            if (!(h instanceof BaseStation)) {
                throw new IllegalArgumentException("One of the new children is not of type BaseStation");
            }
        }
        this.assignChildren(newChildren);
    }

    public double calculateTxTimeToSibling(int chunkSize, EdgeServer siblingHost) {
        double bandwidthToSibling = findLinkWithSibling(siblingHost).getAvailableBandwidth();// findLinkWithSibling(siblingHost).getTotalBandwidth();
        double transmissionDelay = chunkSize / bandwidthToSibling;
        double distance = this.location.distanceTo(siblingHost.location);
        double propagationDelay = distance / TransmissionConstants.LIGHT_SPEED;

        return transmissionDelay + propagationDelay;
    }

    public boolean containsBaseStation(BaseStation baseStation) {
        return getChildrenUnits().contains(baseStation);
    }

    /***
     * For intra-ES handover, this method again puts the chunks that were in the source BS in the Edge's buffer
     * @param targetBS The new BS the movingUE is now.
     * @param pendingChunksInSourceBS The chunks that need to be sent again to the target BS
     */
    public void reSendChunksFromSourceBS(BaseStation targetBS, List<ContentChunk> pendingChunksInSourceBS) {
        if (pendingChunksInSourceBS.size() > 0) {
            addChunksToDownlinkBuffer(targetBS, pendingChunksInSourceBS);
            sendChunksDownwardsAfterHandover();
        }
    }

    public void updateBSInRoutingTable(UserEquipment userEquipment, BaseStation targetBS) {
        final RoutingTable routingTable = Simulation.getInstance().getNetworkArchitecture().getRoutingTable();
        routingTable.updateBsPartForUserEquipment(userEquipment, targetBS);
    }

    /***
     * Assigns the host's sibling units. The bandwidth is also set in this method.
     * @param newSiblings The list of siblings to assign.
     */
    protected void assignSiblings(List<Host> newSiblings) {
        this.siblingUnits = newSiblings;

        double bandwidthPerSibling = this.bandwidthForSiblings / newSiblings.size();
        validateBandwidth(bandwidthPerSibling);
        for (Host sibling : newSiblings) {
            FullDuplexLink link = findLinkWithSibling((EdgeServer) sibling);
            if (link == null) {
                // then there is no link, we have to create it!
                link = new FullDuplexLink(this, sibling, bandwidthPerSibling);
            }
            this.fullDuplexLinksToSiblings.add(link);
        }
    }

    /***
     * Sets the total bandwidth available for siblings (used in the {@link FullDuplexLink}).
     * @param bandwidthForSiblings The total bandwidth for siblings.
     */
    public void setBandwidthForSiblings(double bandwidthForSiblings) {
        this.bandwidthForSiblings = bandwidthForSiblings;
    }

    /***
     * Gets the total bandwidth for sibling units. Recall that this is shared (full duplex) between a pair of sibling
     * hosts.
     * @return The total bandwidth for the host's sibling units.
     */
    public double getBandwidthForSiblings() {
        return bandwidthForSiblings;
    }

    /***
     * Finds an existing {@link FullDuplexLink} between this host and the specified sibling
     * @param siblingHost The host to look against.
     * @return The corresponding {@link FullDuplexLink} if any
     */
    private FullDuplexLink findLinkWithSibling(EdgeServer siblingHost) {
        List<FullDuplexLink> linksOfSibling = siblingHost.getFullDuplexLinksToSiblings();
        for (FullDuplexLink link : linksOfSibling) {
            // if (link.isThereALinkWith(siblingHost))
            if (link.isThereALinkWith(this))
                return link;
        }
         return null;
//        throw new IllegalArgumentException("The specified host is not a sibling of this host");
    }

    /***
     * Gets the list of {@link FullDuplexLink} for this host's sibling units.
     * @return The list of links to this host's sibling units.
     */
    public List<FullDuplexLink> getFullDuplexLinksToSiblings() {
        return fullDuplexLinksToSiblings;
    }

//    /***
//     * Sets the list of {@link FullDuplexLink} to siblings.
//     * @param linksToSiblings The list of links to set.
//     */
//    public void setLinkToSiblings(List<FullDuplexLink> linksToSiblings) {
//        this.fullDuplexLinksToSiblings = linksToSiblings;
//    }

    /***
     * Sends the ACK about received chunk to sender. Notice that here it can be a parent or children or sibling.
     * @param ackedHost The host to which the ACK must be sent.
     * @param ackedChunk The chunk that has been received and whose ack should be notified
     */
    @Override
    public boolean sendAck(Host ackedHost, ContentChunk ackedChunk) {
        if (super.sendAck(ackedHost, ackedChunk)) return true;
        double txTime;
        if (siblingUnits.contains(ackedHost)) {
            txTime = calculateTxTimeToSibling(CHUNK_SIZE_FOR_ACK, (EdgeServer) ackedHost);
        } else {
            return false;
        }
        double timestamp = Simulation.getInstance().getCurrentTime() + txTime + PROCESSING_DELAY;
        TestbedEvent ackEvent = new AckReceivedEvent(this, ackedHost, timestamp, ackedChunk);
        Simulation.getInstance().postEvent(ackEvent);
        return true;
    }

    public void addChunksForTransmissionToSibling(EdgeServer nextHopES, RequestEvent requestEvent,
                                                  int totalChunks, int chunkSize) {
        for (int chunkNum = 0; chunkNum < totalChunks; chunkNum++) {
            this.siblingsChunksBuffer.addToBuffer(nextHopES,
                    new ContentChunk(requestEvent, chunkNum, totalChunks, chunkSize));
        }
    }

    public void addChunkForTransmissionToSibling(EdgeServer edgeServer, ContentChunk contentChunk) {
        this.siblingsChunksBuffer.addToBuffer(edgeServer, contentChunk);
    }

    /**
     * This method is called to try to relay a chunk recently received by the current host and that is addressed to the
     * child device.
     *
     * @param siblingES The child device to whom chunks must be delivered.
     */
    public void sendChunkForSibling(EdgeServer siblingES) {
        // We will only send if there is nothing pending for this host
        if (siblingsChunksAwaitingAck.noChunksForHost(siblingES)) {
            nextSiblingHostIndex = siblingUnits.indexOf(siblingES);
            final List<ContentChunk> chunksForHost = siblingsChunksBuffer.getChunksForNextHopHost(siblingES);
            sendNextChunkForSibling(siblingES, chunksForHost.get(0));
        }
    }

    private void sendNextChunkForSibling(EdgeServer nextHopHost, ContentChunk nextChunk) {
        FullDuplexLink linkToSibling = findLinkWithSibling(nextHopHost);
        if (linkToSibling.getAvailableBandwidth() > nextChunk.getChunkSize()) {
            siblingsChunksAwaitingAck.addToBuffer(nextHopHost, nextChunk);
            // Simulation.getInstance().getStatisticsCollector().onChunkTransmittedToSibling(this, nextChunk);
            Simulation.getInstance().getEventsBroker().onChunkTransmittedToSibling(this, nextChunk);
            final boolean b = siblingsChunksBuffer.removeChunkFromBuffer(nextHopHost, nextChunk);
            if (!b) throw new RuntimeException("I could not delete the chunk from buffer!!");

            double txTime = calculateTxTimeToSibling(nextChunk.getChunkSize(), nextHopHost);
            txTime += PROCESSING_DELAY; // This adds a processing delay to each sent chunk
            linkToSibling.reduceAvailableBandwidth(nextChunk.getChunkSize());
            TestbedEvent sendChunkEvent = TransmissionUtils.createEventForChunk(this, nextHopHost,
                    txTime, nextChunk);
            Simulation.getInstance().postEvent(sendChunkEvent);
        } else {
            printWarning(String.format("Link bandwidth from %s %s to %s %s is not enough to move %s bits",
                    getTypeAsString(), this.id, nextHopHost.getTypeAsString(), nextHopHost.id,
                    nextChunk.getChunkSize()));
        }
    }

    public void increaseBandwidthWithSibling(EdgeServer siblingHost, ContentChunk chunk) {
        final FullDuplexLink linkWithSibling = findLinkWithSibling(siblingHost);
        linkWithSibling.increaseAvailableBandwidth(chunk.getChunkSize());
    }

    public void deleteChunkFromSiblingsAckBuffer(EdgeServer edgeServer, ContentChunk chunk) {
        this.siblingsChunksAwaitingAck.removeChunkFromBuffer(edgeServer, chunk);
    }

    public void sendChunksForSiblingsAfterAck() {
        EdgeServer nextSibling;
        if (siblingsChunksBuffer.areTherePendingChunks()) {
            // This code searches for the next host with pending chunks.
            List<ContentChunk> chunksForSibling;
            do {
                nextSiblingHostIndex++;
                if (nextSiblingHostIndex >= siblingUnits.size()) {
                    nextSiblingHostIndex = 0;
                }

                nextSibling = (EdgeServer) siblingUnits.get(nextSiblingHostIndex);
                chunksForSibling = siblingsChunksBuffer.getChunksForNextHopHost(nextSibling);
            } while (chunksForSibling.isEmpty());

            final ContentChunk nextChunk = chunksForSibling.get(0);
            sendNextChunkForSibling(nextSibling, nextChunk);
        }
    }

    @Override
    public String getTypeAsString() {
        return "ES";
    }

    public List<Host> getSiblingUnits() {
        return siblingUnits;
    }
}
