package edu.ucc.network.devices;

import edu.ucc.core.transport.links.GroupedLink;

import java.util.List;

import static edu.ucc.utils.Logging.printError;


public class BaseStation extends RelayingHost {
    private final double range;

    public BaseStation(int id, long storage, Location location, int mtu, double range) {
        super(id, storage, location, mtu);
        this.range = range;
    }

    @Override
    public String toString() {
        return String.format("BaseStation %s with Edge Server %s as parent, and %s children UEs",
                id, parentUnit.id, childrenUnits.size());
    }

    @Override
    public void setParentUnit(Host newParent) {
        if (newParent instanceof EdgeServer) {
            this.setParent(newParent);
        } else {
            throw new RuntimeException("A BaseStation's parent must be an EdgeServer object");
        }
    }

    @Override
    public void setChildrenUnits(List<Host> newChildren) {
        for (Host h : newChildren) {
            if (!(h instanceof UserEquipment)) {
                throw new IllegalArgumentException("One of the new children is not of type UserEquipment");
            }
        }

        this.assignChildren(newChildren);
    }


    public void deleteChild(Host child) {
        boolean removed = childrenUnits.remove(child);
        if (!removed) {
            printError("Cannot erase the UE from the list of children for this host");
            throw new RuntimeException("Cannot erase the UE from the list of children for this host");
        }
        child.parentUnit = null;
        child.groupedLinkToParent = null;
        // Here we delete the
        deleteLinkWithChild(child);
    }

    protected void deleteLinkWithChild(Host child) {
        final GroupedLink removed = groupedLinksToChildren.remove(child);
        if (removed == null) {
            printError("Cannot remove the link with specified child");
            throw new RuntimeException("Cannot remove the link with specified child");
        }
        reassignBandwidthWhenRemovingUe();
    }

    private void reassignBandwidthWhenRemovingUe() {
        final int size = childrenUnits.size();
        double uplinkPerChild = totalUplinkBwForChildren / size;
        double downlinkPerChild = totalDownlinkBwForChildren / size;
        for (Host child : childrenUnits) {
            final GroupedLink groupedLink = findLinkWithChild(child);
            final double currentDownloadBw = groupedLink.getTotalBandwidthProvidedByMe(this);
            final double currentUploadBw = groupedLink.getTotalBandwidthProvidedByMe(child);

            final double downloadIncrease = downlinkPerChild - currentDownloadBw;
            final double uploadIncrease = uplinkPerChild - currentUploadBw;

            // This is for updating the total and available bandwidth from the BS towards the child UE (download).
            groupedLink.increaseTotalBandwidthFromMe(this, downloadIncrease);
            groupedLink.increaseBandwidthFromMe(this, downloadIncrease);

            // This is for updating the total and available bandwidth from the child UE towards the BS (upload).
            groupedLink.increaseTotalBandwidthFromMe(child, uploadIncrease);
            groupedLink.increaseBandwidthFromMe(child, uploadIncrease);
        }
    }

    public void appendChild(Host child) {
        if (!(child instanceof UserEquipment)) throw new RuntimeException("The child must be of type User Equipment");
        child.setParentUnit(this);
        appendLinkWithChild(child);
    }

    protected void appendLinkWithChild(Host newChild) {

        final int newNumOfChildren = childrenUnits.size() + 1; // Recall that the new host has not been added yet
        double uplinkPerChild = totalUplinkBwForChildren / newNumOfChildren;
        double downlinkPerChild = totalDownlinkBwForChildren / newNumOfChildren;

        // With the new calculated ul and dl, we update existing bw for existing children
        for (Host child : childrenUnits) {
            final GroupedLink groupedLink = findLinkWithChild(child);
            final double currentDownloadBw = groupedLink.getTotalBandwidthProvidedByMe(this);
            final double currentUploadBw = groupedLink.getTotalBandwidthProvidedByMe(child);

            // (the other way around would cause a negative result...)
            final double downloadDecrease = currentDownloadBw - downlinkPerChild;
            final double uploadDecrease = currentUploadBw - uplinkPerChild;

            // This is for updating the total and available bandwidth from the BS towards the child UE (download).
            groupedLink.decreaseTotalBandwidthFromMe(this, downloadDecrease);
            groupedLink.decreaseBandwidthFromMe(this, downloadDecrease); // This is the one with the issue

            // This is for updating the total and available bandwidth from the child UE towards the BS (upload).
            groupedLink.decreaseTotalBandwidthFromMe(child, uploadDecrease);
            groupedLink.decreaseBandwidthFromMe(child, uploadDecrease);
        }

        // Finally, we add the new UE-child.
        this.childrenUnits.add(newChild);
        // And the corresponding grouped link.
        GroupedLink groupedLink = new GroupedLink(this, newChild, downlinkPerChild, uplinkPerChild);
//        groupedLinksToChildren.add(groupedLink);
        groupedLinksToChildren.put(newChild, groupedLink);
        newChild.groupedLinkToParent = groupedLink;
    }

    public double getRange() {
        return range;
    }

    public boolean containsUserEquipment(UserEquipment userEquipment) {
        return childrenUnits.contains(userEquipment);
    }

    @Override
    public String getTypeAsString() {
        return "BS";
    }
}
