package edu.ucc.graphic;

import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.NetworkArchitecture;
import edu.ucc.network.devices.UserEquipment;

import javax.swing.*;
import java.awt.*;
import java.awt.image.BufferedImage;

public class VisualizationPanel extends JPanel {

    private final NetworkArchitecture networkArchitecture;
    private final ArchitectureParameters architectureParameters;
    private BufferedImage bufferImage;
    private final int width;
    private final int height;

    public VisualizationPanel(NetworkArchitecture networkArchitecture,
                              ArchitectureParameters architectureParameters) {
        super();
        this.networkArchitecture = networkArchitecture;
        this.architectureParameters = architectureParameters;
        this.width = (int) architectureParameters.getMapWidth();
        this.height = (int) architectureParameters.getMapHeight();
        this.bufferImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
        setSize(800, 800);
    }

    @Override
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
//        g.drawImage(bufferImage, 0, 0, null);
        int newImageWidth = 800;
        int newImageHeight = 800;
        BufferedImage resizedImage = new BufferedImage(newImageWidth, newImageHeight, BufferedImage.TYPE_INT_ARGB);
        Graphics2D gg = resizedImage.createGraphics();
        gg.drawImage(bufferImage, 0, 0, newImageWidth, newImageHeight, null);
        g.drawImage(resizedImage, 0, 0, null);
        gg.dispose();
    }

    public void drawBaseStations() {
        Graphics2D graphics = bufferImage.createGraphics();
        Font f = new Font("Arial", Font.BOLD, 200);
        final int radio = (int) architectureParameters.getBsRange();
        final int diameter = radio * 2;
        graphics.setColor(Color.BLUE);
        graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        graphics.setStroke(new BasicStroke(10));
        for (Host bs : networkArchitecture.getBaseStations()) {
            final int x = (int) bs.getLocation().getLatitude();
            final int y = (int) bs.getLocation().getLongitude();
            graphics.drawOval(x - radio, y - radio, diameter, diameter);
            graphics.setFont(f);
            graphics.drawString(String.format("%s", bs.getId()), x, y);
        }
        graphics.dispose();
        repaint();
    }

    public void drawUserEquipments() {
        Graphics graphics = bufferImage.createGraphics();
        int radioForUEs = 160;
        int diameter = radioForUEs * 2;
        Font f = new Font("Segoe UI", Font.ITALIC + Font.BOLD, 200);
        for (UserEquipment ue : networkArchitecture.getUserEquipments()) {
            final int x = (int) ue.getLocation().getLatitude();
            final int y = (int) ue.getLocation().getLongitude();
            graphics.setColor(Color.YELLOW);
            graphics.fillOval(x - radioForUEs, y - radioForUEs, diameter, diameter);
            drawCenteredString(graphics, String.format("%s", ue.getId()),
                    new Rectangle(x - radioForUEs, y - radioForUEs, diameter, diameter), f, Color.GREEN.darker());
        }
        graphics.dispose();
        repaint();

    }

    public void drawEdgeServers() {
        Graphics graphics = bufferImage.createGraphics();
        int radioForServers = 150;
        int diameter = radioForServers * 2;
        Font f = new Font("Arial", Font.BOLD, 200);
        for (EdgeServer edgeServer : networkArchitecture.getEdgeServers()) {
            final int x = (int) edgeServer.getLocation().getLatitude();
            final int y = (int) edgeServer.getLocation().getLongitude();
            graphics.setColor(Color.BLUE.darker());
            graphics.fillOval(x - radioForServers, y - radioForServers, diameter, diameter);
            drawCenteredString(graphics, String.format("%s", edgeServer.getId()),
                    new Rectangle(x - radioForServers, y - radioForServers, diameter, diameter), f, Color.YELLOW);
        }
        graphics.dispose();
        repaint();
    }

    /**
     * Draw a String centered in the middle of a Rectangle.
     *
     * @param g     The Graphics instance.
     * @param text  The String to draw.
     * @param rect  The Rectangle to center the text in.
     * @param font  The font to use.
     * @param color The color to put text
     */
    private void drawCenteredString(Graphics g, String text, Rectangle rect, Font font, Color color) {
        g.setColor(color);
        // Get the FontMetrics
        FontMetrics metrics = g.getFontMetrics(font);
        // Determine the X coordinate for the text
        int x = rect.x + (rect.width - metrics.stringWidth(text)) / 2;
        // Determine the Y coordinate for the text (note we add the ascent, as in java 2d 0 is top of the screen)
        int y = rect.y + ((rect.height - metrics.getHeight()) / 2) + metrics.getAscent();
        // Set the font
        g.setFont(font);
        // Draw the String
        g.drawString(text, x, y);
    }
}
