package edu.ucc.core.transport.chunks;

import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.UserEquipment;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

public class ChunksBuffer implements Serializable {
    private final Map<Host, List<ContentChunk>> chunksPerHosts;
    private final Host host;

    public ChunksBuffer(Host host) {
        this.host = host;
        this.chunksPerHosts = new HashMap<>();
    }

    public void addToBuffer(Host hostDevice, ContentChunk chunk) {
        List<ContentChunk> contentChunks = getChunksForHost(hostDevice);
        contentChunks.add(chunk);
        chunksPerHosts.put(hostDevice, contentChunks);
    }

    private List<ContentChunk> getChunksForHost(Host hostDevice) {
        List<ContentChunk> contentChunks = chunksPerHosts.get(hostDevice);
        if (contentChunks == null) {
            contentChunks = new ArrayList<>();
        }
        return contentChunks;
    }

    public void addToBuffer(Host hostDevice, List<ContentChunk> chunks) {
//        this.chunks.addAll(chunks);
        final List<ContentChunk> contentChunks = getChunksForHost(hostDevice);
        contentChunks.addAll(chunks);
        chunksPerHosts.put(hostDevice, contentChunks);

    }


    public boolean removeChunkFromBuffer(Host hostDevice, ContentChunk chunk) {
        final List<ContentChunk> contentChunks = chunksPerHosts.get(hostDevice);
        if (contentChunks != null) {
            return contentChunks.remove(chunk);
        }
        return false;
    }


    public List<ContentChunk> getChunksForUE(int ueId) {
        final Collection<List<ContentChunk>> allChunks = chunksPerHosts.values();
        List<ContentChunk> chunksList = new ArrayList<>();
        allChunks.forEach(chunksList::addAll);
        // get all chunks with destinationHost of type UE and same ueId
//        return chunksList.stream().filter(c -> c.getContentDestinationHost().getId() == ueId).collect(Collectors.toList());
        return chunksList.stream().filter(ch -> {
            final Host contentDestinationHost = ch.getRequestEvent().getContentDestinationHost();
            return contentDestinationHost instanceof UserEquipment && contentDestinationHost.getId() == ueId;
        }).collect(Collectors.toList());
    }

    /***
     * This method removes all chunks whose final destination is the specified UE
     * @param ueId
     */
    public void removeChunksForMovingUe(int ueId) {
        final Set<Host> hosts = chunksPerHosts.keySet();
        for (Host host : hosts) {
            final List<ContentChunk> contentChunks = chunksPerHosts.get(host);
            // remove those chunks with destinationHost of type UE and with same ueId
            // contentChunks.removeIf(c ->  c.getContentDestinationHost().getId() == ueId);
            contentChunks.removeIf(ch -> {
                final Host contentDestinationHost = ch.getRequestEvent().getContentDestinationHost();
                return contentDestinationHost instanceof UserEquipment
                        && contentDestinationHost.getId() == ueId;
            });
        }
    }

    public List<ContentChunk> getChunksForNextHopHost(Host hostDevice) {
        return getChunksForHost(hostDevice);
    }

    public boolean areTherePendingChunks() {
        final int numChunks = chunksPerHosts.values().stream().mapToInt(List::size).sum();
        return numChunks > 0;
    }

    /***
     * Finds out whether there are chunks in this buffer for the specified host.
     * @param hostDevice The host to look chunk records for.
     * @return true if chunks are found, false otherwise.
     */
    public boolean noChunksForHost(Host hostDevice) {
        final List<ContentChunk> contentChunks = chunksPerHosts.get(hostDevice);
        if (contentChunks == null) {
            return true;
        }
        return contentChunks.size() <= 0;
    }

    public void removeChunksForHost(Host parentHost) {
        chunksPerHosts.remove(parentHost);
    }
}
