package edu.ucc.core.simpleimplementations;

import edu.ucc.core.ContentTransmitter;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.core.transport.routing.RoutingTable;
import edu.ucc.entities.Content;
import edu.ucc.network.devices.*;
import edu.ucc.testbedinterface.ITestbedEvents;
import edu.ucc.utils.TransmissionUtils;

import static edu.ucc.utils.Logging.printWarning;
import static edu.ucc.utils.TransmissionUtils.*;

/**
 * This class (implementation of {@link ContentTransmitter}) defines how edge servers handle content requests and associated chunk transmissions.
 */
public class EdgeContentTransmitter implements ContentTransmitter {
    @Override
    public void onPullRequestReceived(Host host, PullRequestEvent pullRequestEvent) {
        EdgeServer edgeServer = (EdgeServer) host;
        Content referredContent = pullRequestEvent.getReferredContent();
        boolean servedLocally = false;
        switch (pullRequestEvent.getEventDirection()) {
            case UPWARDS_DIRECTION:
                //                Record record = records.computeIfAbsent(host.getId(), k -> new Record());
                if (pullRequestEvent instanceof PullFileRequestEvent) {
                    if (edgeServer.getFileSystem().contentExists(referredContent)) {
                        //  Simulation.getInstance().getEventsBroker().onPullRequestReceived(edgeServer, pullRequestEvent);
                        servedLocally = true;
                        Simulation.getInstance().getEventsBroker().onPullRequestServedLocally(edgeServer, pullRequestEvent);
                        processUpwardPullRequestLocally(edgeServer, pullRequestEvent);
                        // record.hit++;
                    } else {
                        TransmissionUtils.propagatePullRequestToParent(edgeServer, pullRequestEvent);
                        // record.miss++;
                    }
                } else {
                    TransmissionUtils.propagatePullRequestToParent(edgeServer, pullRequestEvent);
                    // record.miss++;
                }
                // System.out.printf("Record at ES %s is hit: %s,  miss: %s", host.getId(), record.hit, record.miss);
                break;
            case SAME_LEVEL_DIRECTION:
                throw new RuntimeException(String.format("Same level pull request at %s", edgeServer));
            case DOWNWARDS_DIRECTION:
                final Host dataLocationHost = pullRequestEvent.getPullDestinationHost();
                if (!(dataLocationHost instanceof UserEquipment))
                    throw new RuntimeException("Pull Request in the downlink but dataLocationHost is not a UE");

                // new handling
                if (edgeServer.getFileSystem().contentExists(referredContent)) {
                    servedLocally = true;
                    Simulation.getInstance().getEventsBroker().onPullRequestServedLocally(edgeServer, pullRequestEvent);
                    processDownwardPullRequestLocally(edgeServer, pullRequestEvent);
                } else {
                    final RoutingTable routingTable = Simulation.getInstance().getNetworkArchitecture().getRoutingTable();
                    final BaseStation nextChildHop = routingTable.getEntryForUe((UserEquipment) dataLocationHost).getBaseStationPartInRoute();
                    propagatePullRequestToChild(edgeServer, nextChildHop, pullRequestEvent);
                }
                break;
            default:
                throw new RuntimeException("Unrecognized event direction");
        }
        if (!servedLocally)
            Simulation.getInstance().getEventsBroker().onPullRequestReceived(edgeServer, pullRequestEvent);
    }

    /**
     * This method serves the upward request (requested by a UE and coming from a BS) locally, i.e., it takes the
     * content and send it back in the downlink
     *
     * @param edgeServer   The edge server processing the request.
     * @param requestEvent The received request.
     */
    private void processUpwardPullRequestLocally(EdgeServer edgeServer, PullRequestEvent requestEvent) {
        final RoutingTable routingTable = Simulation.getInstance().getNetworkArchitecture().getRoutingTable();
        // Recall that due to handover, we might receive the request from an ES and send the response through another.
        final UserEquipment userEquipment = (UserEquipment) requestEvent.getPullInitiatorHost();
        final BaseStation nextBaseStation = routingTable.getEntryForUe(userEquipment).getBaseStationPartInRoute();
        final Content referredContent = requestEvent.getReferredContent();
        final Content content = edgeServer.getFileSystem().getTouchContent(referredContent.getURI(), requestEvent.getTimestamp());
        int chunkSize = edgeServer.getMtu();
        int totalChunks = getNumberOfChunksForContent(content, chunkSize);
        edgeServer.addChunksForTransmissionToChild(nextBaseStation, requestEvent, totalChunks, chunkSize);
        edgeServer.sendChunkDownwardsForHost(nextBaseStation);
    }

    /**
     * This method serves the downward request (requested by the cloud or another UE and coming from the cloud, hence
     * the downward characteristic) locally, i.e., it takes the content and send it back in the uplink.
     *
     * @param edgeServer   The edge server processing the request.
     * @param requestEvent The received request.
     */
    private void processDownwardPullRequestLocally(EdgeServer edgeServer, PullRequestEvent requestEvent) {
        final Content referredContent = requestEvent.getReferredContent();
        final Content content = edgeServer.getFileSystem().getTouchContent(referredContent.getURI(),
                requestEvent.getTimestamp());
        int chunkSize = edgeServer.getMtu();
        int totalChunks = getNumberOfChunksForContent(content, chunkSize);
        edgeServer.addChunksForTransmissionToParent(requestEvent, totalChunks, chunkSize);
        edgeServer.sendChunksToParent();
    }

    @Override
    public void onAckReceived(Host host, AckReceivedEvent ackReceivedEvent) {
        EdgeServer edgeServer = (EdgeServer) host;
        switch (ackReceivedEvent.getEventDirection()) {
            case UPWARDS_DIRECTION:
                processAckUpwards(edgeServer, ackReceivedEvent);
                break;
            case SAME_LEVEL_DIRECTION:
                processAckSameLevel(edgeServer, ackReceivedEvent);
                break;
            case DOWNWARDS_DIRECTION:
                processAckDownwards(edgeServer, ackReceivedEvent);
                break;
            default:
                throw new RuntimeException("Just for completing cases");
        }
    }

    private void processAckUpwards(EdgeServer edgeServer, AckReceivedEvent ackReceivedEvent) {
        ContentChunk chunk = ackReceivedEvent.getContentChunk();
        final BaseStation baseStation = (BaseStation) ackReceivedEvent.getEventOriginHost();
        edgeServer.deleteChunkFromDownlinkAckBuffer(baseStation, chunk);
        edgeServer.sendChunksDownwardsAfterAck();
    }

    private void processAckSameLevel(EdgeServer edgeServer, AckReceivedEvent ackReceivedEvent) {
        final EdgeServer ackerEdgeServer = (EdgeServer) ackReceivedEvent.getEventOriginHost();
        edgeServer.deleteChunkFromSiblingsAckBuffer(ackerEdgeServer, ackReceivedEvent.getContentChunk());
        edgeServer.sendChunksForSiblingsAfterAck();
    }

    private void processAckDownwards(EdgeServer edgeServer, AckReceivedEvent ackReceivedEvent) {
        ContentChunk chunk = ackReceivedEvent.getContentChunk();
        edgeServer.deleteChunkFromUplinkAckBuffer(chunk);
        edgeServer.sendChunksToParentAfterAck();
    }

    @Override
    public void onChunkReceived(Host host, ChunkReceivedEvent chunkReceivedEvent) {
        EdgeServer edgeServer = (EdgeServer) host;
        final ContentChunk chunk = chunkReceivedEvent.getChunk();
        final Host eventOriginHost = chunkReceivedEvent.getEventOriginHost();

        // We might receive a chunk to be transmitted to a UE not in the same ES anymore; we still update stats though
        final ITestbedEvents eventsBroker = Simulation.getInstance().getEventsBroker();
        eventsBroker.onChunkReceived(edgeServer, chunkReceivedEvent);
        final int appId = chunk.getRequestEvent().getAppId();

        // alright, this modification is important
        switch (chunkReceivedEvent.getEventDirection()) {
            case UPWARDS_DIRECTION:
                edgeServer.increaseBandwidthFromChild(chunk, eventOriginHost);
                if (chunk.getRequestEvent() instanceof PushDataRequestEvent) {
                    final PushDataRequestEvent originalPDREvent = (PushDataRequestEvent) chunk.getRequestEvent();
                    // For chunks regarding data pushed from UEs, we need to know if the edge level is the destination!
                    // We only let it pass if chunk must reach the cloud.
                    // In other words, we stop the uplink transmission if the push was intended to arrive to an ES
                    // (whether specifically this one or not (due to handovers).
                    if (originalPDREvent.getPushDestinationHost() instanceof CloudUnit) {
                        sendChunkUpwards(edgeServer, chunk);
                    } else {
                        // Destination is the edge, we need to find if the app is running here.
                        final boolean isAppDeployedHere = eventsBroker.isAppDeployedHere(edgeServer, appId);
                        if (!isAppDeployedHere) {
                            Host nextHost = eventsBroker.findDestinationForChunkAtServerWithoutApp(chunk, edgeServer, appId);
                            if (nextHost instanceof CloudUnit) {
                                sendChunkUpwards(edgeServer, chunk);
                            } else {
                                EdgeServer destinationES = (EdgeServer) nextHost;
                                edgeServer.addChunkForTransmissionToSibling(destinationES, chunk);
                                edgeServer.sendChunkForSibling(destinationES);
                            }
                        }
                    }
                    // else {
                    //    if (originalPDREvent.getPushDestinationHost() != edgeServer) {
                    //        System.out.println("Ok it happened, we received a chunk pushed by a UE that moved after " +
                    //                String.format("workload was generated. OriginalDestinationES: %s. I am %s",
                    //                        originalPDREvent.getPushDestinationHost(), edgeServer));
                    //    } else {
                    //          System.out.println(String.format("Received a chunk that is for me %s", edgeServer));
                    //    }
                    // }
                }
                // transmissions for other workload.requests go as intended.
                else if (!chunk.isThisDestinationHost(edgeServer)) sendChunkUpwards(edgeServer, chunk);
                break;
            case SAME_LEVEL_DIRECTION:
                edgeServer.increaseBandwidthWithSibling((EdgeServer) eventOriginHost, chunk);
                //  if (!chunk.isThisDestinationHost(edgeServer))
                //    throw new RuntimeException("Same level tx but ES is not destination");
                break;
            case DOWNWARDS_DIRECTION:
                edgeServer.increaseBandwidthFromParent(chunk);
                if (!chunk.isThisDestinationHost(edgeServer)) sendChunkDownwards(edgeServer, chunkReceivedEvent);
                break;
            default:
                throw new RuntimeException("Unrecognized direction");
        }
        edgeServer.sendAck(chunkReceivedEvent.getEventOriginHost(), chunk);
    }

    private void sendChunkUpwards(EdgeServer edgeServer, ContentChunk chunk) {
        edgeServer.addChunkForTransmissionToParent(chunk);
        edgeServer.sendChunksToParent();
    }

    private void sendChunkDownwards(EdgeServer edgeServer, ChunkReceivedEvent chunkReceivedEvent) {
        ContentChunk chunk = chunkReceivedEvent.getChunk();
        final RequestEvent requestEvent = chunk.getRequestEvent();
        int requestId = requestEvent.getRequestId();

        Host requestingHost = requestEvent.getContentDestinationHost();
        final double eventTimestamp = chunkReceivedEvent.getTimestamp();

        BaseStation baseStation = (BaseStation) TransmissionUtils.findNextHopForChunkDownlink((UserEquipment) requestingHost, edgeServer);

        if (edgeServer.containsBaseStation(baseStation)) {
            edgeServer.addChunkToDownlinkBuffer(baseStation, chunk);
            edgeServer.sendChunkDownwardsForHost(baseStation);
        } else {
            printWarning((String.format("At ES %s: DISCARDING Chunk %s for request %s received @ %s, BS and ES " +
                    "changed due to HO", edgeServer.getId(), chunk.getChunkNumber(), requestId, eventTimestamp)));
        }
    }

    @Override
    public void onPushRequestReceived(Host host, PushRequestEvent pushRequestEvent) {
        EdgeServer edgeServer = (EdgeServer) host;
        if (pushRequestEvent.getPushDestinationHost() instanceof EdgeServer) {
            final EdgeServer pushDestinationES = (EdgeServer) pushRequestEvent.getContentDestinationHost();
            if (edgeServer == pushDestinationES) throw new RuntimeException("Can't push to the same Edge Server");
            Simulation.getInstance().getEventsBroker().onPushRequestReceived(edgeServer, pushRequestEvent);
            final Content referredContent = pushRequestEvent.getReferredContent();

            if (referredContent.isCachable()) {
                if (edgeServer.getFileSystem().contentExists(referredContent)) {
                    final int chunkSize = edgeServer.getMtu();
                    final int totalChunks = getNumberOfChunksForContent(referredContent, chunkSize);
                    // final EdgeServer pushDestinationES = (EdgeServer) pushRequestEvent.getPushDestinationHost();
                    edgeServer.addChunksForTransmissionToSibling(pushDestinationES, pushRequestEvent, totalChunks, chunkSize);
                    edgeServer.sendChunkForSibling(pushDestinationES);
                } else {
                    printWarning(String.format("Pushing content rejected: the referred content %s does not exist at " +
                            "pushing %s %s", referredContent.getURI(), edgeServer.getTypeAsString(), edgeServer.getId()));
                }
            } else {
                final int chunkSize = edgeServer.getMtu();
                final int totalChunks = getNumberOfChunksForContent(referredContent, chunkSize);
                // final EdgeServer pushDestinationES = (EdgeServer) pushRequestEvent.getPushDestinationHost();
                edgeServer.addChunksForTransmissionToSibling(pushDestinationES, pushRequestEvent, totalChunks, chunkSize);
                edgeServer.sendChunkForSibling(pushDestinationES);
            }

            // Had to replace this with the previous lines to avoid making a reference to a class that the testbed
            // does not recognize (the eventdata is part of the solution module, not the testbed).

            //            // Event data is never in storage, we just consider it to be in a buffer
            //            if (referredContent instanceof EventData) {
            //                final int chunkSize = edgeServer.getMtu();
            //                final int totalChunks = getNumberOfChunksForContent(referredContent, chunkSize);
            //                // final EdgeServer pushDestinationES = (EdgeServer) pushRequestEvent.getPushDestinationHost();
            //                edgeServer.addChunksForPushRequestToNeighbor(pushDestinationES, pushRequestEvent, totalChunks, chunkSize);
            //                edgeServer.sendChunkForSibling(pushDestinationES);
            //            } else {
            //                if (edgeServer.getFileSystem().contentExists(referredContent)) {
            //                    final int chunkSize = edgeServer.getMtu();
            //                    final int totalChunks = getNumberOfChunksForContent(referredContent, chunkSize);
            //                    // final EdgeServer pushDestinationES = (EdgeServer) pushRequestEvent.getPushDestinationHost();
            //                    edgeServer.addChunksForPushRequestToNeighbor(pushDestinationES, pushRequestEvent, totalChunks, chunkSize);
            //                    edgeServer.sendChunkForSibling(pushDestinationES);
            //                } else {
            //                    printWarning(String.format("Pushing content rejected: the referred content %s does not exist at " +
            //                            "pushing %s %s", referredContent.getURI(), edgeServer.getTypeAsString(), edgeServer.getId()));
            //                }
            //            }
        } else if (pushRequestEvent.getPushDestinationHost() instanceof CloudUnit) {
            Simulation.getInstance().getEventsBroker().onPushRequestReceived(edgeServer, pushRequestEvent);
            // We assume that content is here, so we just take it from the request.
            final Content referredContent = pushRequestEvent.getReferredContent();
            final int chunkSize = edgeServer.getMtu();
            int totalChunks = getNumberOfChunksForContent(referredContent, chunkSize);
            edgeServer.addChunksForTransmissionToParent(pushRequestEvent, totalChunks, chunkSize);
            edgeServer.sendChunksToParent();
        }
    }

    //    class Record {
    //        int hit;
    //        int miss;
    //    }
    //
    //    Map<Integer, Record> records = new HashMap<>();
}
