package edu.ucc.core.simpleimplementations;

import edu.ucc.core.ContentTransmitter;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.core.transport.routing.RoutingTable;
import edu.ucc.entities.Content;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.UserEquipment;
import edu.ucc.utils.TransmissionUtils;

import static edu.ucc.utils.TransmissionUtils.*;

/**
 * This class (implementation of {@link ContentTransmitter}) defines how the cloud handles content requests
 * and the associated chunk transmissions.
 */
public class CloudContentTransmitter implements ContentTransmitter {
    @Override
    public void onPullRequestReceived(Host host, PullRequestEvent pullRequestEvent) {
        CloudUnit cloud = (CloudUnit) host;
        final Host contentLocationHost = pullRequestEvent.getContentLocationHost();
        Simulation.getInstance().getEventsBroker().onPullRequestReceived(cloud, pullRequestEvent);

        switch (pullRequestEvent.getEventDirection()) {
            case UPWARDS_DIRECTION:
                if (contentLocationHost == cloud) {
                    // if it is a request for the cloud, we complete it locally
                    processPullRequestLocally(cloud, pullRequestEvent);
                } else {
                    // if not, then this pull request is for a UE, we must pass it towards that!
                    if (!(contentLocationHost instanceof UserEquipment)) {
                        throw new RuntimeException("Pull Request in the downlink but dataLocationHost is not a UE");
                    }
                    passRequestInDownlinkToUE(cloud, pullRequestEvent, (UserEquipment) contentLocationHost);
                }
                break;
            case SAME_LEVEL_DIRECTION:
                throw new RuntimeException(String.format("Same level pull request at %s", cloud));
            case DOWNWARDS_DIRECTION:
                if (!(contentLocationHost instanceof UserEquipment))
                    throw new RuntimeException("Pull Request started by the cloud downwards but dataLocationHost is not a UE");
                passRequestInDownlinkToUE(cloud, pullRequestEvent, (UserEquipment) contentLocationHost);
                break;
            default:
                throw new RuntimeException("Unrecognized event direction");
        }
    }

    private void passRequestInDownlinkToUE(CloudUnit cloud, PullRequestEvent requestEvent, UserEquipment contentLocationHost) {
        final RoutingTable routingTable = Simulation.getInstance().getNetworkArchitecture().getRoutingTable();
        final EdgeServer nextChildHop = routingTable.getEntryForUe(contentLocationHost).getEdgeServerPartInRoute();
        propagatePullRequestToChild(cloud, nextChildHop, requestEvent);
    }

    private void processPullRequestLocally(CloudUnit cloud, PullRequestEvent requestEvent) {
        final RoutingTable routingTable = Simulation.getInstance().getNetworkArchitecture().getRoutingTable();
        // Recall that due to handover, we might receive the request from an ES and send the response through another.
        final UserEquipment userEquipment = (UserEquipment) requestEvent.getPullInitiatorHost();
        final EdgeServer nextHopES = routingTable.getEntryForUe(userEquipment).getEdgeServerPartInRoute();

        final Content referredContent = requestEvent.getReferredContent();
        int chunkSize = cloud.getMtu();
        int totalChunks = getNumberOfChunksForContent(referredContent, chunkSize);
        cloud.addChunksForTransmissionToChild(nextHopES, requestEvent, totalChunks, chunkSize);
        cloud.sendChunkDownwardsForHost(nextHopES);
    }
    
    @Override
    public void onAckReceived(Host host, AckReceivedEvent ackReceivedEvent) {
        // ACKs received by the cloud always come from EdgeServers, that's why we don't have a switch here
        CloudUnit cloudUnit = (CloudUnit) host;
        Simulation.getInstance().getEventsBroker().onAckReceived(cloudUnit, ackReceivedEvent);
        EdgeServer edgeServer = (EdgeServer) ackReceivedEvent.getEventOriginHost();
        ContentChunk chunk = ackReceivedEvent.getContentChunk();
        cloudUnit.deleteChunkFromDownlinkAckBuffer(edgeServer, chunk);
        cloudUnit.sendChunksDownwardsAfterAck();
    }

    @Override
    public void onChunkReceived(Host host, ChunkReceivedEvent chunkReceivedEvent) {
        CloudUnit cloud = (CloudUnit) host;
        final ContentChunk chunk = chunkReceivedEvent.getChunk();
        final RequestEvent requestEvent = chunkReceivedEvent.getChunk().getRequestEvent();

        //        if (!(requestEvent instanceof PushDataRequestEvent || requestEvent instanceof PullDataRequestEvent)) {
        //            throw new RuntimeException("The cloud only processes incoming chunks regarding Data workload.requests");
        //        }

        final Host originHost = chunkReceivedEvent.getEventOriginHost();
        Simulation.getInstance().getEventsBroker().onChunkReceived(cloud, chunkReceivedEvent);

        // Regardless of requesting host, we ACK
        cloud.sendAck(originHost, chunk);
        cloud.increaseBandwidthFromChild(chunk, originHost);

        // We resend chunks only if the cloud is not the destination
        if (!chunk.isThisDestinationHost(cloud)) {
            // Currently, the cloud only receives a chunk to be retransmitted if it belongs to a UE-UE pull data request
            if (requestEvent instanceof PullRequestEvent) {
                final PullRequestEvent asPullEvent = (PullRequestEvent) requestEvent;
                sendChunksDownwards(cloud, chunkReceivedEvent, asPullEvent);
            }
            //            else {
            //                printWarning("Unhandled case: a chunk received in the cloud regarding a PushRequest on " +
            //                        "which the cloud is not the destination");
            //            }
        }
    }

    @Override
    public void onPushRequestReceived(Host host, PushRequestEvent pushRequestEvent) {
        CloudUnit cloud = (CloudUnit) host;
        //  Time passes so fast, now it is possible because the cloud processes PushDataRequests for sharing storage changes
        //        if (pushRequestEvent instanceof PushDataRequestEvent)
        //            throw new RuntimeException("Not implemented yet, pushing data from the cloud");

        Simulation.getInstance().getEventsBroker().onPushRequestReceived(cloud, pushRequestEvent);

        final Content referredContent = pushRequestEvent.getReferredContent();
        int chunkSize = cloud.getMtu();
        int totalChunks = getNumberOfChunksForContent(referredContent, chunkSize);
        final EdgeServer pushDestinationES = (EdgeServer) pushRequestEvent.getPushDestinationHost();
        cloud.addChunksForTransmissionToChild(pushDestinationES, pushRequestEvent, totalChunks, chunkSize);
        cloud.sendChunkDownwardsForHost(pushDestinationES);
    }

    private void sendChunksDownwards(CloudUnit cloud, ChunkReceivedEvent chunkReceivedEvent, PullRequestEvent pullRequestEvent) {
        final ContentChunk chunk = chunkReceivedEvent.getChunk();
        final UserEquipment userEquipment = (UserEquipment) pullRequestEvent.getPullInitiatorHost();
        Host nextHop = TransmissionUtils.findNextHopForChunkDownlink(userEquipment, cloud);
        cloud.addChunkToDownlinkBuffer(nextHop, chunk);
        cloud.sendChunkDownwardsForHost(nextHop);
    }
}
