package edu.ucc.core;

import edu.ucc.core.events.simulationevents.SimulationEvent;
import edu.ucc.network.devices.NetworkArchitecture;
import edu.ucc.statisticscollection.StatisticsCollector;
import edu.ucc.testbedinterface.ITestbedEvents;
import edu.ucc.testbedinterface.ObservableTestbedEventListener;
import edu.ucc.testbedinterface.TestbedBroker;
import edu.ucc.workload.Workload;

import java.io.Serializable;
import java.util.List;
import java.util.Objects;
import java.util.TreeSet;

import static edu.ucc.utils.Logging.printInfo;

public class Simulation implements Serializable {
    public static int REQUEST_ID = 0;
    private final TreeSet<SimulationEvent> eventsQueue;
    private NetworkArchitecture networkArchitecture;
    private Workload workload;

    private double simulationClock;
    private double simulationLength;
    private double runningTime;

    private StatisticsCollector statisticsCollector;

    private static Simulation simulation = null;


    public static Simulation getInstance() {
        if (simulation == null) {
            simulation = new Simulation();
        }
        return simulation;
    }

    public void init(NetworkArchitecture networkArchitecture, Workload workload,
                     ObservableTestbedEventListener solution,
                     double simulationLength) {
        this.networkArchitecture = networkArchitecture;
        this.workload = workload;
        this.statisticsCollector = new StatisticsCollector(networkArchitecture.getCloudRoot(),
                networkArchitecture.getEdgeServers(), networkArchitecture.getBaseStations(),
                networkArchitecture.getUserEquipments());
        TestbedBroker eventsBroker = new TestbedBroker(this.statisticsCollector, solution);
        this.networkArchitecture.setiEventsBroker(eventsBroker);
        this.simulationLength = simulationLength;
        printInfo("Generating initial workload");
        generateInitialEventsFromWorkload();
        registerInitialLocations(eventsBroker, workload);

        final List<SimulationEvent> watchdogEvent = solution.createInitialControlEvents();
        if (null != watchdogEvent) eventsQueue.addAll(watchdogEvent);
    }

    private void registerInitialLocations(TestbedBroker eventsBroker, Workload workload) {
        eventsBroker.registerInitialMobilityEvent(workload.getInitialMobilityEvents());
    }

    private Simulation() {
        this.eventsQueue = new TreeSet<>();
        this.simulationClock = 0;
    }

    public double getCurrentTime() {
        return this.simulationClock;
    }

    public double getSimulationLength() {
        return simulationLength;
    }

    public void startSimulation() {
        long startTimestamp = System.currentTimeMillis();
        System.out.println("Starting simulation");
        while (true) {
            if (this.simulationClock >= this.simulationLength) {
                System.out.println("Total simulation time reached");
                break;
            }
            if (this.eventsQueue.isEmpty()) {
                System.out.println("No more events in queue to process");
                break;
            }

            SimulationEvent nextEvent = eventsQueue.pollFirst();
            this.setSimulationClock(Objects.requireNonNull(nextEvent).getTimestamp());
            nextEvent.execute();
        }
        System.out.println("Simulation ended");
        long endTimestamp = System.currentTimeMillis();
        runningTime = (endTimestamp - startTimestamp) / 1000.0;
        System.out.printf("Simulation time in this computer: %s%n", runningTime);
    }

    public void setSimulationClock(double simulationClock) {
        this.simulationClock = simulationClock;
    }

    private void generateInitialEventsFromWorkload() {
        eventsQueue.addAll(workload.getMobilityEvents());
        eventsQueue.addAll(workload.getRequestEvents());
    }

    public void postEvent(SimulationEvent simulationEvent) {
        eventsQueue.add(simulationEvent);
    }

    public NetworkArchitecture getNetworkArchitecture() {
        return networkArchitecture;
    }

    public ITestbedEvents getEventsBroker() {
        return networkArchitecture.getiEventsBroker();
    }

    public Workload getWorkload() {
        return workload;
    }

    public StatisticsCollector getStatisticsCollector() {
        return statisticsCollector;
    }

    // Only to be called after a simulation is done and we need to run another one.
    // This is due to the singleton behaviour.
    // Before calling this method, you should get the statisticsCollector for studying results.
    public void restart() {
        REQUEST_ID = 0;
        simulationClock = 0;
        runningTime = 0;
        eventsQueue.clear();
    }

    public double getRunningTime() {
        return runningTime;
    }
}
