package core.processingchain.events.detectors.pool.popularity;

import core.context.ContextAPI;
import core.processingchain.events.SolutionEvent;
import edu.ucc.entities.Content;
import edu.ucc.network.devices.Host;

import java.util.*;
import java.util.stream.Collectors;

import static core.processingchain.events.detectors.pool.popularity.PopularityChange.*;

public class PopularityManager {
    private final Map<Host, Map<Content, Boolean>> contentPopularity;

    public PopularityManager() {
        this.contentPopularity = new HashMap<>();
    }

    /**
     * Obtains the popularity status for the given content at the specified host.
     *
     * @param host    The host where popularity will be investigated.
     * @param content The content to look popularity for.
     * @return A PopularityStatus regarding the specified content.
     */
    PopularityStatus obtainContentStatusPerHostAndContent(Host host, Content content) {
        Map<Content, Boolean> statuses = contentPopularity.get(host);
        if (statuses == null) {
            return new PopularityStatus(content, false);
        }

        Boolean popularityStatus = statuses.get(content);
        return new PopularityStatus(content, Objects.requireNonNullElse(popularityStatus, false));
    }

    void setPopularityStatus(Host host, PopularityStatus popularityStatus) {
        Map<Content, Boolean> statuses = contentPopularity.get(host);
        if (statuses == null) {
            statuses = new HashMap<>();
        }

        statuses.put(popularityStatus.content, popularityStatus.popular);
        contentPopularity.put(host, statuses);
    }

    boolean isContentPopular(ContextAPI contextAPI, Host localHost, Content content, double analysisTimestamp,
                             double timeWindow, int requestThreshold) {
        final double startTime = Math.max(0, analysisTimestamp - timeWindow);
        final List<SolutionEvent> requestsInInterval = contextAPI.getRequestsPerContentAndIntervalAndHost(content,
                startTime, analysisTimestamp, localHost);
        //        printDebug(String.format("Content %s has been requested %s times during the interval [%s <-> %s] in this host",
        //                content.getDescription(),
        //                requestsInInterval.size(),
        //                startTime,
        //                analysisTimestamp
        //        ));

        return requestsInInterval.size() >= requestThreshold;
    }

    PopularityChange detectAndHandlePopularityChange(Content content, ContextAPI contextAPI, Host host,
                                                     double analysisTimestamp, double timeWindow, int requestThreshold) {
        boolean currentPopularity = isContentPopular(contextAPI, host, content, analysisTimestamp, timeWindow, requestThreshold);
        final PopularityStatus mostRecentPopularityStatus = obtainContentStatusPerHostAndContent(host, content);
        if (currentPopularity) {
            if (!mostRecentPopularityStatus.isPopular()) {
                mostRecentPopularityStatus.setPopular(true);
                setPopularityStatus(host, mostRecentPopularityStatus);

                return CHANGE_NOW_POPULAR;
            } else {
                return NO_CHANGE_STILL_POPULAR;
            }
        } else {
            if (mostRecentPopularityStatus.isPopular()) {
                mostRecentPopularityStatus.setPopular(false);
                setPopularityStatus(host, mostRecentPopularityStatus);

                return CHANGE_NOW_NOT_POPULAR;
            } else {
                return NO_CHANGE_STILL_NOT_POPULAR;
            }
        }
    }

    public List<Content> getNotPopularContentForHost(Host host) {
        Map<Content, Boolean> contentsPerHost = contentPopularity.get(host);
        if (contentsPerHost == null) return new ArrayList<>();
        final List<Content> notPopularContent = contentsPerHost.entrySet()
                .stream()
                .filter(contentBooleanEntry -> !contentBooleanEntry.getValue())
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
        return notPopularContent;
    }
}
