package core.processingchain.events.detectors.pool.popularity;

import core.context.ContextAPI;
import core.processingchain.events.SolutionEvent;
import core.processingchain.events.detectors.BaseEventDetector;
import core.processingchain.events.pool.HighPopularityEvent;
import core.processingchain.events.pool.LowPopularityEvent;
import core.processingchain.preprocessors.pool.PPOutcomePullRequest;
import edu.ucc.core.events.simulationevents.PullRequestEvent;
import edu.ucc.entities.Content;
import edu.ucc.network.devices.Host;

import java.util.List;

import static core.processingchain.events.detectors.pool.popularity.PopularityChange.CHANGE_NOW_NOT_POPULAR;
import static core.processingchain.events.detectors.pool.popularity.PopularityChange.CHANGE_NOW_POPULAR;
import static edu.ucc.utils.Logging.printInfo;

public class PopularityEventDetector extends BaseEventDetector<PPOutcomePullRequest> {
    private final double timeWindow;
    private final int requestThreshold;
    PopularityManager popularityManager;

    public PopularityEventDetector(double timeWindow, int requestThreshold) {
        this.timeWindow = timeWindow;
        this.requestThreshold = requestThreshold;
        this.popularityManager = new PopularityManager();
    }

    @Override
    public SolutionEvent execute(ContextAPI contextAPI, Host localHost, PPOutcomePullRequest ppOutcome) {
        final PullRequestEvent requestEvent = (PullRequestEvent) ppOutcome.getObservableTestbedEvent();
        final double pullRequestTimestamp = requestEvent.getTimestamp();
        final Content content = requestEvent.getReferredContent();
        final double eventDetectionDelay = 0.00015;
        final double popularityEventTs = ppOutcome.getTimestamp() + eventDetectionDelay;
        final int appId = ppOutcome.getAppId();
        PopularityChange popularityChange = popularityManager.detectAndHandlePopularityChange(content, contextAPI,
                localHost, pullRequestTimestamp, timeWindow, requestThreshold);

        if (popularityChange == CHANGE_NOW_POPULAR) {
            printInfo(String.format("PopularityEventDetector: %s %s. Content %s IS NOW popular @ %s",
                    localHost.getTypeAsString(),
                    localHost.getId(),
                    content.getDescription(),
                    popularityEventTs
            ));
            return new HighPopularityEvent(appId, localHost, ppOutcome, null, popularityEventTs, content);
        } else if (popularityChange == CHANGE_NOW_NOT_POPULAR) {
            printInfo(String.format("PopularityEventDetector: %s %s. Content %s is NOW NOT popular @ %s",
                    localHost.getTypeAsString(),
                    localHost.getId(),
                    content.getDescription(),
                    popularityEventTs
            ));
            return new LowPopularityEvent(appId, localHost, ppOutcome, null, popularityEventTs, content);
        }

        return null;
    }

    public void updatePopularityForContents(Host localHost, ContextAPI contextAPI, double analysisTimestamp) {
        final List<Content> contentForApp = localHost.getFileSystem().getContentForApp(contextAPI.getApp().getAppId());
        for (Content content : contentForApp) {
            popularityManager.detectAndHandlePopularityChange(content, contextAPI, localHost, analysisTimestamp, timeWindow, requestThreshold);
        }
    }

    public List<Content> getNotPopularContentForHost(Host host) {
        return popularityManager.getNotPopularContentForHost(host);
    }

    /**
     * This method is called whenever a content is stored due to external popularity, opportunistic, and direct push request.
     *
     * @param host    The host where content is stored
     * @param content The new content stored
     */
    public void onContentAdded(Host host, Content content) {
        PopularityStatus status = new PopularityStatus(content, false);
        popularityManager.setPopularityStatus(host, status);
    }
}