package core.processingchain.actions.pool;

import core.app.App;
import core.processingchain.actions.ExternalAction;
import core.processingchain.actions.exceptions.ContentNotFoundException;
import core.processingchain.actions.exceptions.SolutionException;
import core.processingchain.actions.policies.BasePolicy;
import core.processingchain.actions.policies.pool.PushPolicy;
import core.processingchain.events.SolutionEvent;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.PushDataRequestEvent;
import edu.ucc.core.events.simulationevents.PushFileRequestEvent;
import edu.ucc.core.events.simulationevents.PushReason;
import edu.ucc.core.events.simulationevents.PushRequestEvent;
import edu.ucc.entities.Content;
import edu.ucc.entities.Data;
import edu.ucc.entities.File;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;

import static edu.ucc.utils.Logging.printInfo;
import static edu.ucc.utils.TransmissionConstants.PROCESSING_DELAY;

public class PushAction extends ExternalAction {

    public PushAction() {
        super("PushAction");
    }

    @Override
    public void execute(App app, Host localHost, SolutionEvent solutionEvent, BasePolicy policy) throws SolutionException {
        final Content content = solutionEvent.getReferredContent();
        // final Host localHost = solutionEvent.getDetectingHost();

        PushPolicy pushPolicy = (PushPolicy) policy;
        validate(pushPolicy, localHost);

        if (localHost instanceof EdgeServer) {
            EdgeServer edgeServer = (EdgeServer) localHost;

            if (content instanceof File) {
                if (!localHost.getFileSystem().contentExists(content)) {
                    final String message = String.format("Content %s does not exist in %s %s",
                            content.getDescription(), edgeServer.getTypeAsString(), edgeServer.getId());
                    ContentNotFoundException contentNotFoundException = new ContentNotFoundException(message, content, edgeServer);
                    throw new SolutionException(this, message, contentNotFoundException);
                }
            }
            // This would fail because:
            // Exception in thread "main" java.lang.RuntimeException: The cloud only processes incoming chunks regarding Data workload.requests
            // Which is ok!
            // if (pushPolicy.isPushToParent()) {
            //     PushRequestEvent pushRequestEvent = createPushRequest(popularityEvent, edgeServer.getParentUnit());
            //     System.out.println(String.format("Pushing content %s to cloud", content.getDescription()));
            //     Simulation.getInstance().postEvent(pushRequestEvent);
            // }

            if (pushPolicy.isPushToNeighbors()) {
                for (Host siblingES : edgeServer.getSiblingUnits()) {
                    if (app.isDeployedAtEdgeServer(siblingES.getId())) {
                        printInfo(String.format("PushAction: pushing content %s from ES %s to ES %s",
                                content.getDescription(),
                                edgeServer.getId(),
                                siblingES.getId()));

                        final PushRequestEvent pushRequestEvent = createPushRequest(solutionEvent, localHost, siblingES);
                        Simulation.getInstance().postEvent(pushRequestEvent);
                    } else {
                        System.out.println(String.format("App is not deployed at ES %s so I won't push it there.", siblingES.getId()));
                    }
                }
            }
        } else {
            CloudUnit cloud = (CloudUnit) localHost;
            // printWarning("PushAction: Cloud to ESs pushing not supported at the moment");
            for (Host childES : cloud.getChildrenUnits()) {
                if (app.isDeployedAtEdgeServer(childES.getId())) {
                    final PushRequestEvent pushRequestEvent = createPushRequest(solutionEvent, localHost, childES);
                    Simulation.getInstance().postEvent(pushRequestEvent);
                }
            }
        }
    }

    private PushRequestEvent createPushRequest(SolutionEvent solutionEvent, Host localHost, Host pushDestinationHost) {
        final double eventPostingTimestamp = solutionEvent.getTimestamp() + PROCESSING_DELAY;
        Content content = solutionEvent.getReferredContent();

        if (content instanceof File) {
            return new PushFileRequestEvent(
                    solutionEvent.getAppId(),
                    null,
                    localHost,
                    eventPostingTimestamp,
                    (File) content,
                    localHost,
                    pushDestinationHost,
                    PushReason.POPULARITY
            );
        } else {
            return new PushDataRequestEvent(
                    solutionEvent.getAppId(),
                    null,
                    localHost,
                    eventPostingTimestamp,
                    (Data) content,
                    localHost,
                    pushDestinationHost, PushReason.POPULARITY
            );
        }
    }

    private void validate(PushPolicy pushPolicy, Host localHost) {
        if (pushPolicy == null) throw new RuntimeException("No pushing policy specified for PushAction");

        if (localHost instanceof EdgeServer) {
            if (pushPolicy.isPushToChildren()) throw new RuntimeException("Can't push from an ES to BSs");
        }
        // Remember that a policy is specified from the perspective of an edge server. At the cloud, nothing else is
        // investigated, notifications must flow to edge servers always. This part is reached only if the app was not
        // running at the edge server.
        // if (localHost instanceof CloudUnit) {
        // if (pushPolicy.isPushToNeighbors()) throw new RuntimeException("Cloud doesn't have neighbors");
        // if (pushPolicy.isPushToParent()) throw new RuntimeException("Cloud doesn't have parent");
        // }
    }
}
