package core.distributed;

import core.Solution;
import core.app.App;
import core.control.AppPendingProcessingEvent;
import core.processingchain.PipelineResult;
import core.processingchain.ProcessingPipeline;
import core.processingchain.events.SolutionEvent;
import core.processingchain.preprocessors.PPOutcome;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.network.devices.Host;

import java.util.Iterator;
import java.util.List;

import static edu.ucc.utils.Logging.printWarning;

abstract class LocalSolution {
    protected final Solution solution;
    protected final Host localHost;
    protected final ProcessingPipeline pipeline;
    protected final double delayFactor = 0.001;

    public LocalSolution(Solution solution, Host localHost) {
        this.solution = solution;
        this.localHost = localHost;
        this.pipeline = new ProcessingPipeline(localHost);
    }

    public void registerApp(App app) {
        pipeline.registerApp(app);
    }

    public List<App> getApps() {
        return pipeline.getApps();
    }

    public App getApp(int appId) {
        return pipeline.obtainApp(appId);
    }

    public abstract void onContentReceptionCompleted(ContentReceivedEvent contentReceivedEvent);

    public abstract void onPullRequestServedLocally(PullRequestEvent pullRequestEvent);

    public abstract void onUserEquipmentHandover(HandoverEvent handoverEvent);

    public abstract List<SimulationEvent> createControlEvents();

    /**
     * Executes the pending events in the queue and returns the last timestamp of generated event.
     *
     * @param appId The if of the app whose pending events are to be processed.
     * @return The timestamp of the last generated event (used to create the event to notify about changes in storage).
     */
    public double executePendingEventsForApp(int appId) {
        final Simulation simulation = Simulation.getInstance();
        double processingTimestamp = simulation.getCurrentTime();
        App app = pipeline.obtainApp(appId);
        final List<ObservableTestbedEvent> pendingEvents = app.getPendingEventsAtHost(localHost);

        if (!pendingEvents.isEmpty()) {
            printWarning(String.format("Executing %s pending events for app %s in %s %s @ %s",
                    pendingEvents.size(),
                    app.getAppId(),
                    localHost.getTypeAsString(),
                    localHost.getId(),
                    processingTimestamp
            ));

            final Iterator<ObservableTestbedEvent> iterator = pendingEvents.iterator();
            while (iterator.hasNext()) {
                final ObservableTestbedEvent event = iterator.next();
                iterator.remove();
                final PipelineResult pipelineResult = pipeline.processEvent(localHost, event, processingTimestamp);

                final PPOutcome ppOutcome = pipelineResult.getPpOutcome();
                final SolutionEvent detectedEvent = pipelineResult.getDetectedEvent();
                if (ppOutcome == null && detectedEvent == null) {
                    processingTimestamp += delayFactor;
                } else {
                    if (detectedEvent != null) {
                        processingTimestamp = detectedEvent.getTimestamp() + delayFactor;
                    } else {
                        processingTimestamp = ppOutcome.getTimestamp() + delayFactor;
                    }
                }
            }

            app.deleteDataFromPendingEvents(localHost);
            double newEventTimestamp = processingTimestamp + app.getProcessingInterval();
            simulation.postEvent(new AppPendingProcessingEvent(this.solution, app, newEventTimestamp));
        }
        return processingTimestamp;
    }

    public Host getLocalHost() {
        return localHost;
    }
}