package core.context;

import core.app.App;
import core.processingchain.events.SolutionEvent;
import edu.ucc.entities.Content;
import edu.ucc.network.devices.Host;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class represents the repository for context information regarding all applications.
 * The structure is:
 * appId -> Hashmap<hostId, Contents>
 * hostId -> Hashmap<contentId, List<>>
 * contentId -> List of contents in the host for that app
 */
public class ContextRepository {
    private final Map<Integer, Map<Integer, Map<Long, List<SolutionEvent>>>> contextDataMap;


    public ContextRepository() {
        contextDataMap = new HashMap<>();
    }

    public void addContextData(App app, SolutionEvent contextData) {
        Map<Integer, Map<Long, List<SolutionEvent>>> contextPerApp = contextDataMap.computeIfAbsent(app.getAppId(), k -> new HashMap<>());
        final Host detectingHost = contextData.getDetectingHost();
        Map<Long, List<SolutionEvent>> contextDataForHost = contextPerApp.computeIfAbsent(detectingHost.getId(), k -> new HashMap<>());
        long contentId = contextData.getReferredContent().getId();
        List<SolutionEvent> contextForContent = contextDataForHost.computeIfAbsent(contentId, k -> new ArrayList<>());
        contextForContent.add(contextData);
    }

    public List<SolutionEvent> getContextDataForAppAndHostAndContent(App app, Host host, Content content) {
        Map<Integer, Map<Long, List<SolutionEvent>>> contextPerApp = contextDataMap.get(app.getAppId());
        if (contextPerApp == null) {
            return new ArrayList<>();
        }
        Map<Long, List<SolutionEvent>> contextPerHost = contextPerApp.get(host.getId());
        if (contextPerHost == null) {
            return new ArrayList<>();
        }

        // FIXME Improvement: Popularity in uplink (IoT data) is yet to be implemented.
        //  The data produced by a UE gets assigned a new ID, so for that UE we will have different records for each
        //  data upload (rather than just one). Since Files do not change,  there is no problem for them, but for
        //  IoT data we need to ensure yet another mechanism (e.g., use a separated list for them).
        //  Another possibility is to make that the data produced by an UE always have the same ID. For instance,
        //  /Cloud/ES0/BS0/UE0/Speed at 60s and /Cloud/ES0/BS0/UE0/Speed at 70s should have the same ID, and appear in
        //  the array list below
        List<SolutionEvent> contextPerContent = contextPerHost.get(content.getId());
        if (contextPerContent == null) {
            return new ArrayList<>();
        }
        return contextPerContent;
    }
}
