package core.app;

import core.context.ContextAPI;
import core.processingchain.actions.BaseAction;
import core.processingchain.actions.exceptions.AppOutOfStorageException;
import core.processingchain.actions.exceptions.HostOutOfStorageException;
import core.processingchain.actions.policies.BasePolicy;
import core.processingchain.events.SolutionEvent;
import core.processingchain.events.detectors.BaseEventDetector;
import core.processingchain.events.detectors.pool.popularity.PopularityEventDetector;
import core.processingchain.events.pool.HighPopularityEvent;
import core.processingchain.events.pool.PopularityEvent;
import core.processingchain.metadata.MetadataType;
import core.processingchain.metadata.extractor.AppMetadataExtractor;
import core.processingchain.metadata.extractor.MetadataExtractor;
import core.processingchain.metadata.extractor.RequestMetadataExtractor;
import core.processingchain.metadata.pool.BaseMetadata;
import core.processingchain.metadata.pool.PopularityEventMetadata;
import core.processingchain.preprocessors.PreProcessor;
import core.processingchain.preprocessors.pool.PPOutcomePullRequest;
import core.processingchain.preprocessors.pool.PreProcessorPullRequest;
import core.processingchain.preprocessors.pool.PreProcessorPushRequest;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.entities.Content;
import edu.ucc.entities.File;
import edu.ucc.network.devices.*;

import java.lang.reflect.Type;
import java.util.*;

import static core.SolutionConstants.IMMEDIATE_PROCESSING;
import static edu.ucc.utils.FileConstants.MEGABYTE;
import static edu.ucc.utils.Logging.printInfo;
import static edu.ucc.utils.Logging.printWarning;

@SuppressWarnings("rawtypes")
public class App {
    private final int appId;
    private final String name;
    private final String uriBase;
    private long totalSpace;
    private long availableSpace;
    private boolean opportunisticStorage;
    private boolean freeOnlyRequiredSpace;
    private int[] serversDeployed;
    private boolean deployedAtAllEdgeServers;
    private boolean storageFixed;
    private long[] storagePerServer;

    private StorageAllocationType storageAllocationType;
    private DataRelayPolicy dataRelayPolicy;


    /**
     * The current storage balance for the app. A negative value means the app has less space available (some content
     * was stored). A positive value means the app has more available space (some content was deleted).
     */
    private long storageBalance;

    /**
     * The storage threshold that must be surpassed in order to notify the cloud about changes in storage balance.
     * Whenever the balance is equals or greater than the abs of this value, the notification is created.
     */
    private final long storageThresholdForNotifications;


    private final ContextAPI contextAPI;
    private final Map<MetadataType, MetadataExtractor> metadataExtractors;
    private final Map<Type, PreProcessor> preProcessors;
    private final Map<Type, BaseEventDetector> eventDetectors;
    private final Map<Type, List<String>> externalSubscribers;
    private final Map<Type, List<BaseAction>> actions;
    private final Map<Type, BasePolicy> policies;
    private final Map<Integer, List<ObservableTestbedEvent>> pendingEventsAtEdgeServers;
    private final List<ObservableTestbedEvent> pendingEventsAtCloud;

    private double processingInterval;

    /**
     * Base constructor (for flexible storage and deployed at all edge servers)
     *
     * @param appId      the Id of the App
     * @param name       The name of the App
     * @param uriBase    A base URI
     * @param totalSpace The total space that the app will use.
     */
    public App(int appId, String name, String uriBase, long totalSpace) {
        this.appId = appId;
        this.name = name;
        this.uriBase = uriBase;
        //        this.distributedSpace = distributedSpace;
        //        this.freeDistributedSpace = distributedSpace;
        this.totalSpace = totalSpace;
        this.availableSpace = totalSpace;
        this.storageBalance = 0;
        this.opportunisticStorage = false;
        this.freeOnlyRequiredSpace = true;
        this.storageThresholdForNotifications = MEGABYTE;
        this.processingInterval = IMMEDIATE_PROCESSING;

        this.deployedAtAllEdgeServers = true;
        this.storageFixed = false;
        this.storageAllocationType = StorageAllocationType.FLEXIBLE_AND_FULLY_DISTRIBUTED;
        this.dataRelayPolicy = DataRelayPolicy.RELAY_TO_CLOUD;

        this.contextAPI = new ContextAPI(this);
        this.preProcessors = new HashMap<>();
        this.eventDetectors = new HashMap<>();
        this.externalSubscribers = new HashMap<>();
        this.actions = new HashMap<>();
        this.policies = new HashMap<>();
        this.pendingEventsAtEdgeServers = new HashMap<>();
        this.pendingEventsAtCloud = new ArrayList<>();

        this.metadataExtractors = new HashMap<>();
        metadataExtractors.put(MetadataType.APP_METADATA, new AppMetadataExtractor());
        metadataExtractors.put(MetadataType.REQUEST_METADATA, new RequestMetadataExtractor());
        metadataExtractors.put(MetadataType.INTERNAL_EVENT_METADATA, new MetadataExtractor<PopularityEvent>() {
            @Override
            public BaseMetadata extractMetadata(PopularityEvent hpEvent) {
                boolean highPopularity = hpEvent instanceof HighPopularityEvent;
                return new PopularityEventMetadata(hpEvent.getAppId(), hpEvent.getTestbedEvent(), highPopularity);
            }
        });

        preProcessors.put(PullDataRequestEvent.class, new PreProcessorPullRequest());
        preProcessors.put(PullFileRequestEvent.class, new PreProcessorPullRequest());
        preProcessors.put(PushDataRequestEvent.class, new PreProcessorPushRequest());
        preProcessors.put(PushFileRequestEvent.class, new PreProcessorPushRequest());
    }

    // Used when we need to specify the storage per edge server (deployed at all servers)
    public App(int appId, String name, String uriBase, long totalSpace, long[] storagePerServer) {
        this(appId, name, uriBase, totalSpace);
        this.deployedAtAllEdgeServers = true;
        this.storagePerServer = storagePerServer;
        this.storageFixed = true;
        this.storageAllocationType = StorageAllocationType.FIXED_AND_FULLY_DISTRIBUTED;
    }

    // Used when we need to specify the storage per edge server (deployed at some servers)
    public App(int appId, String name, String uriBase, long totalSpace, int[] serversDeployed, long[] storagePerServer) {
        this(appId, name, uriBase, totalSpace, storagePerServer);
        this.deployedAtAllEdgeServers = false;
        this.storagePerServer = storagePerServer;
        this.serversDeployed = serversDeployed;
        this.storageFixed = true;
        this.storageAllocationType = StorageAllocationType.FIXED_AND_PARTIALLY_DISTRIBUTED;
    }

    // Used when the storage per server is flexible (deployed at some servers)
    public App(int appId, String name, String uriBase, long totalSpace, int[] serversDeployed) {
        this(appId, name, uriBase, totalSpace);
        this.storageAllocationType = StorageAllocationType.FLEXIBLE_AND_PARTIALLY_DISTRIBUTED;
        this.deployedAtAllEdgeServers = false;
        this.serversDeployed = serversDeployed;
        this.storageFixed = false;
    }

    public int[] getServersDeployed() {
        return serversDeployed;
    }

    public int getAppId() {
        return appId;
    }

    public String getName() {
        return name;
    }

    public String getUriBase() {
        return uriBase;
    }

    public StorageAllocationType getStorageAllocationType() {
        return storageAllocationType;
    }

    public long getTotalSpace() {
        return totalSpace;
    }

    public Map<Type, BaseEventDetector> getEventDetectors() {
        return eventDetectors;
    }

    public void setOpportunisticStorage(boolean opportunisticStorage) {
        this.opportunisticStorage = opportunisticStorage;
    }

    public boolean isOpportunisticStorage() {
        return opportunisticStorage;
    }

    public void setFreeOnlyRequiredSpace(boolean freeOnlyRequiredSpace) {
        this.freeOnlyRequiredSpace = freeOnlyRequiredSpace;
    }

    public boolean isFreeOnlyRequiredSpace() {
        return freeOnlyRequiredSpace;
    }

    public MetadataExtractor getMetadataExtractorForType(MetadataType entityType) {
        return metadataExtractors.get(entityType);
    }

    public void addMetadataExtractor(MetadataType metadataType, MetadataExtractor metadataExtractor) {
        this.metadataExtractors.put(metadataType, metadataExtractor);
    }

    public void addPreProcessor(Type dataType, PreProcessor preProcessor) {
        this.preProcessors.put(dataType, preProcessor);
    }

    public PreProcessor getPreProcessorForSimulationEventType(Type dataType) {
        return preProcessors.get(dataType);
    }

    public Map<Type, PreProcessor> getPreProcessors() {
        return preProcessors;
    }

    public void addEventDetector(Type detectorType, BaseEventDetector baseEventDetector) {
        this.eventDetectors.put(detectorType, baseEventDetector);
    }

    public BaseEventDetector getEventDetectorForPreProcessorOutcomeType(Type outcomeType) {
        return this.eventDetectors.get(outcomeType);
    }

    public void setExternalSubscriberPerEventType(Type eventType, List<String> subscribersURLs) {
        this.externalSubscribers.put(eventType, subscribersURLs);
    }

    public List<String> getExternalSubscribersPerEventType(Type eventType) {
        return externalSubscribers.get(eventType);
    }

    public void addActionForEventType(Type eventType, BaseAction action) {
        List<BaseAction> actionsThisEvent = this.actions.get(eventType);
        if (actionsThisEvent == null) {
            actionsThisEvent = new ArrayList<>();
        }
        actionsThisEvent.add(action);
        this.actions.put(eventType, actionsThisEvent);
    }

    public List<BaseAction> getActionsForEventForEventType(Type eventType) {
        List<BaseAction> actions = this.actions.get(eventType);
        if (actions == null) actions = new ArrayList<>();
        return actions;
    }

    public void setPolicyForActionType(Type actionType, BasePolicy policy) {
        BasePolicy currentPolicy = this.policies.get(actionType);
        if (currentPolicy != null) {
            printWarning("Warning, you are overwriting a policy");
        }

        this.policies.put(actionType, policy);
    }

    public BasePolicy getPolicyForActionType(Type actionType) {
        return this.policies.get(actionType);
    }

    public List<BasePolicy> getPolicies() {
        return new ArrayList<>(policies.values());
    }

    public ContextAPI getContextAPI() {
        return contextAPI;
    }

    public void addContextData(SolutionEvent contextData) {
        contextAPI.addContextData(contextData);
    }

    public void storeLocalPopularContent(Host host, Content content, double timestamp) throws AppOutOfStorageException, HostOutOfStorageException {
        storeContent(host, content, timestamp, StorageReason.LOCAL_POPULARITY);
    }

    public void storeOpportunisticContent(Host host, Content content, double timestamp) throws AppOutOfStorageException, HostOutOfStorageException {
        storeContent(host, content, timestamp, StorageReason.OPPORTUNISTIC);
        notifyPopularityDetector(host, content);
    }

    public void storeExternalPopularContent(Host host, Content content, double timestamp) throws AppOutOfStorageException, HostOutOfStorageException {
        storeContent(host, content, timestamp, StorageReason.EXTERNAL_POPULARITY);
        notifyPopularityDetector(host, content);
    }

    public void storeContentFromDirectAppRequest(Host host, Content content, double timestamp) throws AppOutOfStorageException, HostOutOfStorageException{
        storeContent(host, content , timestamp, StorageReason.DIRECT_APP_PUSH_REQUEST);
        notifyPopularityDetector(host, content);
    }

    private void storeContent(Host host, Content content, double timestamp, StorageReason reason) throws AppOutOfStorageException, HostOutOfStorageException {
        final FileSystem fileSystem = host.getFileSystem();
        final long contentSize = content.getSize();
        if (fileSystem.contentExists(content)) {
            printInfo(String.format("Content %s already exists in %s %s, only updating last accessed time and storage reason",
                    content.getDescription(),
                    host.getTypeAsString(),
                    host.getId()
            ));
            fileSystem.updateLastAccessedTime(content, timestamp);
            fileSystem.updateStorageReason(content, reason);
        } else {
            if (isThereSpaceForContent(contentSize)) {
                boolean success = fileSystem.storeContent(appId, content, timestamp, reason);
                if (success) {
                    // reduceFreeDistributedSpace(contentSize);
                    reduceAvailableStorage(contentSize);
                    printInfo(String.format("Content %s stored for app %s in %s %s (%s). App free space is %s bytes", content.getDescription(), appId, host.getTypeAsString(), host.getId(), reason, availableSpace));
                } else {
                    String message = String.format("Not enough storage in %s %s for content %s for app %s", host.getTypeAsString(), host.getId(), content.getDescription(), getAppId());
                    throw new HostOutOfStorageException(this, message, host, content);
                }
            } else {
                String message = String.format("Not enough storage for app %s to store content %s in %s %s", appId, content.getDescription(), host.getTypeAsString(), host.getId());
                throw new AppOutOfStorageException(this, host, content, message);
            }
        }
    }

    private boolean isThereSpaceForContent(long contentSize) {
        //return (availableSpace + storageBalance) > contentSize;
        return availableSpace >= contentSize;
        // return freeDistributedSpace >= contentSize;
    }

    public long getAvailableSpace() {
        return availableSpace;
    }

    public void setAvailableSpace(long availableSpace) {
        this.availableSpace = availableSpace;
    }

    public void setTotalSpace(long totalSpace) {
        this.totalSpace = totalSpace;
    }

    private void notifyPopularityDetector(Host host, Content content) {
        final PopularityEventDetector popularityDetector = getPopularityDetector();
        if (popularityDetector != null) {
            popularityDetector.onContentAdded(host, content);
        }
    }

    public long getStorageBalance() {
        return storageBalance;
    }

    public boolean isOverStorageBalance() {
        return Math.abs(this.storageBalance) > this.storageThresholdForNotifications;
    }

    public void clearStorage(Host host) {
        final PopularityEventDetector popularityDetector = getPopularityDetector();
        if (popularityDetector != null) {
            final List<Content> notPopularContent = popularityDetector.getNotPopularContentForHost(host);
            final Iterator<Content> iterator = notPopularContent.iterator();

            printInfo(String.format("Clearing storage in %s %s for app %s", host.getTypeAsString(), host.getId(), appId));
            while (iterator.hasNext()) {
                Content content = iterator.next();
                boolean deleted = host.getFileSystem().deleteContent(content);
                if (deleted) {
                    // increaseFreeDistributedSpace(content.getSize());
                    increaseAvailableStorage(content.getSize());
                    printInfo(String.format("Content %s deleted", content.getDescription()));
                    iterator.remove();
                } else
                    printWarning(String.format("Could not remove Content %s from FS in Host %s (content does not exists)",
                            content.getURI(), host));
            }
        }
    }

    public void makeSpaceForContent(Host host, Content contentToAdd) {
        final long spaceToFree = contentToAdd.getSize();
        long freedSpace = 0L;
        final PopularityEventDetector popularityDetector = getPopularityDetector();
        if (popularityDetector != null) {
            final List<Content> notPopularContent = popularityDetector.getNotPopularContentForHost(host);
            final Iterator<Content> iterator = notPopularContent.iterator();
            printInfo(String.format("Making space in %s %s for app %s", host.getTypeAsString(), host.getId(), appId));
            while (iterator.hasNext()) {
                Content content = iterator.next();
                boolean deleted = host.getFileSystem().deleteContent(content);
                if (deleted) {
                    freedSpace += content.getSize();
                    // increaseFreeDistributedSpace(content.getSize());
                    increaseAvailableStorage(content.getSize());
                    printInfo(String.format("Content %s deleted", content.getDescription()));
                    iterator.remove();
                } else
                    printWarning(String.format("Could not remove Content %s from FS in Host %s (content does not exists)",
                            content.getURI(), host));
                if (freedSpace >= spaceToFree) break;
            }
        }
    }

    public void deleteDataFromPendingEvents(Host host) {
        // System.out.println("To be implemented, you will need to refactor common parts of makeSpace and clear storage methods");
        final List<Content> contents = host.getFileSystem().getContentForAppAndStorageReason(appId, StorageReason.PENDING_EVENT_PROCESSING);
        for (Content content : contents) {
            boolean deleted = host.getFileSystem().deleteContent(content);
            if (deleted) {
                increaseAvailableStorage(content.getSize());
                printInfo(String.format("Content %s deleted", content.getURI()));
            } else {
                printWarning(String.format("Could not remove Content %s from FS in Host %s (content does not exists)", content.getURI(), host));
            }
        }
    }

    /**
     * Called whenever space is freed after deleting some files
     *
     * @param value The value to increase
     */
    private void increaseAvailableStorage(long value) {
        this.storageBalance += value;
        this.availableSpace += value;
    }

    /**
     * Called whenever space is used after storing files.
     *
     * @param value The value to reduce
     */
    private void reduceAvailableStorage(long value) {
        this.storageBalance -= value;
        this.availableSpace -= value;
    }

    /**
     * Called when storage balance must be set to 0 (i.e., when the cloud instructs a new storage value).
     */
    public void resetStorageBalance() {
        this.storageBalance = 0;
    }

    public void makeSpace(Host host, Content referredContent) {
        if (freeOnlyRequiredSpace) {
            makeSpaceForContent(host, referredContent);
        } else {
            clearStorage(host);
        }
    }

    public void updateContentPopularity(Host host, double timestamp) {
        final PopularityEventDetector popularityDetector = getPopularityDetector();
        if (popularityDetector != null) {
            popularityDetector.updatePopularityForContents(host, contextAPI, timestamp);
        }
    }

    private PopularityEventDetector getPopularityDetector() {
        final BaseEventDetector eventDetector = eventDetectors.get(PPOutcomePullRequest.class);
        if (eventDetector != null) return (PopularityEventDetector) eventDetector;
        return null;
    }

    public void setProcessingInterval(double processingInterval) {
        this.processingInterval = processingInterval;
    }

    public double getProcessingInterval() {
        return processingInterval;
    }

    public void addEventForLaterProcessing(ObservableTestbedEvent pendingEvent) throws AppOutOfStorageException, HostOutOfStorageException {
        final Host host = pendingEvent.getEventDestinationHost();
        if (pendingEvent instanceof ContentReceivedEvent) {
            ContentReceivedEvent contentReceivedEvent = (ContentReceivedEvent) pendingEvent;
            storeContent(host, contentReceivedEvent.getReferredContent(), pendingEvent.getTimestamp(), StorageReason.PENDING_EVENT_PROCESSING);
        }
        if (host instanceof CloudUnit) pendingEventsAtCloud.add(pendingEvent);
        else {
            List<ObservableTestbedEvent> pendingEventsAtServer = this.pendingEventsAtEdgeServers.computeIfAbsent(host.getId(), k -> new ArrayList<>());
            pendingEventsAtServer.add(pendingEvent);
        }
    }

    public List<ObservableTestbedEvent> getPendingEventsAtHost(Host host) {
        if (host instanceof CloudUnit) return pendingEventsAtCloud;
        if (host instanceof EdgeServer) {
            final List<ObservableTestbedEvent> observableTestbedEvents = pendingEventsAtEdgeServers.get(host.getId());
            return Objects.requireNonNullElseGet(observableTestbedEvents, ArrayList::new);
        } else
            throw new IllegalArgumentException("No pending events exist in hosts other than Cloud and Edge Servers");
    }

    public boolean isDeployedAtAllEdgeServers() {
        return deployedAtAllEdgeServers;
    }

    public boolean isDeployedAtEdgeServer(int edgeServerId) {
        if (isDeployedAtAllEdgeServers()) return true;
        for (int value : serversDeployed) {
            if (value == edgeServerId)
                return true;
        }
        return false;
    }

    public long getUsedSpace() {
        return totalSpace - availableSpace;
    }

    public boolean isStorageFixed() {
        return storageFixed;
    }

    public long[] getStoragePerServer() {
        return storagePerServer;
    }

    public DataRelayPolicy getDataRelayPolicy() {
        return dataRelayPolicy;
    }

    public void setDataRelayPolicy(DataRelayPolicy dataRelayPolicy) {
        this.dataRelayPolicy = dataRelayPolicy;
    }

    /**
     * Allows to add a prefetch (proactive caching) content event.
     * @param file The file to prefetch (move) from the cloud to an edge server.
     * @param ueIds The UEs that will request this content. The predicted location (edge servers) of these UEs will be used to place content.
     * @param timestamp The timestamp used to predict the UE's location.
     * @param eventTimestamp The timestamp at which the event will be posted (different to the timestamp parameter).
     */
    public void registerContentPrefetchRequest(File file, int[] ueIds, double timestamp, double eventTimestamp) {

    }
}
