package workload.requests;

import edu.ucc.core.events.simulationevents.*;
import edu.ucc.entities.Data;
import edu.ucc.entities.File;
import edu.ucc.entities.IoTData;
import edu.ucc.entities.IoTDataType;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.NetworkArchitecture;
import edu.ucc.network.devices.UserEquipment;
import sampleapps.platerecognition.CameraData;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import static edu.ucc.workload.WorkloadGenerationConstants.RANDOM_SEED;


/***
 * Class to produce a list of random request events.
 * Later on this could be turned into an abstract hierarchy to have different implementations for generating content
 * workload.requests
 */
public class RequestsGenerator {
    private final NetworkArchitecture networkArchitecture;
    private final int appId;
    private final String baseURI;

    private final int numFileRequests;
    private final int numOfFiles;
    private final long minFileSize;
    private final long maxFileSize;

    private final int numPullDataRequests;
    private final long minDataSizeInPullDataRequests;
    private final long maxDataSizeInPullDataRequests;

    private final int numPushDataRequests;
    private final long minDataSizeInPushDataRequests;
    private final long maxDataSizeInPushDataRequests;

    private final double endTime;
    private final double startTime;
    private List<File> generatedFiles;
    // private final List<RealWorldEvent> requestEvents;
    private final List<RequestEvent> requestEvents;

    private final Random randomForFiles;
    private final Random randomForUEsPullData;
    private final Random randomForUEsPushData;
    private final Random randomForTime;

    public RequestsGenerator(NetworkArchitecture networkArchitecture, int appId, String baseURI, int numFileRequests, int numOfFiles,
                             long minFileSize, long maxFileSize, int numPullDataRequests, long minDataSizeInPullDataRequests,
                             long maxDataSizeInPullDataRequests, int numPushDataRequests, long minDataSizeInPushDataRequests,
                             long maxDataSizeInPushDataRequests, double startTime, double endTime) {
        this.networkArchitecture = networkArchitecture;
        this.appId = appId;
        this.baseURI = baseURI;
        this.numFileRequests = numFileRequests;
        this.numOfFiles = numOfFiles;
        this.minFileSize = minFileSize;
        this.maxFileSize = maxFileSize;
        this.numPullDataRequests = numPullDataRequests;
        this.minDataSizeInPullDataRequests = minDataSizeInPullDataRequests;
        this.maxDataSizeInPullDataRequests = maxDataSizeInPullDataRequests;
        this.numPushDataRequests = numPushDataRequests;
        this.minDataSizeInPushDataRequests = minDataSizeInPushDataRequests;
        this.maxDataSizeInPushDataRequests = maxDataSizeInPushDataRequests;
        this.startTime = startTime;
        this.endTime = endTime;
        this.requestEvents = new ArrayList<>();

        this.randomForFiles = new Random(RANDOM_SEED);
        this.randomForUEsPullData = new Random(RANDOM_SEED + 1);
        this.randomForUEsPushData = new Random(RANDOM_SEED + 2);
        this.randomForTime = new Random(RANDOM_SEED + 3);

    }

    public void produceRandomRequestEvents() {
        producePullFileRequests();
        producePullDataRequests();
        producePushDataRequests();
    }

    private void producePullFileRequests() {
        SimpleFileGenerator filesGenerator = new SimpleFileGenerator(baseURI, numOfFiles, minFileSize, maxFileSize);
        filesGenerator.generateFiles();
        generatedFiles = filesGenerator.getGeneratedFiles();

        CloudUnit cloud = networkArchitecture.getCloudRoot();
        List<UserEquipment> userEquipments = networkArchitecture.getUserEquipments();
        for (int i = 0; i < numFileRequests; i++) {
            // Let's produce i workload.requests
            int nextUe = randomForUEsPullData.nextInt(userEquipments.size());
            int nextFile = randomForFiles.nextInt(filesGenerator.getFileList().size());
            double timestamp = calculateTimestampWithinRanges();  // randomForTime.nextInt((int) endTime);


            UserEquipment userEquipment = userEquipments.get(nextUe);
            File fileToRequest = filesGenerator.getFileList().get(nextFile);

            RequestEvent fileRequest = new PullFileRequestEvent(appId, null, userEquipment, timestamp, userEquipment, cloud, fileToRequest);
            requestEvents.add(fileRequest);
        }
    }

    private double calculateTimestampWithinRanges() {
        return startTime + randomForTime.nextInt((int) (endTime - startTime + 1));
    }

    private void producePullDataRequests() {
        List<UserEquipment> userEquipments = networkArchitecture.getUserEquipments();

        if (userEquipments.size() < 2 && numPullDataRequests > 0) {
            throw new IllegalArgumentException("At least 2 UEs are needed for data workload.requests");
        }

        Random randomForDataSize = new Random(RANDOM_SEED);
        Random randomForRequestingHost = new Random(RANDOM_SEED);
        for (int i = 0; i < numPullDataRequests; i++) {
            int requestingUEid = randomForUEsPullData.nextInt(userEquipments.size());
            int dataSourceUEid = requestingUEid;
            while (dataSourceUEid == requestingUEid) {
                dataSourceUEid = randomForUEsPullData.nextInt(userEquipments.size());
            }

            double timestamp = calculateTimestampWithinRanges();// randomForTime.nextInt((int) endTime);

            long dataSize = randomForDataSize.nextInt((int) (maxDataSizeInPullDataRequests - minDataSizeInPullDataRequests) + 1) + minDataSizeInPullDataRequests;
            UserEquipment dataSourceUE = userEquipments.get(dataSourceUEid);
            Host requestingHost = userEquipments.get(requestingUEid);

            // Here, we randomly change the requesting host to be the UE
            if (randomForRequestingHost.nextBoolean()) {
                requestingHost = networkArchitecture.getCloudRoot();
            }

            Data nextData = new IoTData(dataSize, dataSourceUE, IoTDataType.TEMPERATURE, timestamp);
            PullRequestEvent dataRequest = new PullDataRequestEvent(appId, null, requestingHost, timestamp, requestingHost, dataSourceUE, nextData);
            requestEvents.add(dataRequest);
        }
    }

    private void producePushDataRequests() {
        List<UserEquipment> userEquipments = networkArchitecture.getUserEquipments();
        if (userEquipments.size() == 0) {
            throw new IllegalArgumentException("We need at least 1UE for producing push data workload.requests");
        }

        Random randomForDataSize = new Random(RANDOM_SEED);
        // Random randomForPushDestination = new Random(RANDOM_SEED);
        for (int i = 0; i < numPushDataRequests; i++) {
            int pushingUEId = randomForUEsPushData.nextInt(userEquipments.size());
            UserEquipment pushingUE = userEquipments.get(pushingUEId);
            double timestamp = calculateTimestampWithinRanges(); // randomForTime.nextInt((int) endTime);
            long dataSize = randomForDataSize.nextInt((int) (maxDataSizeInPushDataRequests - minDataSizeInPushDataRequests) + 1) + minDataSizeInPushDataRequests;

            // FIXME There is an issue if the UE moves to another ES while pushing data. We might want to restrict that
            //  pushing UEs do not move, or to find a way to say that content should reach current ES, whatever it is.

            // This code can customize where the pushed data should arrive: the cloud or current edge server
            // Data nextData = new Data(i, dataSize, pushingUE, IoTDataType.IMAGE, timestamp);
            // Host pushDestinationHost;
            // if (randomForPushDestination.nextBoolean()) {
            //  pushDestinationHost = pushingUE.getParentUnit().getParentUnit();
            // } else {
            //  pushDestinationHost = networkArchitecture.getCloudRoot();
            //  }

            Data nextData = new CameraData(dataSize, pushingUE, timestamp, 800, 600, 128, pushingUE.getLocation());
            Host destinationEdgeServer = pushingUE.getParentUnit().getParentUnit();

            RequestEvent dataRequest = new PushDataRequestEvent(appId, null, pushingUE, timestamp, nextData, pushingUE, destinationEdgeServer, PushReason.NORMAL_TRAFFIC);
            requestEvents.add(dataRequest);
        }
    }

    public List<File> getGeneratedFiles() {
        return generatedFiles;
    }

    public List<RequestEvent> getRequestEvents() {
        return requestEvents;
    }
}


