package workload.mobility;

import edu.ucc.core.events.simulationevents.HandoverEvent;
import edu.ucc.network.devices.BaseStation;
import edu.ucc.network.devices.LocationFix;
import edu.ucc.network.devices.UserEquipment;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import static edu.ucc.utils.HierarchyUtils.getBaseStationForLocation;

public class MobilityGenerator {
    private final List<UserEquipment> userEquipments;
    private final double bsRange;
    private final boolean mobilityEnabled;
    private final List<List<BaseStation>> baseStations;
    private final MobilityTraceGenerator mobilityTraceGenerator;

    private List<HandoverEvent> mobilityEvents;
    private List<HandoverEvent> initialEnterEvents;

    public MobilityGenerator(List<UserEquipment> userEquipments, List<List<BaseStation>> baseStations,
                             MobilityParameters mobilityParameters) {
        this.bsRange = mobilityParameters.getBsRange();
        this.userEquipments = userEquipments;
        this.baseStations = baseStations;

        double mapWidth = mobilityParameters.getMapWidth();
        double mapHeight = mobilityParameters.getMapHeight();
        double timeLength = mobilityParameters.getTrajectoryTime();
        double minSpeed = mobilityParameters.getMinSpeed();
        double maxSpeed = mobilityParameters.getMaxSpeed();
        double maxPause = mobilityParameters.getMaxPause();
        this.mobilityEnabled = mobilityParameters.isMobilityEnabled();

        this.mobilityTraceGenerator = new workload.mobility.MobilityTraceGenerator(userEquipments.size(), mapWidth, mapHeight, timeLength,
                minSpeed, maxSpeed, maxPause);
    }

    /***
     * This method must obtain the real world workload.mobility events as well as the edu.ucc.network architecture.
     * The architecture must take into account that later on the bandwidth by Edge Servers will be assigned to BSs
     * and that Cloud will assign bandwidth to Edge Servers as well.
     */
    public void createMobility() {
        this.initialEnterEvents = new ArrayList<>();
        List<HandoverEvent> mobilityEvents = new ArrayList<>();

        // Create the traces of workload.mobility
        final List<MobilityTrace> mobilityTraces = this.mobilityTraceGenerator.generateMobilityTraces();
        List<BaseStation> flatBaseStations = new ArrayList<>();
        baseStations.forEach(flatBaseStations::addAll);

        // Follow the workload.mobility to detect whenever the user connects or disconnects from each BSs
        for (workload.mobility.MobilityTrace trace : mobilityTraces) {
            // The trace is a list, it is safe to access through its id, it will be the same UE as in our UE list
            final UserEquipment userEquipment = userEquipments.get(trace.getUserEquipmentId());

            // Do it for each user equipment and its locations
            Iterator<LocationFix> iterator = trace.getLocationFixes().iterator();
            LocationFix initialLocation = iterator.next();
            if (initialLocation == null) continue;

            BaseStation currentBS = getBaseStationForLocation(initialLocation, flatBaseStations, bsRange);

            // Adding initial location. This will be in a separate list as it is used to initialize architecture
            HandoverEvent event = createHandoverEvent(initialLocation, null, currentBS, userEquipment);
            this.initialEnterEvents.add(event);

            if (mobilityEnabled) {
                while (iterator.hasNext()) {
                    LocationFix location = iterator.next();
                    BaseStation nextBS = getBaseStationForLocation(location, flatBaseStations, bsRange);
                    if (nextBS != currentBS) {
                        // this represents a HO event. We need to express it as a RealWorldEvent
                        HandoverEvent handoverEvent = createHandoverEvent(location, currentBS, nextBS, userEquipment);
                        mobilityEvents.add(handoverEvent);
                        currentBS = nextBS;
                    }
                }
            }
            mobilityEvents.sort((o1, o2) -> (int) (o1.getTimestamp() - o2.getTimestamp()));
        }
        this.mobilityEvents = mobilityEvents;
    }

    private HandoverEvent createHandoverEvent(LocationFix location, BaseStation sourceBS, BaseStation targetBS,
                                              UserEquipment userEquipment) {
        double timestamp = location.getTimestamp();
        // Map<String, Object> eventMetadata = new LinkedHashMap<>();
        // eventMetadata.put(SOURCE_BS, sourceBS);
        // eventMetadata.put(TARGET_BS, targetBS);
        // eventMetadata.put(UE_LOCATION_FIX, location);
        // return new RealWorldHandoverEvent(userEquipment, timestamp, sourceBS, targetBS, location);
        return new HandoverEvent(null, userEquipment, timestamp, sourceBS, targetBS, userEquipment,
                null, null, location);
    }

    public List<HandoverEvent> getMobilityEvents() {
        return mobilityEvents;
    }

    public List<HandoverEvent> getInitialEnterEvents() {
        return initialEnterEvents;
    }
}
