package loader;

import edu.ucc.core.events.simulationevents.HandoverEvent;
import edu.ucc.core.events.simulationevents.RequestEvent;
import edu.ucc.core.transport.routing.RoutingEntry;
import edu.ucc.core.transport.routing.RoutingTable;
import edu.ucc.entities.File;
import edu.ucc.network.architecture.ArchitectureGenerator;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.network.devices.*;
import edu.ucc.workload.Workload;
import workload.mobility.MobilityGenerator;
import workload.mobility.MobilityParameters;
import workload.mobility.UserEquipmentsPerBsScoreboard;
import workload.requests.ContentRequestParameters;
import workload.requests.RequestsGenerator;

import java.util.*;

/***
 * For synthetic scenarios, this class creates the edu.ucc.network architecture, initially assigns user equipments to base
 * stations and creates the list of real world event to launch the platform.
 */
public abstract class AbstractSystemLoader {
    protected final UserEquipmentsPerBsScoreboard uesPerBsScoreboard;
    protected final ArchitectureParameters architectureParameters;
    protected final MobilityParameters mobilityParameters;
    protected final Map<Integer, ContentRequestParameters> contentParametersPerApp;
    protected NetworkArchitecture networkArchitecture;
    protected Workload workload;
    protected ArchitectureGenerator architectureGenerator;
    protected MobilityGenerator mobilityGenerator;
    protected final List<File> generatedFiles;
    private final Map<Integer, List<RequestEvent>> requestsPerApp;


    public AbstractSystemLoader(ArchitectureParameters architectureParameters, MobilityParameters mobilityParameters,
                                Map<Integer, ContentRequestParameters> contentParametersPerApp) {
        this.architectureParameters = architectureParameters;
        this.mobilityParameters = mobilityParameters;
        this.contentParametersPerApp = contentParametersPerApp;
        this.requestsPerApp = new HashMap<>();
        this.uesPerBsScoreboard = new UserEquipmentsPerBsScoreboard();
        this.generatedFiles = new ArrayList<>();
    }

    public abstract void loadSystem();

    public Workload getWorkload() {
        return workload;
    }

    protected void loadWorkloadEvents() {
        this.workload = new Workload();

        // Loading the mobility events
        final List<HandoverEvent> mobilityEvents = mobilityGenerator.getMobilityEvents();
        workload.setMobilityEvents(mobilityEvents);
        workload.setInitialMobilityEvents(mobilityGenerator.getInitialEnterEvents());

        // Loading the request events
        for (Map.Entry<Integer, List<RequestEvent>> entry : requestsPerApp.entrySet()) {
            workload.setRequestEvents(entry.getKey(), entry.getValue());
        }
        workload.setFilesUniverse(generatedFiles);
    }

    /***
     * Creates the original relationship between BSs and UEs, i.e., sets the initial children (UEs) for base stations
     * according to the generated workload.mobility.
     * This method completes the creation of the edu.ucc.network architecture. From here it
     * can be used by the simulator
     */
    protected void linkUserEquipmentsWithBaseStations() {
        final List<HandoverEvent> initialMobilityEvents = mobilityGenerator.getInitialEnterEvents();
        for (HandoverEvent handoverEvent : initialMobilityEvents) {
            final BaseStation baseStation = handoverEvent.getTargetBS();
            // recognize and manage that other user equipments could be handled by the same baseStation
            final UserEquipment userEquipment = handoverEvent.getMovingUE();
            uesPerBsScoreboard.associateUserEquipment(baseStation, userEquipment);
        }

        // Now that the associations are identified, the children (Ues) can be properly set for base stations
        final List<BaseStation> usedBaseStations = uesPerBsScoreboard.getBaseStationsWithUserEquipments();
        for (BaseStation bs : usedBaseStations) {
            final List<UserEquipment> uesWithBs = uesPerBsScoreboard.getUEsAssociatedWithBaseStation(bs);
            bs.setChildrenUnits(new ArrayList<>(uesWithBs));
        }
    }

    protected void generateRequests() {
        final Set<Map.Entry<Integer, ContentRequestParameters>> entries = contentParametersPerApp.entrySet();
        entries.forEach(entry -> {
            final Integer appId = entry.getKey();
            ContentRequestParameters contentRequestParameters = entry.getValue();
            RequestsGenerator generator = new RequestsGenerator(networkArchitecture,
                    appId,
                    contentRequestParameters.getBaseURI(),
                    contentRequestParameters.getNumFileRequests(),
                    contentRequestParameters.getNumOfFiles(),
                    contentRequestParameters.getMinFileSize(),
                    contentRequestParameters.getMaxFileSize(),
                    contentRequestParameters.getNumPullDataRequests(),
                    contentRequestParameters.getMinDataSizeInPullDataRequests(),
                    contentRequestParameters.getMaxDataSizeInPullDataRequests(),
                    contentRequestParameters.getNumPushDataRequests(),
                    contentRequestParameters.getMinDataSizeInPushDataRequests(),
                    contentRequestParameters.getMaxDataSizeInPushDataRequests(),
                    contentRequestParameters.getRequestStartTime(),
                    contentRequestParameters.getRequestEndTime());

            generator.produceRandomRequestEvents();
            requestsPerApp.put(appId, generator.getRequestEvents());
            generatedFiles.addAll(generator.getGeneratedFiles());
        });
    }

    protected void setInitialUserEquipmentsLocation() {
        final List<HandoverEvent> initialMobilityEvents = mobilityGenerator.getInitialEnterEvents();
        for (HandoverEvent handoverEvent : initialMobilityEvents) {
            final LocationFix location = handoverEvent.getLocationFix();
            // final UserEquipment userEquipment = (UserEquipment) handoverEvent.getEventOriginHost();
            final UserEquipment userEquipment = handoverEvent.getMovingUE();
            userEquipment.setLocation(location);
        }
    }

    protected void generateArchitecture() {
        this.architectureGenerator = new ArchitectureGenerator(architectureParameters);
        this.architectureGenerator.createNetworkArchitecture();
        this.networkArchitecture = architectureGenerator.getNetworkArchitecture();
    }

    protected void generateMobility() {
        final List<UserEquipment> userEquipments = architectureGenerator.getUnassignedUEs();
        List<List<BaseStation>> baseStations = architectureGenerator.getBaseStations();
        this.mobilityGenerator = new MobilityGenerator(userEquipments, baseStations, mobilityParameters);
        mobilityGenerator.createMobility();

    }

    public NetworkArchitecture getNetworkArchitecture() {
        return networkArchitecture;
    }

    protected void createRoutingTable() {
        RoutingTable routingTable = new RoutingTable();
        final CloudUnit cloud = networkArchitecture.getCloudRoot();
        for (UserEquipment userEquipment : networkArchitecture.getUserEquipments()) {
            BaseStation baseStation = (BaseStation) userEquipment.getParentUnit();
            EdgeServer edgeServer = (EdgeServer) baseStation.getParentUnit();
            RoutingEntry routingEntry = new RoutingEntry(cloud, edgeServer, baseStation, userEquipment);
            routingTable.addEntryForUE(userEquipment, routingEntry);
        }
        networkArchitecture.setRoutingTable(routingTable);
    }

}
