package integrationtests.tests;

import core.Solution;
import core.app.App;
import core.processingchain.actions.policies.pool.PushPolicy;
import core.processingchain.actions.pool.PushAction;
import core.processingchain.actions.pool.StoreAction;
import core.processingchain.events.detectors.pool.popularity.PopularityEventDetector;
import core.processingchain.events.pool.HighPopularityEvent;
import core.processingchain.preprocessors.pool.PPOutcomePullRequest;
import edu.ucc.core.Simulation;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;
import edu.ucc.utils.Logging;
import edu.ucc.utils.TimeConstants;
import integrationtests.tests.sysloaders.CustomWorkloadSystemLoader;
import integrationtests.tests.sysloaders.ICustomWorkloadGenerator;
import integrationtests.tests.sysloaders.WorkloadGeneratorFactory;
import loader.AbstractSystemLoader;
import org.junit.jupiter.api.Test;
import workload.mobility.MobilityParameters;
import workload.requests.ContentRequestParameters;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static edu.ucc.utils.FileConstants.MEGABYTE;
import static utils.ScenarioConstants.TRAJECTORY_TIME;
import static utils.Utils.createGenericArchitectureParameters;
import static utils.Utils.createGenericMobilityParameters;

public class ScenarioStorageAllocationInDeployedAppsTest {

    @Test
    void executeTestDeployAcrossAllServersWithFlexibleStorage() {
        App app = new App(1, "app1", "www.app1.com/", 10 * 10 * MEGABYTE);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);

        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadEmpty());
        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();

        // As we are deploying all the apps across all of edge servers, we need to see that used space is 0 and that available space is all for all of them.
        App cloudApp = solution.getAppAtCloud(1);
        assert cloudApp.getUsedSpace() == 0;
        for (var server : edgeServers) {
            final App appAtEdgeServer = solution.getAppAtEdgeServer(1, server.getId());
            assert appAtEdgeServer != null;
            assert appAtEdgeServer.getUsedSpace() == 0;
            assert appAtEdgeServer.getTotalSpace() == cloudApp.getTotalSpace();
        }
    }

    @Test
    void executeTestDeployAcrossAllServersWithFixedStorage() {
        final long[] storagePerServer = {6 * 10 * MEGABYTE, 4 * 10 * MEGABYTE, 2 * 10 * MEGABYTE};
        App app = new App(1, "app1", "www.app1.com/", 12 * 10 * MEGABYTE, storagePerServer);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);

        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadEmpty());
        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();

        // As we are deploying all the apps across all of edge servers, we need to see that used space is 0 and that available space is all for all of them.
        App cloudApp = solution.getAppAtCloud(1);
        assert cloudApp.getUsedSpace() == 0;
        int[] serversDeployed = solution.getNetworkArchitecture().getEdgeServers().stream().mapToInt(Host::getId).toArray();
        for (int i = 0; i < serversDeployed.length; i++) {
            int serverId = serversDeployed[i];
            final App appAtEdgeServer = solution.getAppAtEdgeServer(1, serverId);
            assert appAtEdgeServer.getUsedSpace() == 0;
            assert appAtEdgeServer.getTotalSpace() == storagePerServer[i];

        }
    }

    @Test
    void executeTestDeployAcrossSomeServersWithFlexibleStorage() {
        final int[] serversDeployed = {0, 2};
        final int totalSpace = 10 * 10 * MEGABYTE;
        App app = new App(1, "app1", "www.app1.com/", totalSpace, serversDeployed);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);

        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadEmpty());
        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();

        // As we are deploying all the apps across all of edge servers, we need to see that used space is 0 and that available space is all for all of them.
        App cloudApp = solution.getAppAtCloud(1);
        assert cloudApp.getUsedSpace() == 0;
        assert cloudApp.getTotalSpace() == totalSpace;

        for (var serverId : serversDeployed) {
            final App appAtEdgeServer = solution.getAppAtEdgeServer(1, serverId);
            assert appAtEdgeServer != null;
            assert appAtEdgeServer.getUsedSpace() == 0;
            assert appAtEdgeServer.getTotalSpace() == totalSpace;
        }

        final App appAtEdgeServer = solution.getAppAtEdgeServer(1, 1);
        assert appAtEdgeServer == null;
    }

    @Test
    void executeTestDeployAcrossSomeServersWithFixedStorage() {
        final int[] serversDeployed = {0, 2};
        final int totalSpace = 10 * 10 * MEGABYTE;
        final long[] storagePerServer = {6 * 10 * MEGABYTE, 4 * 10 * MEGABYTE};
        App app = new App(1, "app1", "www.app1.com/", totalSpace, serversDeployed, storagePerServer);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);

        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadEmpty());
        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();

        // As we are deploying all the apps across all of edge servers, we need to see that used space is 0 and that available space is all for all of them.
        App cloudApp = solution.getAppAtCloud(1);
        assert cloudApp.getUsedSpace() == 0;
        assert cloudApp.getTotalSpace() == totalSpace;

        for (int i = 0; i < serversDeployed.length; i++) {
            int serverId = serversDeployed[i];
            final App appAtEdgeServer = solution.getAppAtEdgeServer(1, serverId);
            if (serverId == 1) {
                assert appAtEdgeServer == null;
            } else {
                assert appAtEdgeServer.getUsedSpace() == 0;
                assert appAtEdgeServer.getTotalSpace() == storagePerServer[i];
            }
        }
    }

    private Solution executeScenario(App app, ICustomWorkloadGenerator workloadGenerator) {
        ArchitectureParameters architectureParameters = createGenericArchitectureParameters();
        architectureParameters.setNumEdgeServers(3);

        MobilityParameters mobilityParameters = createGenericMobilityParameters();

        final ContentRequestParameters contentRequestParametersForAppDownlink = createContentParametersForAppDownlink(mobilityParameters.getTrajectoryTime());

        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(1, contentRequestParametersForAppDownlink);

        AbstractSystemLoader systemLoader = new CustomWorkloadSystemLoader(architectureParameters, mobilityParameters, parametersPerApp, workloadGenerator);

        systemLoader.loadSystem();
        final Solution solution = new Solution(systemLoader.getNetworkArchitecture());

        solution.registerApp(app);

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO);
        final Simulation simulation = Simulation.getInstance();

        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, TRAJECTORY_TIME);
        simulation.startSimulation();
        simulation.restart();
        return solution;
    }

    private ContentRequestParameters createContentParametersForAppDownlink(double requestEndTime) {
        // This is for pull workload.requests
        final int numFileRequests = 1;
        final int numOfFiles = 2;
        final long minFileSize = 10 * MEGABYTE;
        final long maxFileSize = 10 * MEGABYTE; // 2 * MEGABYTE;

        final int numPullDataRequests = 0;

        // This is for push workload.requests
        final int numPushDataRequests = 0;
        final long minDataSizeInPushDataRequests = 0;
        final long maxDataSizeInPushDataRequests = 0;
        return new ContentRequestParameters(1,
                "www.app1.com/",
                numFileRequests,
                numOfFiles,
                minFileSize,
                maxFileSize,
                numPullDataRequests,
                0,
                0,
                numPushDataRequests,
                minDataSizeInPushDataRequests,
                maxDataSizeInPushDataRequests,
                0.0, requestEndTime);
    }
}
