package integrationtests.tests;

import core.Solution;
import core.app.App;
import edu.ucc.core.Simulation;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.utils.Logging;
import integrationtests.tests.sysloaders.CustomWorkloadSystemLoader;
import integrationtests.tests.sysloaders.ICustomWorkloadGenerator;
import integrationtests.tests.sysloaders.WorkloadGeneratorFactory;
import loader.AbstractSystemLoader;
import org.junit.jupiter.api.Test;
import workload.mobility.MobilityParameters;
import workload.requests.ContentRequestParameters;

import java.util.HashMap;
import java.util.Map;

import static edu.ucc.utils.FileConstants.GIGABYTE;
import static edu.ucc.utils.FileConstants.MEGABYTE;
import static utils.ScenarioConstants.TRAJECTORY_TIME;
import static utils.Utils.createGenericArchitectureParameters;
import static utils.Utils.createGenericMobilityParameters;

public class ScenarioExploringMobilityAwarePrefetchingTest {

    @Test
    void executeTestMobilityAwarePrefetching() {
        final long totalSpace = 10 * 10 * GIGABYTE;
        App app = new App(1, "app1", "www.app1.com/", totalSpace);

        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadForTestingMobilityAwarePrefetching());

        final App appAtCloud = solution.getAppAtCloud(1);
        final int prefetchedFileSize = MEGABYTE * 100;

        final App appAtEdgeServer = solution.getAppAtEdgeServer(1, 0);

        assert appAtCloud.getAvailableSpace() == totalSpace - prefetchedFileSize;
        assert appAtEdgeServer.getAvailableSpace() == totalSpace - prefetchedFileSize;

    }

    private Solution executeScenario(App app, ICustomWorkloadGenerator workloadGenerator) {
        ArchitectureParameters architectureParameters = createGenericArchitectureParameters();
        architectureParameters.setNumUserEquipments(3);

        MobilityParameters mobilityParameters = createGenericMobilityParameters();
        mobilityParameters.setNumUserEquipments(3);

        final ContentRequestParameters contentRequestParametersForAppDownlink = createContentParametersForAppDownlink(mobilityParameters.getTrajectoryTime());

        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(1, contentRequestParametersForAppDownlink);

        AbstractSystemLoader systemLoader = new CustomWorkloadSystemLoader(architectureParameters, mobilityParameters, parametersPerApp, workloadGenerator);

        systemLoader.loadSystem();
        final Solution solution = new Solution(systemLoader.getNetworkArchitecture());

        solution.registerApp(app);

        // Here, we inject the initial mobility events, so that the mobility can be "predicted".
        // The following are the mobility events that will be produced by out solution, so perhaps we can register
        // events that resemble these records and then deliberately produce a push event for the app, and a pull
        // request for the mobile ue.
        solution.registerMobilityChange(1, 9, 8, 902.8989250731111); // Server 0
        solution.registerMobilityChange(1, 8, 14, 10013.87186030815); // Server 1
        solution.registerMobilityChange(1, 14, 3, 14774.458200586829); // Server 0
        solution.registerMobilityChange(1, 3, 10, 18350.505507887865); // Server 0
        solution.registerMobilityChange(1, 10, 14, 22713.613494773806); // Server 1
        solution.registerMobilityChange(1, 14, 13, 24050.06250717477); // Server 0
        solution.registerMobilityChange(1, 13, 4, 29396.568745928816); // Server 0
        solution.registerMobilityChange(1, 4, 12, 31502.459641254754); // Server 0
        solution.registerMobilityChange(1, 12, 4, 34180.15667382072); // Server 0
        solution.registerMobilityChange(1, 4, 10, 37468.707984582376); // Server 0
        solution.registerMobilityChange(1, 10, 2, 40529.462056758384); // Server 0
        solution.registerMobilityChange(1, 2, 5, 43200.0); // Server 0

        // SO, here we must inject in the workload an event for a prefetch request.


        //        final List<HandoverEvent> mobilityEvents = systemLoader.getWorkload().getMobilityEvents();
        //        for (int i = 0; i < 3; i++) {
        //            int finalI = i;
        //            final List<HandoverEvent> filtered = mobilityEvents.stream().filter(m -> m.getMovingUE().getId() == finalI).collect(Collectors.toList());
        //            for (var f : filtered) {
        //                System.out.println(f);
        //            }
        //        }
        //        HO{UE=0, sourceBS=6, targetBS=5, pendingChunksInHost=0, ts=1080.9719447749858}
        //        HO{UE=0, sourceBS=5, targetBS=8, pendingChunksInHost=0, ts=5592.045922195382}
        //        HO{UE=0, sourceBS=8, targetBS=4, pendingChunksInHost=0, ts=7732.107195498409}
        //        HO{UE=0, sourceBS=4, targetBS=10, pendingChunksInHost=0, ts=12716.794175446717}
        //        HO{UE=0, sourceBS=10, targetBS=12, pendingChunksInHost=0, ts=18023.642056060773}
        //        HO{UE=0, sourceBS=12, targetBS=3, pendingChunksInHost=0, ts=28551.381489538915}
        //        HO{UE=0, sourceBS=3, targetBS=5, pendingChunksInHost=0, ts=33480.12093907322}
        //        HO{UE=0, sourceBS=5, targetBS=13, pendingChunksInHost=0, ts=39448.16026304005}
        //        HO{UE=0, sourceBS=13, targetBS=9, pendingChunksInHost=0, ts=41731.79543060159}
        //        HO{UE=1, sourceBS=9, targetBS=8, pendingChunksInHost=0, ts=902.8989250731111}
        //        HO{UE=1, sourceBS=8, targetBS=14, pendingChunksInHost=0, ts=10013.87186030815}
        //        HO{UE=1, sourceBS=14, targetBS=3, pendingChunksInHost=0, ts=14774.458200586829}
        //        HO{UE=1, sourceBS=3, targetBS=10, pendingChunksInHost=0, ts=18350.505507887865}
        //        HO{UE=1, sourceBS=10, targetBS=14, pendingChunksInHost=0, ts=22713.613494773806}
        //        HO{UE=1, sourceBS=14, targetBS=13, pendingChunksInHost=0, ts=24050.06250717477}
        //        HO{UE=1, sourceBS=13, targetBS=4, pendingChunksInHost=0, ts=29396.568745928816}
        //        HO{UE=1, sourceBS=4, targetBS=12, pendingChunksInHost=0, ts=31502.459641254754}
        //        HO{UE=1, sourceBS=12, targetBS=4, pendingChunksInHost=0, ts=34180.15667382072}
        //        HO{UE=1, sourceBS=4, targetBS=10, pendingChunksInHost=0, ts=37468.707984582376}
        //        HO{UE=1, sourceBS=10, targetBS=2, pendingChunksInHost=0, ts=40529.462056758384}
        //        HO{UE=1, sourceBS=2, targetBS=5, pendingChunksInHost=0, ts=43200.0}
        //        HO{UE=2, sourceBS=13, targetBS=12, pendingChunksInHost=0, ts=1939.5054352823345}
        //        HO{UE=2, sourceBS=12, targetBS=10, pendingChunksInHost=0, ts=4376.417067144016}
        //        HO{UE=2, sourceBS=10, targetBS=2, pendingChunksInHost=0, ts=7680.10743161656}
        //        HO{UE=2, sourceBS=2, targetBS=6, pendingChunksInHost=0, ts=9996.672476060372}
        //        HO{UE=2, sourceBS=6, targetBS=2, pendingChunksInHost=0, ts=11942.558351971298}
        //        HO{UE=2, sourceBS=2, targetBS=13, pendingChunksInHost=0, ts=20421.996694720878}
        //        HO{UE=2, sourceBS=13, targetBS=4, pendingChunksInHost=0, ts=24875.27222097206}
        //        HO{UE=2, sourceBS=4, targetBS=9, pendingChunksInHost=0, ts=27232.943730049723}
        //        HO{UE=2, sourceBS=9, targetBS=14, pendingChunksInHost=0, ts=31808.449870834396}
        //        HO{UE=2, sourceBS=14, targetBS=2, pendingChunksInHost=0, ts=37550.62248606999}
        //        HO{UE=2, sourceBS=2, targetBS=14, pendingChunksInHost=0, ts=41763.281805768784}

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO | Logging.PREFETCH_FILE_REQUEST | Logging.DEBUG);
        final Simulation simulation = Simulation.getInstance();

        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, TRAJECTORY_TIME);
        simulation.startSimulation();
        simulation.restart();
        return solution;
    }

    private ContentRequestParameters createContentParametersForAppDownlink(double requestEndTime) {
        // This is for pull workload.requests
        final int numFileRequests = 1;
        final int numOfFiles = 2;
        final long minFileSize = 10 * MEGABYTE;
        final long maxFileSize = 10 * MEGABYTE; // 2 * MEGABYTE;

        final int numPullDataRequests = 0;

        // This is for push workload.requests
        final int numPushDataRequests = 0;
        final long minDataSizeInPushDataRequests = 0;
        final long maxDataSizeInPushDataRequests = 0;
        return new ContentRequestParameters(1,
                "www.app1.com/",
                numFileRequests,
                numOfFiles,
                minFileSize,
                maxFileSize,
                numPullDataRequests,
                0,
                0,
                numPushDataRequests,
                minDataSizeInPushDataRequests,
                maxDataSizeInPushDataRequests, 0.0, requestEndTime);
    }
}
