package experiments;

import core.Solution;
import core.app.App;
import core.processingchain.actions.policies.pool.PushPolicy;
import core.processingchain.actions.pool.PushAction;
import core.processingchain.actions.pool.StoreAction;
import core.processingchain.events.detectors.pool.popularity.PopularityEventDetector;
import core.processingchain.events.pool.HighPopularityEvent;
import core.processingchain.preprocessors.pool.PPOutcomePullRequest;
import edu.ucc.core.Simulation;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.statisticscollection.StatisticsCalculator;
import edu.ucc.utils.FileConstants;
import edu.ucc.utils.Logging;
import edu.ucc.utils.TimeConstants;
import loader.AbstractSystemLoader;
import loader.SystemLoader;
import workload.mobility.MobilityParameters;
import workload.requests.ContentRequestParameters;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import static edu.ucc.utils.FileConstants.MEGABYTE;
import static edu.ucc.utils.TimeConstants.MINUTE;
import static edu.ucc.utils.TransmissionConstants.KILOMETER;
import static utils.ScenarioConstants.*;
import static utils.Utils.createGenericMobilityParameters;

public class EdgeCDNLauncher {
    int appId = 1;
    String appName = "app1";
    String baseURI = "www.app1.com/";

    private App createAppCDNWithCaching() {
        App appCDN = new App(1, "app1", "www.app1.com/", 512 * FileConstants.GIGABYTE);
        appCDN.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        appCDN.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        appCDN.addActionForEventType(HighPopularityEvent.class, new PushAction());
        appCDN.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        appCDN.setOpportunisticStorage(true);

        return appCDN;
    }

    private App createAppCDNNoCaching() {
        return new App(1, "app1", "www.app1.com/", 512 * FileConstants.GIGABYTE);
    }

    private ArchitectureParameters createNetworkArchitectureParameters() {
        return new ArchitectureParameters(
                BS_RANGE,
                KILOMETER * 100, KILOMETER * 100,
                TOTAL_UPLINK_BW_FOR_CLOUD_CHILDREN, TOTAL_DOWNLINK_BW_FOR_CLOUD_CHILDREN,
                TOTAL_UPLINK_BW_FOR_ES_CHILDREN, TOTAL_DOWNLINK_BW_FOR_ES_CHILDREN, TOTAL_BW_FOR_ES_SIBLINGS,
                TOTAL_UPLINK_BW_FOR_BS_CHILDREN, TOTAL_DOWNLINK_BW_FOR_BS_CHILDREN,
                NUM_EDGE_SERVERS, STORAGE_PER_EDGE_SERVER,
                600, MTU);
    }

    private ContentRequestParameters createContentParametersForAppCDN(double requestEndTime) {
        // This is for pull workload.requests
        final int numFileRequests = 10000;
        final int numOfFiles = 20000;
        final long maxFileSize = 25 * MEGABYTE; // 2 * MEGABYTE;

        final int numPullDataRequests = 0;
        final long minDataSizeInPullDataRequests = 0;
        final long maxDataSizeInPullDataRequests = 0;

        // This is for push workload.requests
        final int numPushDataRequests = 0;
        final long minDataSizeInPushDataRequests = 0;
        final long maxDataSizeInPushDataRequests = 0;
        return new ContentRequestParameters(appId, baseURI, numFileRequests, numOfFiles, MEGABYTE,
                maxFileSize, numPullDataRequests, minDataSizeInPullDataRequests,
                maxDataSizeInPullDataRequests, numPushDataRequests, minDataSizeInPushDataRequests, maxDataSizeInPushDataRequests,
                0.0, requestEndTime);
    }

    private void executeCachingScenario(String outputDirPath) {
        // 1. Define edu.ucc.network's architecture
        ArchitectureParameters architectureParameters = createNetworkArchitectureParameters();
        MobilityParameters mobilityParameters = createGenericMobilityParameters();
        mobilityParameters.setMobilityEnabled(true);  // explicitly enable mobility (enabled by default)

        // 2. Define workload per individual app
        final ContentRequestParameters contentRequestParameters = createContentParametersForAppCDN(mobilityParameters.getTrajectoryTime());

        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(1, contentRequestParameters);

        // 3. Create apps, specifying/selecting the preprocessors, event detectors and policies.
        App edgeCDNApp = createAppCDNWithCaching();

        // 4. Create the solution, linking it with the testbed
        AbstractSystemLoader systemLoader = new SystemLoader(architectureParameters, mobilityParameters, parametersPerApp);
        systemLoader.loadSystem();
        final Solution solution = new Solution(systemLoader.getNetworkArchitecture());

        // 5. Register apps with the solution
        solution.registerApp(edgeCDNApp);

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO | Logging.PUSH_DATA_REQUEST);
        final Simulation simulation = Simulation.getInstance();

        // 6. Load testbed's queue with the events of workload (workload.requests + workload.mobility)
        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, TRAJECTORY_TIME + MINUTE);

        // 7. Launch simulation
        simulation.startSimulation();

        // 8. Analyse results by the statistics collector / calculator
        StatisticsCalculator statisticsCalculator = new StatisticsCalculator(simulation.getStatisticsCollector(), simulation.getWorkload(), simulation.getRunningTime());
        final int numRequests = systemLoader.getWorkload().getRequestEvents().size();
        final int numOfFiles = contentRequestParameters.getNumOfFiles();
        final int numOfUploads = contentRequestParameters.getNumPushDataRequests();
        statisticsCalculator.writeStatistics(outputDirPath, architectureParameters.getNumUserEquipments(), numRequests, numOfFiles, numOfUploads);
        simulation.restart();
    }

    private void executeNoCachingScenario(String outputDirPath) {
        // 1. Define edu.ucc.network's architecture
        ArchitectureParameters architectureParameters = createNetworkArchitectureParameters();
        MobilityParameters mobilityParameters = createGenericMobilityParameters();
        mobilityParameters.setMobilityEnabled(true);  // explicitly enable mobility (enabled by default)

        // 2. Define workload per individual app
        final ContentRequestParameters contentRequestParameters = createContentParametersForAppCDN(mobilityParameters.getTrajectoryTime());

        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(1, contentRequestParameters);

        // 3. Create apps, specifying/selecting the preprocessors, event detectors and policies.
        App iotApp = createAppCDNNoCaching();

        // 4. Create the solution, linking it with the testbed
        AbstractSystemLoader systemLoader = new SystemLoader(architectureParameters, mobilityParameters, parametersPerApp);
        systemLoader.loadSystem();
        final Solution solution = new Solution(systemLoader.getNetworkArchitecture());

        // 5. Register apps with the solution
        solution.registerApp(iotApp);

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO | Logging.PUSH_DATA_REQUEST);
        final Simulation simulation = Simulation.getInstance();

        // 6. Load testbed's queue with the events of workload (workload.requests + workload.mobility)
        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, TRAJECTORY_TIME + MINUTE);

        // 7. Launch simulation
        simulation.startSimulation();

        // 8. Analyse results by the statistics collector / calculator
        StatisticsCalculator statisticsCalculator = new StatisticsCalculator(simulation.getStatisticsCollector(), simulation.getWorkload(), simulation.getRunningTime());
        final int numRequests = systemLoader.getWorkload().getRequestEvents().size();
        final int numOfFiles = contentRequestParameters.getNumOfFiles();
        final int numOfUploads = contentRequestParameters.getNumPushDataRequests();
        statisticsCalculator.writeStatistics(outputDirPath, architectureParameters.getNumUserEquipments(), numRequests, numOfFiles, numOfUploads);
        simulation.restart();
    }

    public static void main(String[] args) {
        //        String outputDirPath = System.getProperty("user.home") + File.separator + "Desktop" + File.separator + "edge-cdn" + File.separator;
        //        new EdgeCDNLauncher().executeCachingScenario(outputDirPath);

        String outputDirPath = System.getProperty("user.home") + File.separator + "Desktop" + File.separator + "edge-cdn-no-cache" + File.separator;
        new EdgeCDNLauncher().executeNoCachingScenario(outputDirPath);

    }
}
