package experiments;

import core.Solution;
import core.app.App;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.PrefetchRequestEvent;
import edu.ucc.core.events.simulationevents.PullFileRequestEvent;
import edu.ucc.core.events.simulationevents.RequestEvent;
import edu.ucc.entities.File;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.UserEquipment;
import edu.ucc.statisticscollection.StatisticsCalculator;
import edu.ucc.utils.FileConstants;
import edu.ucc.utils.Logging;
import integrationtests.tests.sysloaders.CustomWorkloadSystemLoader;
import integrationtests.tests.sysloaders.ICustomWorkloadGenerator;
import loader.AbstractSystemLoader;
import utils.Utils;
import workload.mobility.MobilityParameters;
import workload.requests.ContentRequestParameters;

import java.util.*;

import static edu.ucc.utils.FileConstants.MEGABYTE;
import static edu.ucc.utils.TimeConstants.DAY;
import static edu.ucc.utils.TimeConstants.MINUTE;
import static edu.ucc.utils.TransmissionConstants.KILOMETER;
import static edu.ucc.workload.WorkloadGenerationConstants.RANDOM_SEED;
import static utils.ScenarioConstants.*;

public class ContentPreAllocationLauncher {

    private final int appId;
    private final String appName;
    private final String appURI;
    private final double experimentLength;
    private final double delayForPrefetchEvents;
    private Random randomForTime;
    private final double requestStartTime;
    private final int numOfUserEquipments;
    private final int numOfFiles;
    private final int numOfEdgeServers;
    private final int uesCutOut;
    private final int mapWidth;
    private final int mapHeight;

    public ContentPreAllocationLauncher() {
        this.appId = 1;
        this.appName = "app1";
        this.appURI = "www.app1.com/";
        this.experimentLength = DAY * 2; // 3600 * 12;
        this.requestStartTime = DAY; // 3600 * 6;
        this.delayForPrefetchEvents = 3600;

        this.numOfUserEquipments = 600;
        this.uesCutOut = 400;
        this.numOfFiles = 400; //numOfUserEquipments - (numOfUserEquipments / 2); // numOfUserEquipments; //
        this.numOfEdgeServers = 5;

        mapWidth = KILOMETER * 100;
        mapHeight = KILOMETER * 100;

    }

    private App createContentPreAllocationApp() {
        // No further aspects are configured
        // contentPreAllocationApp.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));
        //
        // contentPreAllocationApp.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        // contentPreAllocationApp.addActionForEventType(HighPopularityEvent.class, new PushAction());
        // contentPreAllocationApp.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        // contentPreAllocationApp.setOpportunisticStorage(true);

        return new App(appId, appName, appURI, FileConstants.TERABYTE);
    }

    private ArchitectureParameters createNetworkArchitectureParameters() {

        return new ArchitectureParameters(
                BS_RANGE,
                mapWidth, mapHeight,
                TOTAL_UPLINK_BW_FOR_CLOUD_CHILDREN, TOTAL_DOWNLINK_BW_FOR_CLOUD_CHILDREN,
                TOTAL_UPLINK_BW_FOR_ES_CHILDREN, TOTAL_DOWNLINK_BW_FOR_ES_CHILDREN, TOTAL_BW_FOR_ES_SIBLINGS,
                TOTAL_UPLINK_BW_FOR_BS_CHILDREN, TOTAL_DOWNLINK_BW_FOR_BS_CHILDREN,
                numOfEdgeServers, STORAGE_PER_EDGE_SERVER,
                numOfUserEquipments, MTU);
    }

    private ContentRequestParameters createContentParametersForContentPreAllocationApp(double requestStartTime, double requestEndTime) {
        // This is for pull workload.requests
        final int numFileRequests = 0;
        final long maxFileSize = 25 * MEGABYTE; // 2 * MEGABYTE;
        final int minFileSize = 5 * MEGABYTE; // 10 * KILOBYTE;

        final int numPullDataRequests = 0;
        final long minDataSizeInPullDataRequests = 0;
        final long maxDataSizeInPullDataRequests = 0;

        // This is for push workload.requests
        final int numPushDataRequests = 0;
        final long minDataSizeInPushDataRequests = 0;
        final long maxDataSizeInPushDataRequests = 0;

        return new ContentRequestParameters(appId, appURI, numFileRequests, numOfFiles, minFileSize,
                maxFileSize, numPullDataRequests, minDataSizeInPullDataRequests,
                maxDataSizeInPullDataRequests, numPushDataRequests, minDataSizeInPushDataRequests, maxDataSizeInPushDataRequests,
                requestStartTime, requestEndTime);
    }

    static class RequestByUE {
        UserEquipment userEquipment;
        File file;
        double timestamp;

        public RequestByUE(UserEquipment userEquipment, File file, double timestamp) {
            this.userEquipment = userEquipment;
            this.file = file;
            this.timestamp = timestamp;
        }
    }

    private ICustomWorkloadGenerator defineWorkloadWithoutPreAllocation() {
        this.randomForTime = new Random(RANDOM_SEED);
        return (networkArchitecture, workload, mobilityEvents, generatedFiles) -> {
            final CloudUnit cloud = networkArchitecture.getCloudRoot();
            List<RequestEvent> events = new ArrayList<>();

            final List<UserEquipment> userEquipments = networkArchitecture.getUserEquipments();

            // Lets assume UE 0 is gonna request file 0, and so on.
            for (int i = 0; i < numOfUserEquipments; i++) {
                final UserEquipment ue = userEquipments.get(i);
                final File file = generatedFiles.get(i % numOfFiles);
                final double timestamp = Utils.getNumberBetweenInterval(requestStartTime, experimentLength, randomForTime);
                events.add(new PullFileRequestEvent(appId, null, ue, timestamp, ue, cloud, file));
            }

            workload.setRequestEvents(appId, events);
        };
    }


    /**
     * This method defines the workload for this experiment. The idea is to make first some requests from the cloud to put
     * the content at the predicted locations.
     * Then, we simply create requests from the clients to those files.
     *
     * @return An {@link ICustomWorkloadGenerator} with the described workload.
     */
    private ICustomWorkloadGenerator defineWorkloadWithPreAllocation() {
        this.randomForTime = new Random(RANDOM_SEED);
        return (networkArchitecture, workload, mobilityEvents, generatedFiles) -> {
            final CloudUnit cloud = networkArchitecture.getCloudRoot();
            final Host prefetchInitiatorHost = cloud;
            List<RequestEvent> events = new ArrayList<>();

            final List<UserEquipment> userEquipments = networkArchitecture.getUserEquipments();

            // Lets assume UE 0 is gonna request file 0, and so on. The time assigned here, we assume it will be known by the cloud
            List<RequestByUE> requestsPerUE = new ArrayList<>();
            double latestRequestFirstHalfOfDevices = requestStartTime + ((experimentLength - requestStartTime) / 2);
            // first 400 devices are going to use that time
            for (int i = 0; i < uesCutOut; i++) {
                final UserEquipment ue = userEquipments.get(i);
                final File file = generatedFiles.get(i % numOfFiles);
                // final double timestamp = requestStartTime + randomForTime.nextInt((int) (latestRequestFirstHalfOfDevices + 1));
                final double timestamp = Utils.getNumberBetweenInterval(requestStartTime + delayForPrefetchEvents,
                        latestRequestFirstHalfOfDevices + delayForPrefetchEvents, randomForTime);
                requestsPerUE.add(new RequestByUE(ue, file, timestamp));
                events.add(new PullFileRequestEvent(appId, null, ue, timestamp, ue, cloud, file));
            }

            // second part is gonna request "duplicated" files
            // the start time for them is gonna be the latestRequestFirstHalfOfDevices
            for (int i = uesCutOut; i < numOfUserEquipments; i++) {
                final UserEquipment ue = userEquipments.get(i);
                final File file = generatedFiles.get(i % numOfFiles);
                // final double timestamp = latestRequestFirstHalfOfDevices + randomForTime.nextInt((int) (experimentLength - latestRequestFirstHalfOfDevices + 1));
                final double timestamp = Utils.getNumberBetweenInterval(latestRequestFirstHalfOfDevices + 1, experimentLength - delayForPrefetchEvents,
                        randomForTime);
                events.add(new PullFileRequestEvent(appId, null, ue, timestamp, ue, cloud, file));
            }

            // Now, let's make the cloud to prefetch those files for those UEs at the timestamp - 3600; (one hour earlier)
            // The cloud will be receiving all of these requests at the "offset" time.
            // this is for only the first part of devices, to have the files sent to the edge servers only once!
            final double timestampForEventRequest = experimentLength - requestStartTime;
            for (int i = 0; i < uesCutOut; i++) {
                RequestByUE pullRequest = requestsPerUE.get(i);
                final int[] destinationUEs = {pullRequest.userEquipment.getId()};
                final double prefetchTimestamp = pullRequest.timestamp - delayForPrefetchEvents;
                events.add(new PrefetchRequestEvent(appId, null, cloud, prefetchInitiatorHost, timestampForEventRequest,
                        pullRequest.file, destinationUEs, false, prefetchTimestamp));
            }
            workload.setRequestEvents(appId, events);
        };
    }

    /**
     * This method will produce lots of mobility events.
     *
     * @return The parameters for mobility generation
     */
    private MobilityParameters createMobilityParameters() {
        return new MobilityParameters(BS_RANGE, mapWidth, mapHeight, numOfEdgeServers,
                numOfUserEquipments, experimentLength, MIN_SPEED, MAX_SPEED, MAX_PAUSE);
    }

    private void executeContentPreAllocationScenario(String outputDirPath) {
        // 1. Define edu.ucc.network's architecture
        ArchitectureParameters architectureParameters = createNetworkArchitectureParameters();
        MobilityParameters mobilityParameters = createMobilityParameters();
        mobilityParameters.setMobilityEnabled(true);  // explicitly enable mobility (enabled by default)

        // 2. Define workload per individual app
        // This will provide an offset, enough for the solution to learn the mobility for day 1 (hence the 3600 * 24).

        final ContentRequestParameters contentRequestParameters = createContentParametersForContentPreAllocationApp(requestStartTime, experimentLength);

        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(appId, contentRequestParameters);

        // 3. Create apps, specifying/selecting the preprocessors, event detectors and policies.
        App contentPreAllocationApp = createContentPreAllocationApp();

        // 4. Create the solution, linking it with the testbed
        // AbstractSystemLoader systemLoader = new SystemLoader(architectureParameters, mobilityParameters, parametersPerApp);
        AbstractSystemLoader systemLoader = new CustomWorkloadSystemLoader(architectureParameters, mobilityParameters, parametersPerApp, defineWorkloadWithPreAllocation());
        systemLoader.loadSystem();
        final Solution solution = new Solution(systemLoader.getNetworkArchitecture());

        // 5. Register apps with the solution
        solution.registerApp(contentPreAllocationApp);

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.PULL_FILE_REQUEST | Logging.PREFETCH_FILE_REQUEST); // | Logging.PULL_FILE_REQUEST); // | Logging.PREFETCH_FILE_REQUEST | Logging.HANDOVER);
        final Simulation simulation = Simulation.getInstance();

        // 6. Load testbed's queue with the events of workload (workload.requests + workload.mobility)
        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, experimentLength + MINUTE);

        // 7. Launch simulation
        simulation.startSimulation();

        // 8. Analyse results by the statistics collector / calculator
        StatisticsCalculator statisticsCalculator = new StatisticsCalculator(simulation.getStatisticsCollector(), simulation.getWorkload(), simulation.getRunningTime());
        final int numRequests = systemLoader.getWorkload().getRequestEvents().size();
        final int numOfFiles = contentRequestParameters.getNumOfFiles();
        final int numOfUploads = contentRequestParameters.getNumPushDataRequests();
        statisticsCalculator.writeStatistics(outputDirPath, architectureParameters.getNumUserEquipments(), numRequests, numOfFiles, numOfUploads);
        simulation.restart();
    }

    private void executeContentPreAllocationNoCachingScenario(String outputDirPath) {
        // 1. Define edu.ucc.network's architecture
        ArchitectureParameters architectureParameters = createNetworkArchitectureParameters();
        MobilityParameters mobilityParameters = createMobilityParameters();
        mobilityParameters.setMobilityEnabled(true);  // explicitly enable mobility (enabled by default)

        // 2. Define workload per individual app
        // This will provide an offset, enough for the solution to learn the mobility for day 1 (hence the 3600 * 24).

        final ContentRequestParameters contentRequestParameters = createContentParametersForContentPreAllocationApp(requestStartTime, experimentLength);

        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(appId, contentRequestParameters);

        // 3. Create apps, specifying/selecting the preprocessors, event detectors and policies.
        App contentPreAllocationApp = createContentPreAllocationApp();

        // 4. Create the solution, linking it with the testbed
        // AbstractSystemLoader systemLoader = new SystemLoader(architectureParameters, mobilityParameters, parametersPerApp);
        AbstractSystemLoader systemLoader = new CustomWorkloadSystemLoader(architectureParameters, mobilityParameters, parametersPerApp, defineWorkloadWithoutPreAllocation());
        systemLoader.loadSystem();
        final Solution solution = new Solution(systemLoader.getNetworkArchitecture());

        // 5. Register apps with the solution
        solution.registerApp(contentPreAllocationApp);

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO | Logging.PUSH_DATA_REQUEST);
        final Simulation simulation = Simulation.getInstance();

        // 6. Load testbed's queue with the events of workload (workload.requests + workload.mobility)
        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, experimentLength + MINUTE);

        // 7. Launch simulation
        simulation.startSimulation();

        // 8. Analyse results by the statistics collector / calculator
        StatisticsCalculator statisticsCalculator = new StatisticsCalculator(simulation.getStatisticsCollector(), simulation.getWorkload(), simulation.getRunningTime());
        final int numRequests = systemLoader.getWorkload().getRequestEvents().size();
        final int numOfFiles = contentRequestParameters.getNumOfFiles();
        final int numOfUploads = contentRequestParameters.getNumPushDataRequests();
        statisticsCalculator.writeStatistics(outputDirPath, architectureParameters.getNumUserEquipments(), numRequests, numOfFiles, numOfUploads);
        simulation.restart();
    }

    public static void main(String[] args) {
        String outputDirPath = System.getProperty("user.home") + java.io.File.separator + "Desktop" + java.io.File.separator + "pre-allocation" + java.io.File.separator;
        new ContentPreAllocationLauncher().executeContentPreAllocationScenario(outputDirPath);

//        outputDirPath = System.getProperty("user.home") + java.io.File.separator + "Desktop" + java.io.File.separator + "pre-allocation-no-cache" + java.io.File.separator;
//        new ContentPreAllocationLauncher().executeContentPreAllocationNoCachingScenario(outputDirPath);
    }
}
