/* -*-  Mode: C++; c-file-style: "gnu"; indent-tabs-mode:nil; -*- */
/*
 * Copyright (c) 2011 Centre Tecnologic de Telecomunicacions de Catalunya (CTTC)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Jaume Nin <jaume.nin@cttc.cat>
 */

#include "ns3/lte-helper.h"
#include "ns3/epc-helper.h"
#include "ns3/core-module.h"
#include "ns3/network-module.h"
#include "ns3/ipv4-global-routing-helper.h"
#include "ns3/internet-module.h"
#include "ns3/mobility-module.h"
#include "ns3/lte-module.h"
#include "ns3/applications-module.h"
#include "ns3/point-to-point-helper.h"
#include "ns3/config-store.h"
//#include "ns3/gtk-config-store.h"


#include "ns3/tap-bridge-helper.h"
#include "ns3/csma-helper.h"
#include "ns3/netanim-module.h"

#include <iostream>
#include <fstream>
#include <sstream>

using namespace ns3;

/**
 * Sample simulation script for LTE+EPC. It instantiates several eNodeB,
 * attaches one UE per eNodeB starts a flow for each UE to  and from a remote host.
 * It also  starts yet another flow between each UE pair.
 * MISL - in this exmple we want to keep the remote host and add 2 UE nodes, with TAP 
 *
 * the HardLimit setting, which will cause the program to terminate if it cannot keep up with real time.
 *
 * Call this script with ./waf --run lena-simple-epc-tap-2ue --command="%s --ns3::RealtimeSimulatorImpl::SynchronizationMode=HardLimit"
 *
 * // output logged info
 * export NS_LOG=LenaSimpleTap2UE=info
 *
 * // log all the outlet levels
 * export NS_LOG=LenaSimpleTap2UE=level_all
 *
 * // reset logging
 * export NS_LOG=
 *
 * // output all logged info
 * export 'NS_LOG=*=level_all|prefix_func|prefix_time'
 *
 * // then call the script
 * ./waf --run lena-simple-epc-tap-2ue 
 *
 * // for all level output
 * ./waf --run lena-simple-epc-tap-2ue > log.out 2>&1
 */

// from 'lena-simple-tap.cc'
NS_LOG_COMPONENT_DEFINE ("LenaSimpleTap2UE");

// from lena-x2-handover-measures.cc
void
NotifyConnectionEstablishedEnb (std::string context,
                                uint64_t imsi,
                                uint16_t cellid,
                                uint16_t rnti)
{
//  std::cout << context
  std::cout << "eNB CellId " << cellid
            << " IMSI " << imsi
            << " RNTI " << rnti
            << std::endl;
}

// function to stop the simulation
void checkSimulation(double simTime)
{

    std::ifstream inputFile("lteStopFile.txt");
    std::string line;

    if ( inputFile.is_open() )
    {

        std::getline(inputFile, line);

        std::cout << "At time instance " << simTime << " the content of the file is " << line <<  "\n";


        if ( line == "1" )
        {
            std::cout << "Kill the clients at " << simTime << "\n";
            Simulator::Stop(Simulator::Now());
        }

        inputFile.close();

    }
    else
    {

        std::cout << "File not opening" <<  "\n";
    }
}

// function to stop the simulation
void writeSimulationValue()
{

    std::ofstream inputFile("lteStopFile.txt");
    std::string line;

    if ( inputFile.is_open() )
    {

        inputFile << "0";
        inputFile.close();

    }
    else
    {

        std::cout << "File not opening" <<  "\n";
        return;
    }
}


// function to get the bandwidth of the enb and ue
void checkBandwidth(Ptr<NetDevice> ueDevice, Ptr<NetDevice> enbDevice)
{

  Ptr<LteEnbNetDevice> enbLteDevice = enbDevice->GetObject<LteEnbNetDevice> ();
  Ptr<LteEnbPhy> enbPhy = enbLteDevice->GetPhy ();

  Ptr<LteUeNetDevice> ueLteDevice = ueDevice->GetObject<LteUeNetDevice> ();
  Ptr<LteUeRrc> ueRrc = ueLteDevice->GetRrc ();
  Ptr<LteUePhy> uePhy = ueLteDevice->GetPhy ();

  std::cout << "\n";
  uint16_t ueCellId = ueRrc->GetCellId ();
  std::cout << "ueCellId " << ueCellId << "\n";
  uint16_t enbCellId = enbLteDevice->GetCellId ();
  std::cout << "enbCellId " << enbCellId << "\n";
  uint8_t ueDlBandwidth = ueRrc->GetDlBandwidth ();
  std::cout << "UE-1 download - # RB = " << (uint32_t) ueDlBandwidth << ", => " << (((uint32_t) ueDlBandwidth) * .2) << "MHz\n";
  uint8_t enbDlBandwidth = enbLteDevice->GetDlBandwidth ();
  std::cout << "ENB download " << (uint32_t) enbDlBandwidth << "\n";
  uint8_t ueUlBandwidth = ueRrc->GetUlBandwidth ();
  std::cout << "UE-1 upload " << (uint32_t) ueUlBandwidth << "\n";
  uint8_t enbUlBandwidth = enbLteDevice->GetUlBandwidth ();
  std::cout << "ENB upload " << (uint32_t) enbUlBandwidth << "\n";
  uint8_t ueDlEarfcn = ueRrc->GetDlEarfcn ();
  std::cout << "ueDlEarfcn " << (uint32_t) ueDlEarfcn << "\n";
  uint8_t enbDlEarfcn = enbLteDevice->GetDlEarfcn ();
  std::cout << "enbDlEarfcn " << (uint32_t) enbDlEarfcn << "\n";
  uint8_t ueUlEarfcn = ueRrc->GetUlEarfcn ();
  std::cout << "ueUlEarfcn " << (uint32_t) ueUlEarfcn << "\n";
  uint8_t enbUlEarfcn = enbLteDevice->GetUlEarfcn ();
  std::cout << "enbUlEarfcn " << (uint32_t) enbUlEarfcn << "\n";
  uint64_t ueImsi = ueLteDevice->GetImsi ();
  std::cout << "ueImsi " << ueImsi << "\n";
  double ueTxPower = uePhy->GetTxPower();
  std::cout << "ueTxPower " << ueTxPower << "\n";
  double enbTxPower = enbPhy->GetTxPower();
  std::cout << "enbTxPower " << enbTxPower << "\n";

}

int
main (int argc, char *argv[])
{

  // *****************  LETS DEFINE SOME VARIABLES ************************
  // add some new variables from 'lena-simple-tap.cc'
  uint16_t numberOfNodes = 12;
  uint16_t numberOfStreamingClients = 12;
  uint16_t numberOfeNodeBs = 1;
  double simTime = 0.5;
  double distance[12] = {50, 100, 150, 200, 250, 300, 350, 400, 450, 500, 550, 600};
  double interPacketInterval = 1;
  double txPower = 40.3342375549;
  double noisePower = 10.0;
  uint16_t bandwidth = 6;
  bool useFading = false;
  bool mobileFading = false;
  bool vehicleFading = false;
  std::string LTEscheduler = "pf";
  std::string staticFadingModel = "static_fading_trace_EPA_30mph_20min_25rbg.fad";
  std::string pedestrianFadingModel = "fading_trace_EPA_3kmph_10m_1ms_25RB.fad";
  std::string vehicularFadingModel = "fading_trace_EVA_30kmph_20m.fad";
  //std::string vehicularFadingModel = "vehicular_fading_trace_EVA_30kmph_20min_6rbg.fad";
  uint16_t numRB = 25;
  uint16_t seedValue = 1;
  bool backgroundTraffic = false;

  // Name of file for animation output
  std::string animFile = "ns3-lte-animation.xml";

  // Command line arguments
  // add some new variables from 'lena-simple-tap.cc'
  CommandLine cmd;
  cmd.AddValue("numberOfNodes", "Number of UEs", numberOfNodes);
  cmd.AddValue("numberOfStreamingClients", "Number of UEs", numberOfStreamingClients);
  cmd.AddValue("numberOfeNodeBs", "Number of eNodeBs", numberOfeNodeBs);
  cmd.AddValue("simTime", "Total duration of the simulation [s])", simTime);
  cmd.AddValue("distance1", "Distance between eNBs [m]", distance[0]);
  cmd.AddValue("distance2", "Distance between eNBs [m]", distance[1]);
  cmd.AddValue("distance3", "Distance between eNBs [m]", distance[2]);
  cmd.AddValue("distance4", "Distance between eNBs [m]", distance[3]);
  cmd.AddValue("distance5", "Distance between eNBs [m]", distance[4]);
  cmd.AddValue("distance6", "Distance between eNBs [m]", distance[5]);
  cmd.AddValue("distance7", "Distance between eNBs [m]", distance[6]);
  cmd.AddValue("distance8", "Distance between eNBs [m]", distance[7]);
  cmd.AddValue("distance9", "Distance between eNBs [m]", distance[8]);
  cmd.AddValue("distance10", "Distance between eNBs [m]", distance[9]);
  cmd.AddValue("distance11", "Distance between eNBs [m]", distance[10]);
  cmd.AddValue("distance12", "Distance between eNBs [m]", distance[11]);
  cmd.AddValue("interPacketInterval", "Inter packet interval [ms])", interPacketInterval);
  cmd.AddValue("txPower", "", txPower);
  cmd.AddValue("bandwidth", "", bandwidth);
  cmd.AddValue("noisePower", "", noisePower);
  cmd.AddValue("useFading", "", useFading);
  cmd.AddValue("mobileFading", "", mobileFading);
  cmd.AddValue("vehicleFading", "", vehicleFading);
  cmd.AddValue("LTEscheduler", "", LTEscheduler);
  cmd.AddValue("seedValue", "", seedValue);
  cmd.AddValue("backgroundTraffic", "", backgroundTraffic);
  cmd.Parse(argc, argv);


  // *****************  LETS IMPLEMENT REAL-TIME SIMULATION ************************
  // from 'lena-simple-tap.cc'
  GlobalValue::Bind ("SimulatorImplementationType",
                     StringValue ("ns3::RealtimeSimulatorImpl"));
  GlobalValue::Bind ("ChecksumEnabled", BooleanValue (true));

  // set a run value, based on passed in parameter
  std::cout << "Seed value of " << seedValue << "\n";
  SeedManager::SetRun(seedValue);

  // *****************  LETS CREATE SOME ADDITIONAL LOG STATEMENTS **************************
  // Enable Logging
  //LogLevel logLevel = (LogLevel)(LOG_PREFIX_FUNC | LOG_PREFIX_TIME | LOG_LEVEL_ALL);
  //LogComponentEnable("Ipv4L3Protocol", logLevel);
  //LogComponentEnable("PointToPointNetDevice", logLevel);
  //LogComponentEnable("TapBridge", logLevel);
  //LogComponentEnable("LteUeNetDevice", logLevel);
  //LogComponentEnable("LteEnbNetDevice", logLevel);
  //LogComponentEnable("LteNetDevice", logLevel);
  //LogComponentEnable("LteUeRrc", logLevel);
  //LogComponentEnable("LteHelper", logLevel);
  //LogComponentEnable("EpcHelper", logLevel);
  //LogComponentEnable("EpcEnbApplication", logLevel);
  //LogComponentEnable("EpcSgwPgwApplication", logLevel);
  //LogComponentEnable("EpcMme", logLevel);
  //LogComponentEnable("LteEnbRrc", logLevel);
  //LogComponentEnable("Ipv4GlobalRouting", logLevel);
  //LogComponentEnable("PointToPointEpcHelper", logLevel);
  //LogComponentEnable("Ipv4ListRouting", logLevel);
  //LogComponentEnable("RealtimeSimulatorImpl", logLevel);

  std::cout << "**** SETUP ****\n";
  std::cout << "LTE and EPC Creation\n";
  Ptr<LteHelper> lteHelper = CreateObject<LteHelper> ();
  Ptr<PointToPointEpcHelper>  epcHelper = CreateObject<PointToPointEpcHelper> ();
  lteHelper->SetEpcHelper (epcHelper);


  // #################  SCHEDULER ###################

  // from 'lena-simple-tap.cc'
  std::cout << "ENB Scheduler\n";
  //lteHelper->SetSchedulerType("ns3::FdBetFfMacScheduler");

  // Pf by default
  //lteHelper->SetSchedulerType ("ns3::PfFfMacScheduler");
  //lteHelper->SetSchedulerAttribute ("CqiTimerThreshold", UintegerValue (3));

  if (LTEscheduler == "fdbet" ){

    std::cout << "FdBetFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::FdBetFfMacScheduler");

  }
  else if (LTEscheduler == "fdmt" ){

    std::cout << "FdMtFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::FdMtFfMacScheduler");

  }
  else if (LTEscheduler == "fdtbfq" ){

    std::cout << "FdTbfqFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::FdTbfqFfMacScheduler");

  }
  else if (LTEscheduler == "pf" ){

    std::cout << "PfFfMacScheduler scheduler is now active - default setting\n";
    lteHelper->SetSchedulerType("ns3::PfFfMacScheduler");

  }
  else if (LTEscheduler == "pss" ){

    std::cout << "PssFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::PssFfMacScheduler");

  }
  else if (LTEscheduler == "rr" ){

    std::cout << "RrFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::RrFfMacScheduler");

  }
  else if (LTEscheduler == "tdbet" ){

    std::cout << "TdBetFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::TdBetFfMacScheduler");

  }
  else if (LTEscheduler == "tdmt" ){

    std::cout << "TdMtFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::TdMtFfMacScheduler");

  }
  else if (LTEscheduler == "tdtbfq" ){

    std::cout << "TdTbfqFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::TdTbfqFfMacScheduler");

  }
  else if (LTEscheduler == "tta" ){

    std::cout << "TtaFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::TtaFfMacScheduler");

  }
  else if (LTEscheduler == "cqa" ){

    std::cout << "CqaFfMacScheduler scheduler is now active\n";
    lteHelper->SetSchedulerType("ns3::CqaFfMacScheduler");

  }

  // ################# RESOURCE BLOCK ALLOCATION ###################

  // This allows us to set the number of resource blocks per UE => * .2 = MHz
  // if the shared number of RB is greater than 100, then it does not work
  std::cout << "Bandwidth value of " << bandwidth << "\n";
  lteHelper->SetEnbDeviceAttribute ("DlBandwidth", UintegerValue (bandwidth));
  lteHelper->SetEnbDeviceAttribute ("UlBandwidth", UintegerValue (bandwidth));
  // WRONG lteHelper->SetUeDeviceAttribute ("TxPower", DoubleValue (15));
  // WRONG lteHelper->SetUeDeviceAttribute ("NoiseFigure", DoubleValue (5));


  // ################# PATHLOSS MODEL ###################

  //Pathloss Model
  // FriisPropagationLossModel by default
  lteHelper->SetAttribute ("PathlossModel", StringValue ("ns3::LogDistancePropagationLossModel"));


  // #################  FADING MODEL ###################  

  // if we having fading
  if (useFading){

    std::cout << "Using Fading Model\n";

      lteHelper->SetAttribute ("FadingModel", StringValue ("ns3::TraceFadingLossModel"));
      std::ifstream ifTraceFile;

      // select the correct fading model for mobile
      if (mobileFading){ 

           // select the vehicular fading model
           if (vehicleFading){ 
      
              ifTraceFile.open (vehicularFadingModel.c_str(), std::ifstream::in);
              if (ifTraceFile.good ())
                {

                  std::cout << "Mobile Vehicular Fading Model Loaded\n";

                  // script launched by test.py
                  lteHelper->SetFadingModelAttribute ("TraceFilename", StringValue (vehicularFadingModel));

                  // these parameters have to be set only if the trace format 
                  // differs from the standard one, that is:
                  // - 10 seconds length trace
                  // - 10,000 samples
                  // - 0.5 seconds for window size
                  // - 100 RB
                  lteHelper->SetFadingModelAttribute ("TraceLength", TimeValue (Seconds (1200.0)));
                  lteHelper->SetFadingModelAttribute ("SamplesNum", UintegerValue (1200000));
                  // lteHelper->SetFadingModelAttribute ("WindowSize", TimeValue (Seconds (0.5)));
                  lteHelper->SetFadingModelAttribute ("RbNum", UintegerValue (numRB));
                }
                else{
                    std::cout << "Mobile Vehicular Fading Model Failed to Load\n";
                }
            }
            // select the pedestrian fading model
            else {

                ifTraceFile.open (pedestrianFadingModel.c_str(), std::ifstream::in);
                if (ifTraceFile.good ())
                {

                  std::cout << "Mobile Pedestrian Fading Model Loaded\n";

                  // script launched by test.py
                  lteHelper->SetFadingModelAttribute ("TraceFilename", StringValue (pedestrianFadingModel));

                  // these parameters have to be set only if the trace format 
                  // differs from the standard one, that is:
                  // - 10 seconds length trace
                  // - 10,000 samples
                  // - 0.5 seconds for window size
                  // - 100 RB
                  lteHelper->SetFadingModelAttribute ("TraceLength", TimeValue (Seconds (600.0)));
                  lteHelper->SetFadingModelAttribute ("SamplesNum", UintegerValue (600000));
                  // lteHelper->SetFadingModelAttribute ("WindowSize", TimeValue (Seconds (0.5)));
                  lteHelper->SetFadingModelAttribute ("RbNum", UintegerValue (numRB));
                }
                else{
                    std::cout << "Mobile Pedestrian Fading Model Failed to Load\n";
                }
            }
       }
       // use the static pedestrian fading model - too static
       //else {

       // ifTraceFile.open ("static_fading_trace_EPA_0kmph_1ms.fad", std::ifstream::in);
       // if (ifTraceFile.good ())
       //   {

       //      std::cout << "Static Fading Model Loaded\n";

             // script launched by test.py
       //      lteHelper->SetFadingModelAttribute ("TraceFilename", StringValue ("static_fading_trace_EPA_0kmph_1ms.fad"));

             // these parameters have to be set only if the trace format 
             // differs from the standard one, that is:
             // - 10 seconds length trace
             // - 10,000 samples
             // - 0.5 seconds for window size
             // - 100 RB
       //      lteHelper->SetFadingModelAttribute ("TraceLength", TimeValue (Seconds (600.0)));
       //      lteHelper->SetFadingModelAttribute ("SamplesNum", UintegerValue (600000));
             // lteHelper->SetFadingModelAttribute ("WindowSize", TimeValue (Seconds (0.5)));
       //      lteHelper->SetFadingModelAttribute ("RbNum", UintegerValue (25));
       //   }
      //}

      // use the slightly moving static pedestrian fading model
      else {

        ifTraceFile.open (staticFadingModel.c_str(), std::ifstream::in);
        if (ifTraceFile.good ())
          {

             std::cout << "Static Fading Model Loaded\n";

             // script launched by test.py
             lteHelper->SetFadingModelAttribute ("TraceFilename", StringValue (staticFadingModel));

             // these parameters have to be set only if the trace format 
             // differs from the standard one, that is:
             // - 10 seconds length trace
             // - 10,000 samples
             // - 0.5 seconds for window size
             // - 100 RB
             lteHelper->SetFadingModelAttribute ("TraceLength", TimeValue (Seconds (1200.0)));
             lteHelper->SetFadingModelAttribute ("SamplesNum", UintegerValue (1200000));
             // lteHelper->SetFadingModelAttribute ("WindowSize", TimeValue (Seconds (0.5)));
             lteHelper->SetFadingModelAttribute ("RbNum", UintegerValue (numRB));
          }
          else{
              std::cout << "Static Fading Model Failed to Load\n";
          }
      }

  }


  // add some background trafic
  if (backgroundTraffic)
    {
        // add one aditional UE for background traffic
        numberOfStreamingClients=numberOfNodes+1;
    }
  else {
      // clients equal the number of nodes
      numberOfStreamingClients=numberOfNodes;
    }


  // #################  CONFIG SETUP ###################

  // existing
  std::cout << "Update config settings\n";
  ConfigStore inputConfig;
  inputConfig.ConfigureDefaults();
  // parse again so you can override default values from the command line
  cmd.Parse(argc, argv);
  Ptr<Node> pgw = epcHelper->GetPgwNode ();
  // lets add a name
  Names::Add ("PGW", pgw);

  //Config::SetDefault ("ns3::LteEnbRrc::DefaultTransmissionMode", UintegerValue (2));
  //Config::SetDefault ("ns3::RadioEnvironmentMapHelper::Bandwidth", UintegerValue (100));
//Config::SetDefault ("ns3::LteEnbPhy::TxPower", DoubleValue(-23.0));
  //Config::SetDefault ("ns3::LteUePhy::TxPower", DoubleValue(1.0));
  Config::SetDefault ("ns3::LteUePhy::TxPower", DoubleValue(15.0));

  // *****************  CREATE PROTOCOL STACKS **********************
  Ipv4StaticRoutingHelper ipv4RoutingHelper;

  Ipv4ListRoutingHelper list1;
  list1.Add (ipv4RoutingHelper, 10);

  // *****************  LETS CREATE THE INTERNET SIDE OF THE EPC **********************
  // create our containers - code found online
  std::cout << "\n";
  std::cout << "**** SERVER NODE ****\n";
  std::cout << "Create Server nodes\n";
  NodeContainer remoteNodes;
  //remoteNodes.Create (1);
  remoteNodes.Create (2);
  Ptr<Node> leftTapLeg = remoteNodes.Get (0);
  Ptr<Node> remoteHost = remoteNodes.Get (1);
  // lets add some more names
  Names::Add ("Remote.1", remoteNodes.Get (0));
  Names::Add ("Remote.2", remoteNodes.Get (1));

  // Create the Internet
  std::cout << "Create CSMA & Internet Helpers\n";
  CsmaHelper csmaRemote;
  csmaRemote.SetChannelAttribute ("DataRate", DataRateValue (DataRate ("100Gb/s")));

  NetDeviceContainer internetDevicesRemote = csmaRemote.Install (remoteNodes);
  // lets add some more names
  Names::Add ("Remote_Left_CSMA", internetDevicesRemote.Get (0));
  Names::Add ("Remote_Right_CSMA", internetDevicesRemote.Get (1));
  InternetStackHelper internetRemote;

  internetRemote.SetRoutingHelper (list1); //has an effect only on the next Install()

  internetRemote.Install (remoteNodes);
  internetRemote.Reset(); //clear the stack

  Ipv4AddressHelper ipv4Remote;
  ipv4Remote.SetBase ("11.0.0.0", "255.255.255.0");
  // burn the IP address, we use 
  ipv4Remote.NewAddress();
  Ipv4InterfaceContainer interfacesLeft = ipv4Remote.Assign (internetDevicesRemote);

  // get the translation values between the ueNodes and TAP
  Ipv4Address addr8 = interfacesLeft.GetAddress (0);
  std::cout << "Remote Left IP Address = " << addr8 << ".\n";
  Ipv4Address addr7 = interfacesLeft.GetAddress (1);
  std::cout << "Remote Right IP Address = " << addr7 << ".\n";

  TapBridgeHelper tapBridgeRemote;
  // UseLocal is used for linking a ns3 device and a local linux host
  // tapBridge.SetAttribute ("Mode", StringValue ("UseLocal"));UseBridge
  // UseBridge is used for extending a logical bridge into ns3
  tapBridgeRemote.SetAttribute ("Mode", StringValue ("UseBridge"));
  tapBridgeRemote.SetAttribute ("DeviceName", StringValue ("tapNas"));
  tapBridgeRemote.Install (leftTapLeg, internetDevicesRemote.Get (0));


  std::cout << "\n";
  std::cout << "Create P2P Helper\n";
  PointToPointHelper p2ph;
  p2ph.SetDeviceAttribute ("DataRate", DataRateValue (DataRate ("100Gb/s")));
  p2ph.SetDeviceAttribute ("Mtu", UintegerValue (1500));
  p2ph.SetChannelAttribute ("Delay", TimeValue (Seconds (0.020)));
  Ipv4AddressHelper ipv4h;
  ipv4h.SetBase ("1.0.0.0", "255.255.255.0");
  NetDeviceContainer internetDevices = p2ph.Install (pgw, remoteHost);

  Ipv4InterfaceContainer internetIpIfaces = ipv4h.Assign (internetDevices);
  // lets add some more names
  Names::Add ("Remote_P2P", internetDevices.Get (1));
  Names::Add ("Pgw_P2P", internetDevices.Get (0));

  Ipv4Address addr9 = internetIpIfaces.GetAddress (1);
  std::cout << "remote Host IP = " << addr9 << ".\n";
  Ipv4Address addr10 = internetIpIfaces.GetAddress (0);
  std::cout << "pgw IP = " << addr10 << ".\n";

/**
  // routing for the direct link
  NodeContainer interNode;
  interNode.Create (2);
  Ptr<Node> interLeg = interNode.Get (0);
  Names::Add ("interLeg.1", interNode.Get (0));
  Names::Add ("interLeg.2", interNode.Get (1));
  internetRemote.Install (interNode);
  NetDeviceContainer internetDevicesLeg = p2ph.Install (interLeg, remoteHost);

  //Ipv4AddressHelper ipv4h;
  ipv4h.SetBase ("2.0.0.0", "255.255.255.0");
  Ipv4InterfaceContainer internetIpIfacesLeg = ipv4h.Assign (internetDevicesLeg);
  // lets add some more names
  Names::Add ("Remote_P2P_Leg", internetDevicesLeg.Get (1));
  Names::Add ("Leg_P2P", internetDevicesLeg.Get (0));

  Ipv4Address addr19 = internetIpIfacesLeg.GetAddress (1);
  std::cout << "remote Host IP LEG = " << addr19 << ".";
  Ipv4Address addr20 = internetIpIfacesLeg.GetAddress (0);
  std::cout << "LEG IP = " << addr20 << ".";
*/

  // *****************  LETS CREATE THE UE AND ENB NODES **********************
  // existing - create the UE and ENB nodes
  std::cout << "\n";
  std::cout << "Create UE and ENB nodes\n";
  NodeContainer ueNodes;
  NodeContainer enbNodes;
  enbNodes.Create(numberOfeNodeBs);
  // lets add some names to our nodes to make them easier to see in the pcap output
  Names::Add ("ENB", enbNodes.Get (0));
  ueNodes.Create(numberOfStreamingClients);

  // add some names to the ueNodes
  for (uint16_t i = 0; i < numberOfStreamingClients; i++)
    {
      // string conversion using IOStreams
      std::stringstream ueNodesName;
      ueNodesName << "UE" << (i+1) << ".1";
      // lets add some more names
      Names::Add (ueNodesName.str(), ueNodes.Get (i));

    }

  // existing - Install Mobility Model
  std::cout << "Install Mobility Model\n";
  Ptr<ListPositionAllocator> positionAlloc = CreateObject<ListPositionAllocator> ();

  // The first Vector is the position of the Base Station
  positionAlloc->Add (Vector(0, 0, 0));

  // Subsequent values are allocated to the UEs
  for (uint16_t i = 0; i < numberOfNodes; i++)
    {
      // the first UE is at the base station - distance is zero index array
      positionAlloc->Add (Vector(distance[i], 0, 0));
      std::cout << "Mobility allocated for UE-" << i+1 << " at a distance of " << distance[i] << ".\n";

      // the first UE is <distance> away from the base station
      //positionAlloc->Add (Vector(distance * (i+1), 0, 0));
      //std::cout << "Mobility allocated for UE-" << i+1 << " at a distance of " << distance * (i+1) << ".\n";

      //positionAlloc->Add (Vector(0, 0, 0));
      //std::cout << "Mobility allocated for UE-" << i+1 << " at a distance of " << distance  << ".\n";

    }
  MobilityHelper mobility;
  mobility.SetMobilityModel("ns3::ConstantPositionMobilityModel");
  mobility.SetPositionAllocator(positionAlloc);
  mobility.Install(enbNodes);
  mobility.Install(ueNodes);

  // existing - Install LTE Devices to the nodes
  std::cout << "Create UE and ENB Containers\n";
  NetDeviceContainer enbLteDevs = lteHelper->InstallEnbDevice (enbNodes);
  // lets add some more names
  Names::Add ("ENB1", enbLteDevs.Get (0));
  NetDeviceContainer ueLteDevs = lteHelper->InstallUeDevice (ueNodes);

  // add some names to the ueLteDevs
  for (uint16_t i = 0; i < numberOfStreamingClients; i++)
    {
      // string conversion using IOStreams
      std::stringstream ueLteDevsName;
      ueLteDevsName << "LTE_UE" << (i+1);
      // lets add some more names
      Names::Add (ueLteDevsName.str(), ueLteDevs.Get (i));

    }

  // Install the IP stack on the UEs
  std::cout << "Install Internet Stack on UEs\n";
  InternetStackHelper internet;
  //internet.Install (ueNodes);

  //install protocol stack to the UEs
  for (uint32_t i = 0; i < ueNodes.GetN (); i++)
    {
       internet.SetRoutingHelper (list1);
       internet.Install (ueNodes.Get(i));
     }
  internet.Reset();

  Ipv4InterfaceContainer ueIpIface;
  //ueIpIface = epcHelper->AssignUeIpv4Address (ueLteDevs);
  ueIpIface = epcHelper->AssignUeIpv4Address (NetDeviceContainer (ueLteDevs)); 

  // Assign routing to UEs
  //Ipv4StaticRoutingHelper ipv4RoutingHelper; 

  // Assign IP address to UEs, and install applications
  for (uint32_t u = 0; u < ueNodes.GetN (); ++u)
    {
      Ptr<Node> ueNode = ueNodes.Get (u);
      // Set the default gateway for the UE
      Ptr<Ipv4StaticRouting> ueStaticRouting = ipv4RoutingHelper.GetStaticRouting (ueNode->GetObject<Ipv4> ());
      ueStaticRouting->SetDefaultRoute (epcHelper->GetUeDefaultGatewayAddress (), 1);
    }

  for (uint16_t i = 0; i < numberOfStreamingClients; i++)
      {
        lteHelper->Attach (ueLteDevs.Get(i), enbLteDevs.Get(0));
        // side effect: the default EPS bearer will be activated
      }

  // *****************  LETS CREATE SOME UE'S ************************
  // Let's create the first node
  std::cout << "\n";
  std::cout << "Create IP Address Helper\n";
  ipv4h.SetBase ("12.0.0.0", "255.255.255.0");
  std::cout << "View Network Metrics\n";
  Ipv4Address gateway = epcHelper->GetUeDefaultGatewayAddress ();
  std::cout << "Left UE gateway = " << gateway << ".\n";

  // create the UE nodes
  NodeContainer UE;
  UE.Create (numberOfStreamingClients);

  // creat the csma
  CsmaHelper csma;
  csma.SetChannelAttribute ("DataRate", DataRateValue (DataRate ("100Gb/s")));

  // create the tap bridge
  TapBridgeHelper tapBridge;
  tapBridge.SetAttribute ("Mode", StringValue ("UseBridge"));

/*

  std::cout << "**** UE-1 ****\n";
  NodeContainer UE;
  UE.Create (numberOfNodes);
  NodeContainer csmaUE1s (ueNodes.Get (0),UE.Get (0));

  CsmaHelper csma;
  csma.SetChannelAttribute ("DataRate", DataRateValue (DataRate ("100Gb/s")));
  NetDeviceContainer internetDevicesUE1 = csma.Install (csmaUE1s);
  internet.Install (UE.Get (0));
  // lets add some more names
  Names::Add ("UE1.2", UE.Get (0));
  Names::Add ("UE1_Left_CSMA", internetDevicesUE1.Get (0));
  Names::Add ("UE1_Right_CSMA", internetDevicesUE1.Get (1));
  Ipv4InterfaceContainer interfacesUE1 = ipv4h.Assign (internetDevicesUE1);
  std::cout << "Create Tap Bridge\n";
  TapBridgeHelper tapBridge;
  tapBridge.SetAttribute ("Mode", StringValue ("UseBridge"));
  tapBridge.SetAttribute ("DeviceName", StringValue ("tap-UE1"));
  tapBridge.Install (UE.Get (0), internetDevicesUE1.Get (1));


  std::cout << "**** UE-2 ****\n";
  NodeContainer csmaUE2s (ueNodes.Get (1),UE.Get (1));
  NetDeviceContainer internetDevicesUE2 = csma.Install (csmaUE2s);
  internet.Install (UE.Get (1));
  // lets add some more names
  Names::Add ("UE2.2", UE.Get (1));
  Names::Add ("UE2_Left_CSMA", internetDevicesUE2.Get (0));
  Names::Add ("UE2_Right_CSMA", internetDevicesUE2.Get (1));
  Ipv4InterfaceContainer interfacesUE2 = ipv4h.Assign (internetDevicesUE2);
  std::cout << "Create Tap Bridge\n";
  tapBridge.SetAttribute ("DeviceName", StringValue ("tap-UE2"));
  tapBridge.Install (UE.Get (1), internetDevicesUE2.Get (1));

  std::cout << "**** UE-3 ****\n";
  NodeContainer csmaUE3s (ueNodes.Get (2),UE.Get (2));
  NetDeviceContainer internetDevicesUE3 = csma.Install (csmaUE3s);
  internet.Install (UE.Get (2));
  // lets add some more names
  Names::Add ("UE3.2", UE.Get (2));
  Names::Add ("UE3_Left_CSMA", internetDevicesUE3.Get (0));
  Names::Add ("UE3_Right_CSMA", internetDevicesUE3.Get (1));
  Ipv4InterfaceContainer interfacesUE3 = ipv4h.Assign (internetDevicesUE3);
  std::cout << "Create Tap Bridge\n";
  tapBridge.SetAttribute ("DeviceName", StringValue ("tap-UE3"));
  tapBridge.Install (UE.Get (2), internetDevicesUE3.Get (1));

  std::cout << "**** UE-4 ****\n";
  NodeContainer csmaUE4s (ueNodes.Get (3), UE.Get (3));
  NetDeviceContainer internetDevicesUE4 = csma.Install (csmaUE4s);
  internet.Install (UE.Get (3));
  // lets add some more names
  Names::Add ("UE4.2", UE.Get (3));
  Names::Add ("UE4_Left_CSMA", internetDevicesUE4.Get (0));
  Names::Add ("UE4_Right_CSMA", internetDevicesUE4.Get (1));
  Ipv4InterfaceContainer interfacesUE4 = ipv4h.Assign (internetDevicesUE4);
  std::cout << "Create Tap Bridge\n";
  tapBridge.SetAttribute ("DeviceName", StringValue ("tap-UE4"));
  tapBridge.Install (UE.Get (3), internetDevicesUE4.Get (1));


  std::cout << "**** UE-5 ****\n";
  NodeContainer csmaUE5s (ueNodes.Get (4), UE.Get (4));
  NetDeviceContainer internetDevicesUE5 = csma.Install (csmaUE5s);
  internet.Install (UE.Get (4));
  // lets add some more names
  Names::Add ("UE5.2", UE.Get (4));
  Names::Add ("UE5_Left_CSMA", internetDevicesUE5.Get (0));
  Names::Add ("UE5_Right_CSMA", internetDevicesUE5.Get (1));
  Ipv4InterfaceContainer interfacesUE5 = ipv4h.Assign (internetDevicesUE5);
  std::cout << "Create Tap Bridge\n";
  tapBridge.SetAttribute ("DeviceName", StringValue ("tap-UE5"));
  tapBridge.Install (UE.Get (4), internetDevicesUE5.Get (1));

  std::cout << "**** UE-6 ****\n";
  NodeContainer csmaUE6s (ueNodes.Get (5), UE.Get (5));
  NetDeviceContainer internetDevicesUE6 = csma.Install (csmaUE6s);
  internet.Install (UE.Get (5));
  // lets add some more names
  Names::Add ("UE6.2", UE.Get (5));
  Names::Add ("UE6_Left_CSMA", internetDevicesUE6.Get (0));
  Names::Add ("UE6_Right_CSMA", internetDevicesUE6.Get (1));
  Ipv4InterfaceContainer interfacesUE6 = ipv4h.Assign (internetDevicesUE6);
  std::cout << "Create Tap Bridge\n";
  tapBridge.SetAttribute ("DeviceName", StringValue ("tap-UE6"));
  tapBridge.Install (UE.Get (5), internetDevicesUE6.Get (1));

*/

  // add some UEs - based on  the number of devices - this will be one less when background traffic is true
  for (uint16_t i = 0; i < numberOfNodes; i++)
    {
      // string conversion using IOStreams
      std::stringstream ueFirstName;
      std::stringstream ueLeftName;
      std::stringstream ueRightName;
      std::stringstream ueTapName;

      // create the strings
      ueFirstName << "UE" << (i+1) << ".2";
      ueLeftName << "UE" << (i+1) << "_Left_CSMA";
      ueRightName << "UE" << (i+1) << "_Right_CSMA";
      ueTapName << "tap-UE" << (i+1);

      // create this UE
      std::cout << "**** UE-"<<(i+1)<<" ****\n";
      NetDeviceContainer internetDevice = csma.Install (NodeContainer (ueNodes.Get (i), UE.Get (i)));
      internet.Install (UE.Get (i));
      // lets add some more names
      Names::Add (ueFirstName.str(), UE.Get (i));
      Names::Add (ueLeftName.str(), internetDevice.Get (0));
      Names::Add (ueRightName.str(), internetDevice.Get (1));
      Ipv4InterfaceContainer interfacesUE = ipv4h.Assign (internetDevice);
      std::cout << "Create Tap Bridge\n";
      tapBridge.SetAttribute ("DeviceName", StringValue (ueTapName.str()));
      tapBridge.Install (UE.Get (i), internetDevice.Get (1));

    }

  // this will be one less when background traffic is true
  for (uint32_t u = 0; u < numberOfNodes; ++u)
    {
          std::cout << "\n";
          // get the translation values between the ueNodes and TAP
          Ptr<Node> ueNode = ueNodes.Get (u);
          Ptr<Ipv4> ipv4 = ueNode->GetObject<Ipv4> (); 
          Ipv4Address addr1 = ipv4->GetAddress (1, 0).GetLocal ();
          std::cout << "UE-" << u+1 << " LTE IP Address = " << addr1 << ".\n";
          Ipv4Address addr2 = ipv4->GetAddress (2, 0).GetLocal ();
          std::cout << "UE-" << u+1 << " LTE-CSMA IP Address = " << addr2 << ".\n";
          Ipv4Address TapAddr = UE.Get (u)->GetObject<Ipv4> ()->GetAddress (1, 0).GetLocal ();
          std::cout << "UE-" << u+1 << " CSMA-TAP IP Address = " << TapAddr << ".\n";

          Ptr<LteUeNetDevice> ueLteDevice = ueLteDevs.Get (u)->GetObject<LteUeNetDevice> ();
          uint64_t ueImsi = ueLteDevice->GetImsi ();
          std::cout << "UE-" << u+1 << " IMSI = " << ueImsi << ".\n";
    }
    
    // add some background trafic
    if (backgroundTraffic)
    {
        // increase the number of UE
        // numberOfNodes=numberOfNodes+1;
    }

  // *****************  LETS CREATE SOME ROUTING ***********************
  std::cout << "\n";
  std::cout << "Create Routing\n";
  // define some static routes for the REMOTE NODE
  Ptr<Ipv4StaticRouting> remoteHostStaticRouting = ipv4RoutingHelper.GetStaticRouting (remoteHost->GetObject<Ipv4> ());
  // 9.0.0.x (NAS) and 12.0.0.x (Clients) are subnets to LTE and the Clients
  remoteHostStaticRouting->AddNetworkRouteTo (Ipv4Address ("9.0.0.0"), Ipv4Mask ("255.255.255.0"), Ipv4Address ("11.0.0.1"), 1);
  remoteHostStaticRouting->AddNetworkRouteTo (Ipv4Address ("12.0.0.0"), Ipv4Mask ("255.0.0.0"), 3);

  // define some static routes for the PGW
  Ptr<Ipv4StaticRouting> remoteHostStaticRoutingPgw = ipv4RoutingHelper.GetStaticRouting (pgw->GetObject<Ipv4> ());
  // 9.0.0.x (NAS) is a subnets to the NAS
  remoteHostStaticRoutingPgw->AddNetworkRouteTo (Ipv4Address ("9.0.0.0"), Ipv4Mask ("255.255.255.0"), 2);
  remoteHostStaticRoutingPgw->AddNetworkRouteTo (Ipv4Address ("11.0.0.0"), Ipv4Mask ("255.255.255.0"), 2);

/**

  // *****************  LETS SEE IF PING WORKS  *************************
  // PING DOES NOT WORK BETWEEN ALL NODES, AS SOME WORK ONLY USING BEARERS, I THINK??
  std::cout << "\n";
  std::cout << "Run PING test\n";

  std::string remotePing ("9.0.0.2"); // example.com
  Ipv4Address remoteIpPing (remotePing.c_str ());
  
  // interfacesLeft - IPv4 container IP range - 1.0.0.x
  // ueNodes - IPv4 container IP range - 7.0.0.x
  // remoteNodes - IPv4 container IP range - 9.0.0.x
  //  - IPv4 container IP range - 10.0.0.x

  // internetIpIfaces - IPv4 container IP range - 1.0.0.x
  // ueIpIface - IPv4 container IP range - 7.0.0.x
  // interfacesLeft - IPv4 container IP range - 9.0.0.x
  // interfacesLeftUE - IPv4 container IP range - 10.0.0.x

  // change these for ping values
  // Ptr<Node> sourceNode = ueNodes.Get (0);
  Ptr<Node> sourceNode = remoteNodes.Get (1);
  //Ptr<Node> sourceNode = pgw;
  // GetAddress (1, 0) - returns the IP address of interface 1
  Ipv4Address sourceAddress = sourceNode->GetObject<Ipv4> ()->GetAddress (2, 0).GetLocal ();
  // GetAddress (2, 0) - returns the IP address of interface 2
  //Ipv4Address sourceAddress = sourceNode->GetObject<Ipv4> ()->GetAddress (2, 0).GetLocal ();
  Ipv4Address remoteAddress = remoteIpPing;
  // Ipv4Address remoteAddress = interfacesLeft.GetAddress (1);

  std::cout << "Create V4Ping Appliation from " << sourceAddress << " to " << remoteAddress << " .\n";
  Ptr<V4Ping> app = CreateObject<V4Ping> ();
  //app->SetAttribute ("Remote", Ipv4AddressValue (remoteIp));
  app->SetAttribute ("Remote", Ipv4AddressValue (remoteAddress));
  app->SetAttribute ("Verbose", BooleanValue (true) );
  sourceNode->AddApplication (app);
  // ueNodes.Get(0)->AddApplication (app);
  //remoteNodes.Get(0)->AddApplication (app);
  app->SetStartTime (Seconds (1.0));
  app->SetStopTime (Seconds (9.0));
*/

  // *****************  PRODUCE SOME TRACES AND START THE SIMULATION  *************************
  std::cout << "\n";
  std::cout << "Enable Traces\n";
  //lteHelper->EnableTraces ();

  // CREATE WIRESHARK TRACES
  //p2ph.EnablePcapAll("P2P_Link");
  //csma.EnablePcapAll ("CSMA_Devices");


  // *****************  GET SOME VALUES FROM THE SIMULATION  *************************
  for (uint32_t u = 0; u < numberOfStreamingClients; ++u)
    {
      // get bandwidth values for each UE, from within the running simulation
      Simulator::Schedule ( Seconds(0.5+u), &checkBandwidth , ueLteDevs.Get (u), enbLteDevs.Get (0));

    }

  // reset the contents of our stop file
  Simulator::Schedule ( Seconds(0), &writeSimulationValue );

  // check the file contents and stop the simulator if equal to 1
  for (uint32_t u = 1; u < simTime; ++u)
    {
        // stop the simulator
        Simulator::Schedule ( Seconds(u), &checkSimulation, u );
    }


  // ***************** ANIMATION ***************** 
  // Create the animation object and configure for specified output
  //AnimationInterface anim (animFile);
  //anim.EnablePacketMetadata (); // Optional
  //anim.EnableIpv4L3ProtocolCounters (Seconds (0), Seconds (10)); // Optional

  // from lena-x2-handover-measures.cc
  // connect custom trace sinks for RRC connection establishment and handover notification
  Config::Connect ("/NodeList/*/DeviceList/*/LteUeRrc/ConnectionEstablished",
                   MakeCallback (&NotifyConnectionEstablishedEnb));


  // *****************  START THE SIMULATION  *************************
  std::cout << "Start Simulator\n";
  Simulator::Stop(Seconds(simTime));
  Simulator::Run();

  //std::cout << "Animation Trace file created:" << animFile.c_str ()<< std::endl;

  /*GtkConfigStore config;
  config.ConfigureAttributes();*/

  std::cout << "\n";
  std::cout << "Done\n";

  Simulator::Destroy();
  return 0;

}

