package edu.ucc.utils;

import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.ChunkReceivedEvent;
import edu.ucc.core.events.simulationevents.PullRequestEvent;
import edu.ucc.core.events.simulationevents.TestbedEvent;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.core.transport.routing.RoutingEntry;
import edu.ucc.core.transport.routing.RoutingTable;
import edu.ucc.entities.Content;
import edu.ucc.network.devices.*;

import java.util.*;

import static edu.ucc.utils.TransmissionConstants.PROCESSING_DELAY;

public class TransmissionUtils {
    public static final int UPWARDS_DIRECTION = -1;
    public static final int DOWNWARDS_DIRECTION = 1;
    public static final int SAME_LEVEL_DIRECTION = 0;

    public static TestbedEvent createEventForChunk(Host originHost, Host destinationHost, double transmissionDelay, ContentChunk chunk) {
        double receptionTime = Simulation.getInstance().getCurrentTime() + transmissionDelay;

        return new ChunkReceivedEvent(originHost, destinationHost, receptionTime, chunk);
    }

    public static int getNumberOfChunksForContent(Content referredContent, double chunkSize) {
        return (int) Math.ceil(referredContent.getSize() * 8 / chunkSize);
    }

    @SafeVarargs
    public static List<ContentChunk> fuseListsOfChunks(List<ContentChunk>... lists) {
        // Here we got to sort them also by content request!!
        // We separate chunks by request using a hash map
        Map<Integer, List<ContentChunk>> chunksPerRequest = new HashMap<>();
        for (List<ContentChunk> list : lists) {
            for (ContentChunk contentChunk : list) {
                int requestId = contentChunk.getRequestEvent().getRequestId();
                List<ContentChunk> chunksThisRequest = chunksPerRequest.get(requestId);
                if (chunksThisRequest == null) {
                    chunksThisRequest = new ArrayList<>();
                }
                chunksThisRequest.add(contentChunk);
                chunksPerRequest.put(requestId, chunksThisRequest);
            }
        }

        // Now we got them all in the map (and uniquely), so we can iterate over the sorted keys,
        // sort the chunks in each request and put them together in the output list.
        final ArrayList<Integer> sortedRequestIds = new ArrayList<>(chunksPerRequest.keySet());
        sortedRequestIds.sort(Comparator.comparingInt(o -> o));

        final List<ContentChunk> mergedChunks = new ArrayList<>();
        for (Integer requestId : sortedRequestIds) {
            List<ContentChunk> chunksThisRequest = chunksPerRequest.get(requestId);
            chunksThisRequest.sort(Comparator.comparingInt(ContentChunk::getChunkNumber));
            mergedChunks.addAll(chunksThisRequest);
        }
        return mergedChunks;
    }

    public static Host findNextHopForChunkDownlink(UserEquipment userEquipment, Host currentHost) {
        final RoutingTable routingTable = Simulation.getInstance().getNetworkArchitecture().getRoutingTable();
        RoutingEntry routingEntryForUE = routingTable.getEntryForUe(userEquipment);
        if (currentHost instanceof CloudUnit) {
            return routingEntryForUE.getEdgeServerPartInRoute();
        } else if (currentHost instanceof EdgeServer) {
            return routingEntryForUE.getBaseStationPartInRoute();
        } else { // if(this instanceof BaseStation){
            return routingEntryForUE.getUserEquipmentPartInRoute();
        }
    }

    public static void propagatePullRequestToParent(Host currentHost, PullRequestEvent requestEvent) {
        final Host nextHop = currentHost.getParentUnit();
        double txTime = currentHost.calculateTxTimeToParent(currentHost.getMtu());
        txTime += PROCESSING_DELAY;
        TestbedEvent newEvent = requestEvent.createEventForNextHop(nextHop, txTime);
        Simulation.getInstance().postEvent(newEvent);
    }

    public static void propagatePullRequestToChild(RelayingHost currentHost, Host child, PullRequestEvent requestEvent) {
        double txTime = currentHost.calculateTxTimeToChild(child, currentHost.getMtu());
        txTime += PROCESSING_DELAY;
        PullRequestEvent newEvent = requestEvent.createEventForNextHop(child, txTime);
        Simulation.getInstance().postEvent(newEvent);
    }

//    public static StorageReason findReasonWhyContentWasReceived(Host host, ContentReceivedEvent contentReceivedEvent) {
//        final RequestEvent originalRequestEvent = contentReceivedEvent.getOriginalRequestEvent();
//        if (originalRequestEvent instanceof PushRequestEvent) {
//            final Host pushInitiator = ((PushRequestEvent) originalRequestEvent).getPushInitiatorHost();
//            if (pushInitiator instanceof EdgeServer) {
//                return StorageReason.EXTERNAL_POPULARITY;
//            } else if (pushInitiator instanceof CloudUnit) {
//                return StorageReason.DIRECT_APP_PUSH_REQUEST;
//            } else if (pushInitiator instanceof UserEquipment) {
//                return StorageReason.NORMAL_FLOW;
//            }
//        }
//    }
}
