package edu.ucc.network.devices;

import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.AckReceivedEvent;
import edu.ucc.core.events.simulationevents.TestbedEvent;
import edu.ucc.core.transport.chunks.ChunksBuffer;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.core.transport.links.GroupedLink;
import edu.ucc.utils.TransmissionConstants;
import edu.ucc.utils.TransmissionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static edu.ucc.utils.Logging.printWarning;
import static edu.ucc.utils.TransmissionConstants.CHUNK_SIZE_FOR_ACK;
import static edu.ucc.utils.TransmissionConstants.PROCESSING_DELAY;

public abstract class RelayingHost extends Host {
    /**
     * The list of links to children units.
     * Notice that a {@link GroupedLink} is used since it comprises both UpLink and Downlinks.
     */
    protected Map<Host, GroupedLink> groupedLinksToChildren;

    /**
     * The list of references to children units.
     */
    protected List<Host> childrenUnits;

    /**
     * The buffer of chunks to deliver to children
     */
    protected ChunksBuffer downlinkChunksBuffer;

    /**
     * The list of chunks awaiting for ACK from children hosts.
     */
    private final ChunksBuffer downlinkChunksAwaitingAck;

    /**
     * The total bandwidth for the UpLink available for children units.
     */
    protected double totalUplinkBwForChildren;

    /**
     * The total bandwidth for the DownLink available for children units.
     */
    protected double totalDownlinkBwForChildren;

    private int nextChildHostIndex;

    /**
     * Basic constructor.
     *
     * @param id       The host's id.
     * @param storage  The available storage for the host.
     * @param location The host's location coordinates.
     * @param mtu      The host's Maximum Transmission Unit
     */
    public RelayingHost(int id, long storage, Location location, int mtu) {
        super(id, storage, location, mtu);
        this.downlinkChunksAwaitingAck = new ChunksBuffer(this);
        this.childrenUnits = new ArrayList<>();
        this.downlinkChunksBuffer = new ChunksBuffer(this);

        this.groupedLinksToChildren = new HashMap<>();
        this.nextChildHostIndex = 0;
    }


    private double calculateTxTimeToChild(GroupedLink linkToChild, int chunkSize, Host childHost) {
        double bandwidthToChild = linkToChild.getAvailableBandwidthProvidedByMe(this);
        double transmissionDelay = chunkSize / bandwidthToChild;
        double distance = this.location.distanceTo(childHost.location);
        double propagationDelay = 0;
        //        if (this instanceof CloudUnit) propagationDelay = distance / COPPER_PROPAGATION_FACTOR;
        //        else
        propagationDelay = distance / TransmissionConstants.LIGHT_SPEED;

        return transmissionDelay + propagationDelay;
    }

    public double calculateTxTimeToChild(Host childHost, int chunkSize) {
        final GroupedLink linkWithChild = findLinkWithChild(childHost);
        return calculateTxTimeToChild(linkWithChild, chunkSize, childHost);
    }

    public void deleteChunkFromDownlinkAckBuffer(Host childHost, ContentChunk chunk) {
        this.downlinkChunksAwaitingAck.removeChunkFromBuffer(childHost, chunk);
    }

    public List<ContentChunk> getPendingAckChunksForUE(int ueId) {
        return this.downlinkChunksAwaitingAck.getChunksForUE(ueId);
    }

    public void removeChunksWaitingAckForMovingUE(UserEquipment userEquipment) {
        downlinkChunksAwaitingAck.removeChunksForMovingUe(userEquipment.id);
    }

    public void removePendingChunksForMovingUE(UserEquipment movingUE) {
        downlinkChunksBuffer.removeChunksForMovingUe(movingUE.id);
    }

    public void addChunkToDownlinkBuffer(Host childHost, ContentChunk chunk) {
        this.downlinkChunksBuffer.addToBuffer(childHost, chunk);
    }

    public void addChunksToDownlinkBuffer(Host childHost, List<ContentChunk> chunks) {
        this.downlinkChunksBuffer.addToBuffer(childHost, chunks);
    }

    public List<ContentChunk> getPendingChunksForUE(int ueId) {
        return this.downlinkChunksBuffer.getChunksForUE(ueId);
    }

    /**
     * Obtains a reference to the list of children units.
     *
     * @return The host's list of children units.
     */
    public List<Host> getChildrenUnits() {
        return childrenUnits;
    }

    /**
     * Obtains a reference to the specified child device
     *
     * @param childrenId The id of the child device to look for.
     * @return A reference to the child host, null if nothing is found
     */
    public Host getChildUnit(int childrenId) {
        return childrenUnits.stream().filter(cu -> cu.getId() == childrenId).findFirst().orElse(null);
    }

    /**
     * Sets the host's list of children units.
     *
     * @param newChildren The list of children units for the host. Notice that type checking is performed (e.g., a BS is
     *                    not a valid child unit for a CloudUnit.
     */
    public abstract void setChildrenUnits(List<Host> newChildren);

    /**
     * Assigns the host's list of children units.
     *
     * @param newChildren The list of children to assign.
     */
    protected void assignChildren(List<Host> newChildren) {
        this.childrenUnits = newChildren;

        double uplinkPerChild = getTotalUplinkBwForChildren() / newChildren.size();
        double downlinkPerChild = getTotalDownlinkBwForChildren() / newChildren.size();

        validateBandwidth(downlinkPerChild);
        validateBandwidth(uplinkPerChild);
        for (Host child : newChildren) {
            GroupedLink groupedLink = new GroupedLink(this, child, downlinkPerChild, uplinkPerChild);
            groupedLinksToChildren.put(child, groupedLink);
            //            groupedLinksToChildren.add(groupedLink);
            child.setGroupedLinkToParent(groupedLink);
            child.setParentUnit(this);
        }
    }

    /**
     * Validates that the specified bandwidth is zero or positive.
     *
     * @param bandwidth The bandwidth to analyse.
     * @throws IllegalArgumentException If the bandwidth is negative. The simulation will be then stopped.
     */
    protected void validateBandwidth(double bandwidth) {
        if (bandwidth == 0.0) {
            printWarning("Warning, assigning a bandwidth of 0");
        }
        if (bandwidth < 0.0) {
            throw new IllegalArgumentException(String.format("Can't assign bandwidth for the siblings of this unit %s",
                    bandwidth));
        }
    }

    /**
     * Gets the total bandwidth in the UpLink for the host's children units.
     *
     * @return The total bandwidth (UpLink) for the host's children units.
     */
    public double getTotalUplinkBwForChildren() {
        return totalUplinkBwForChildren;
    }

    /**
     * Sets the total bandwidth in the UpLink for the host's children units.
     *
     * @param totalUplinkBwForChildren The total bandwidth (UpLink) to set for the host's children units.
     */
    public void setTotalUplinkBwForChildren(double totalUplinkBwForChildren) {
        this.totalUplinkBwForChildren = totalUplinkBwForChildren;
    }

    /**
     * Gets the total bandwidth in the DownLink for the host's children units.
     *
     * @return The total bandwidth (DownLink)  for the host's children units.
     */
    public double getTotalDownlinkBwForChildren() {
        return totalDownlinkBwForChildren;
    }

    /**
     * Sets the total bandwidth in the Downlink for the host's children units.
     *
     * @param totalDownlinkBwForChildren The total bandwidth (DownLink) to set for the shot's children units.
     */
    public void setTotalDownlinkBwForChildren(double totalDownlinkBwForChildren) {
        this.totalDownlinkBwForChildren = totalDownlinkBwForChildren;
    }

    protected GroupedLink findLinkWithChild(Host childHost) {
        return groupedLinksToChildren.get(childHost);
    }

    private void sendChunksDownwards() {
        Host nextHost;
        if (downlinkChunksBuffer.areTherePendingChunks()) {
            // This code searches for the next host with pending chunks.
            List<ContentChunk> chunksForChild;
            do {
                nextChildHostIndex++;
                if (nextChildHostIndex >= childrenUnits.size()) {
                    nextChildHostIndex = 0;
                }
                nextHost = childrenUnits.get(nextChildHostIndex);
                chunksForChild = downlinkChunksBuffer.getChunksForNextHopHost(nextHost);
            } while (chunksForChild.isEmpty());

            final ContentChunk nextChunk = chunksForChild.get(0);
            sendNextChunkDownwards(nextHost, nextChunk);
        }
    }

    public void sendChunksDownwardsAfterHandover() {
        sendChunksDownwards();
    }

    public void sendChunksDownwardsAfterAck() {
        sendChunksDownwards();
    }

    /**
     * This method is called to try to relay a chunk recently received by the current host and that is addressed to the
     * child device.
     *
     * @param childDevice The child device to whom chunks must be delivered.
     */
    public void sendChunkDownwardsForHost(Host childDevice) {
        // We will only send if there is nothing pending for this host
        if (downlinkChunksAwaitingAck.noChunksForHost(childDevice)) {
            nextChildHostIndex = childrenUnits.indexOf(childDevice);
            final List<ContentChunk> chunksForHost = downlinkChunksBuffer.getChunksForNextHopHost(childDevice);
            sendNextChunkDownwards(childDevice, chunksForHost.get(0));
        }
    }

    private void sendNextChunkDownwards(Host nextHopHost, ContentChunk nextChunk) {
        GroupedLink linkToChild = findLinkWithChild(nextHopHost);
        final double currentTime = Simulation.getInstance().getCurrentTime();
        if (linkToChild.getAvailableBandwidthProvidedByMe(this) > nextChunk.getChunkSize()) {
            downlinkChunksAwaitingAck.addToBuffer(nextHopHost, nextChunk);
            Simulation.getInstance().getEventsBroker().onChunkTransmittedDownwards(this, nextChunk, currentTime);
            final boolean b = downlinkChunksBuffer.removeChunkFromBuffer(nextHopHost, nextChunk);
            if (!b) throw new RuntimeException("I could not delete the chunk from buffer!!");

            double txTime = calculateTxTimeToChild(linkToChild, nextChunk.getChunkSize(), nextHopHost);
            txTime += PROCESSING_DELAY; // This adds a processing delay to each sent chunk
            linkToChild.decreaseBandwidthFromMe(this, nextChunk.getChunkSize());
            TestbedEvent sendChunkEvent = TransmissionUtils.createEventForChunk(this, nextHopHost,
                    txTime, nextChunk);
            Simulation.getInstance().postEvent(sendChunkEvent);
        } else {
            printWarning(String.format("Link bandwidth from %s %s to %s %s is not enough to move %s bits",
                    getTypeAsString(), this.id, nextHopHost.getTypeAsString(), nextHopHost.id,
                    nextChunk.getChunkSize()));
        }
    }


    public void increaseBandwidthFromChild(ContentChunk chunk, Host childHost) {
        final GroupedLink linkWithChild = findLinkWithChild(childHost);
        linkWithChild.increaseBandwidthFromMe(childHost, chunk.getChunkSize());
    }

    /**
     * Sends the ACK about received chunk to sender. Notice that here it can be a parent or children, while for ES it
     * can be a sibling
     *
     * @param ackedHost  The host to which the ACK must be sent.
     * @param ackedChunk The chunk that has been received and whose ack should be notified
     */
    @Override
    public boolean sendAck(Host ackedHost, ContentChunk ackedChunk) {
        if (super.sendAck(ackedHost, ackedChunk)) return true;
        double txTime = 0.0;
        if (childrenUnits.contains(ackedHost)) {
            txTime = calculateTxTimeToChild(ackedHost, CHUNK_SIZE_FOR_ACK);
        } else {
            return false;
        }
        double timestamp = Simulation.getInstance().getCurrentTime() + txTime + PROCESSING_DELAY;
        TestbedEvent ackEvent = new AckReceivedEvent(this, ackedHost, timestamp, ackedChunk);
        Simulation.getInstance().postEvent(ackEvent);
        return true;
    }
}
