package edu.ucc.network.architecture;

import edu.ucc.network.devices.Location;

import java.util.ArrayList;
import java.util.List;

public class TopologyMakerLayers {
    private int numOfLayers;
    private double bsRange;

    public TopologyMakerLayers(int numOfLayers, double bsRange) {
        this.numOfLayers = numOfLayers;
        this.bsRange = bsRange;
    }

    public List<List<Location>> createLocationsForBSs() {
        List<List<Location>> chains = new ArrayList<>();
        for (int i = 0; i < numOfLayers; i++) {
            List<Location> locationsThisLayer = createLocationsForBSsInLayer(i);
            chains.add(locationsThisLayer);
        }
        return chains;
    }

    public List<Location> createLocationsForBSsInLayer(int layer) {
        int circlesToDraw = calculateBSsInLayer(layer);
        double commonMultiplier = Math.sqrt(.75) * bsRange;
        int horizontalCoefficient = layer * 2;
        double verticalCoefficient = 0;
        double x, y, xDirection, yDirection, xDelta, yDelta;
        List<Location> output = new ArrayList<>();
        for (int i = 0; i < circlesToDraw; i++) {
            x = horizontalCoefficient * commonMultiplier;
            y = verticalCoefficient * bsRange;
            output.add(new Location(x, y));
            xDirection = findDirectionForX(i + 1, layer);
            yDirection = findDirectionForY(i + 1, layer);
            xDelta = findDeltaFactorForX(i + 1, layer);
            yDelta = findDeltaFactorForY(i + 1, layer);
            horizontalCoefficient += (xDirection * xDelta);
            verticalCoefficient += (yDirection * yDelta);
        }
        return output;
    }

    /***
     * Finds the magnitude of the change in the X axis for the specified BS.
     * @param bsNumber The BS to calculate the magnitude for.
     * @param layer The layer on which the BS is.
     * @return The magnitude of change in X axis.
     */
    private double findDeltaFactorForX(int bsNumber, int layer) {
        if ((layer + 1) <= bsNumber && bsNumber <= (2 * layer)) return 2;
        if (4 * layer + 1 <= bsNumber && bsNumber <= 5 * layer) return 2;
        return 1;
    }

    /**
     * Finds the magnitude of the change in the Y axis for the specified BS.
     *
     * @param bsNumber The BS to calculate the magnitude for.
     * @param layer    The layer on which the BS is.
     * @return The magnitude of change in Y axis.
     */
    private double findDeltaFactorForY(int bsNumber, int layer) {
        if (layer + 1 <= bsNumber && bsNumber <= 2 * layer) return 0;
        if (4 * layer + 1 <= bsNumber && bsNumber <= 5 * layer) return 0;
        return 1.5;
    }

    /***
     * Finds the direction on which the Y axis is moving for the bsNumber BS in the specified layer.
     * @param bsNumber The BS to find the Y direction for.
     * @param layer The layer on which the BS is.
     * @return +1 if the direction in the Y axis is growing (going up), -1 otherwise
     */
    private byte findDirectionForY(int bsNumber, int layer) {
        if (layer + 1 <= bsNumber && bsNumber <= 2 * layer) return 0;
        if (4 * layer + 1 <= bsNumber && bsNumber <= 5 * layer) return 0;
        if ((1 + 2 * layer <= bsNumber) && bsNumber <= 4 * layer) return -1;
        return 1;
    }

    /***
     * Finds the direction on which the X axis is moving for the bsNumber BS in the specified layer.
     * @param bsNumber The BS to find the X direction for.
     * @param layer The layer on which the BS is.
     * @return +1 if the direction in the X axis is growing (going to right), -1 otherwise.
     */
    private byte findDirectionForX(int bsNumber, int layer) {
        final int totalBSs = calculateBSsInLayer(layer);
        int top = 1 + (totalBSs / 2);
        if (bsNumber < top) return -1;
        return 1;
    }


    /***
     * Calculates the number of base stations to create at the concentric layer
     * @param layer The number of the layer (0-based index).
     * @return The number of base stations to position in the specified layer.
     */
    private int calculateBSsInLayer(int layer) {
        return layer == 0 ? 1 : layer * 6;
    }
}
