package edu.ucc.network.architecture;

import edu.ucc.core.ContentTransmitter;
import edu.ucc.core.HandoverManager;
import edu.ucc.core.simpleimplementations.*;
import edu.ucc.network.devices.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static edu.ucc.utils.HierarchyUtils.getBaseStationForLocation;
import static edu.ucc.utils.HierarchyUtils.launchKMeans;

public class ArchitectureGenerator {
    private final double bsRange;
    private final double mapHeight;
    private final double mapWidth;
    private final double totalUplinkForEsChildren;
    private final double totalDownlinkForEsChildren;
    private final double totalUplinkForCloudChildren;
    private final double totalDownlinkForCloudChildren;
    private final double totalBwForEsSiblings;
    private final double totalDownlinkBwForBsChildren;
    private final double totalUplinkBwForBsChildren;
    private final int numEdgeServers;
    private final int numUserEquipments;

    private final List<List<BaseStation>> baseStations;
    private final List<UserEquipment> unassignedUEs;
    private final int mtu;
    private List<BaseStation> flatBaseStations;
    private NetworkArchitecture networkArchitecture;
    private CloudUnit cloudUnit;
    private final List<EdgeServer> edgeServers;

    private final ContentTransmitter contentTransmitterUserEquipment;
    private final ContentTransmitter contentTransmitterBaseStation;
    private final ContentTransmitter contentTransmitterEdge;
    private final ContentTransmitter contentTransmitterCloud;

    private final HandoverManager handoverManagerUserEquipment;
    private final HandoverManager handoverManagerBaseStation;
    private final HandoverManager handoverManagerEdgeServer;
    private final HandoverManager handoverManagerCloud;

    private final ArchitectureParameters architectureParameters;

    public ArchitectureGenerator(ArchitectureParameters architectureParameters) {
        this.architectureParameters = architectureParameters;
        this.bsRange = architectureParameters.getBsRange();
        this.mapHeight = architectureParameters.getMapHeight();
        this.mapWidth = architectureParameters.getMapWidth();
        this.totalUplinkForCloudChildren = architectureParameters.getTotalUplinkForCloudChildren();
        this.totalDownlinkForCloudChildren = architectureParameters.getTotalDownlinkForCloudChildren();
        this.totalUplinkForEsChildren = architectureParameters.getTotalUplinkForEsChildren();
        this.totalDownlinkForEsChildren = architectureParameters.getTotalDownlinkForEsChildren();
        this.totalBwForEsSiblings = architectureParameters.getTotalBwForEsSiblings();
        this.totalUplinkBwForBsChildren = architectureParameters.getTotalUplinkBwForBsChildren();
        this.totalDownlinkBwForBsChildren = architectureParameters.getTotalDownlinkBwForBsChildren();
        this.numEdgeServers = architectureParameters.getNumEdgeServers();
        this.numUserEquipments = architectureParameters.getNumUserEquipments();
        this.mtu = architectureParameters.getMtu();
        this.baseStations = new ArrayList<>();
        this.edgeServers = new ArrayList<>();
        this.unassignedUEs = new ArrayList<>();

        this.contentTransmitterBaseStation = new BaseStationContentTransmitter();
        this.contentTransmitterEdge = new EdgeContentTransmitter();
        this.contentTransmitterCloud = new CloudContentTransmitter();
        this.contentTransmitterUserEquipment = new UserEquipmentContentTransmitter();

        this.handoverManagerBaseStation = new SimpleBaseStationHandoverManager();
        this.handoverManagerUserEquipment = new SimpleUserEquipmentHandoverManager();
        this.handoverManagerEdgeServer = new SimpleEdgeHandoverManager();
        this.handoverManagerCloud = new SimpleCloudHandoverManager();
    }

    public List<UserEquipment> getUnassignedUEs() {
        return unassignedUEs;
    }

    public void createNetworkArchitecture() {
        createBaseStations();
        createEdgeServersAndAssignBaseStations();
        createCloudUnitAndAssignEdgeServers();
        assignSiblingsForEdgeServers();
        createUserEquipments();
        this.networkArchitecture = new NetworkArchitecture(cloudUnit);
        networkArchitecture.setUserEquipments(unassignedUEs);
        networkArchitecture.setBaseStations(flatBaseStations);
        networkArchitecture.setEdgeServers(edgeServers);
    }

    private void createUserEquipments() {
        for (int i = 0; i < numUserEquipments; i++) {
            UserEquipment userEquipment = new UserEquipment(i, 0, null, mtu);
            userEquipment.setContentTransmitter(contentTransmitterUserEquipment);
            userEquipment.setHandoverManager(handoverManagerUserEquipment);
            this.unassignedUEs.add(userEquipment);
        }
    }

    private void assignSiblingsForEdgeServers() {
        EdgeServersSiblingsAssigner esSiblingAssigner = new EdgeServersSiblingsMeshAssigner(this.edgeServers);
        esSiblingAssigner.assignSiblings();
    }

    private EdgeServer createEdgeServer(int id) {
        final EdgeServer edgeServer = new EdgeServer(id, architectureParameters.getStoragePerEdgeServer(), null, mtu);
        edgeServer.setBandwidthForSiblings(totalBwForEsSiblings);
        return edgeServer;
    }

    private void createEdgeServersAndAssignBaseStations() {
        // first get the raw coordinates for each edge server
        Map<Location, List<BaseStation>> resultSet = findCentroidsForEdgeServers();
        int i = 0;

        for (Map.Entry<Location, List<BaseStation>> entry : resultSet.entrySet()) {
            Location location = entry.getKey();
            List<BaseStation> involvedBss = entry.getValue();
            EdgeServer edgeServer = createEdgeServer(i++);
            BaseStation closestBaseStation = getBaseStationForLocation(location, flatBaseStations, bsRange);
            edgeServer.setLocation(closestBaseStation.getLocation());
            //            double totalDownlinkThisEdge = calculateTotalDownlinkBwBetweenEdgeAndBSs(involvedBss);
            //            double totalUplinkThisEdge = calculateTotalUplinkBwBetweenEdgeAndBSs(involvedBss);
            //            edgeServer.setTotalDownlinkBwForChildren(totalDownlinkThisEdge);
            //            edgeServer.setTotalUplinkBwForChildren(totalUplinkThisEdge);
            edgeServer.setTotalDownlinkBwForChildren(totalDownlinkForEsChildren);
            edgeServer.setTotalUplinkBwForChildren(totalUplinkForEsChildren);
            edgeServer.setContentTransmitter(contentTransmitterEdge);
            edgeServer.setHandoverManager(handoverManagerEdgeServer);

            List<Host> bssAsHosts = new ArrayList<>(involvedBss);
            edgeServer.setChildrenUnits(bssAsHosts);

            for (BaseStation bs : involvedBss) {
                bs.setParentUnit(edgeServer);
            }

            edgeServers.add(edgeServer);
        }
    }

    private Map<Location, List<BaseStation>> findCentroidsForEdgeServers() {
        return launchKMeans(flatBaseStations, numEdgeServers);
    }

    private void createCloudUnitAndAssignEdgeServers() {
        final Location cloudLocation = new Location(mapWidth / 2.0, mapHeight / 2.0);
        // 250 km (~ Dublin)
        //        double requiredDistance = 250 * KILOMETER;
        //        double coordinate = requiredDistance / Math.sqrt(2);
        //        final Location cloudLocation = new Location(-coordinate, -coordinate);
        this.cloudUnit = new CloudUnit(0, Long.MAX_VALUE, cloudLocation, mtu);
        cloudUnit.setTotalUplinkBwForChildren(totalUplinkForCloudChildren);
        cloudUnit.setTotalDownlinkBwForChildren(totalDownlinkForCloudChildren);
        List<Host> edgeServersAsHosts = new ArrayList<>(edgeServers);
        cloudUnit.setChildrenUnits(edgeServersAsHosts);
        cloudUnit.setContentTransmitter(contentTransmitterCloud);
        cloudUnit.setHandoverManager(handoverManagerCloud);

        for (EdgeServer edgeServer : edgeServers) {
            edgeServer.setParentUnit(cloudUnit);
        }
    }

    private void createBaseStations() {
        final int columns = calculateColumnsForBSsCoverage(mapWidth);
        final int rows = calculateRowsForBSsCoverage(mapHeight);
        TopologyMakerMesh topologyMakerMesh = new TopologyMakerMesh(columns, rows, bsRange);
        final List<List<Location>> locationsForBSs = topologyMakerMesh.createLocationsForBSs();

        // Recall that at the moment the mesh generator is returning nested lists column x rows
        int deviceId = 0;
        for (List<Location> column : locationsForBSs) {
            List<BaseStation> baseStationsThisColumn = new ArrayList<>();
            for (Location location : column) {
                BaseStation bs = new BaseStation(deviceId++, 0, location, mtu, bsRange);
                bs.setTotalUplinkBwForChildren(totalUplinkBwForBsChildren);
                bs.setTotalDownlinkBwForChildren(totalDownlinkBwForBsChildren);
                bs.setContentTransmitter(contentTransmitterBaseStation);
                bs.setHandoverManager(handoverManagerBaseStation);
                baseStationsThisColumn.add(bs);
            }
            baseStations.add(baseStationsThisColumn);
        }

        this.flatBaseStations = new ArrayList<>();
        this.baseStations.forEach(this.flatBaseStations::addAll);
    }

    private int calculateRowsForBSsCoverage(double zoneHeight) {
        int division = (int) Math.ceil(zoneHeight / (Math.sqrt(0.75) * bsRange));
        if (division % 2 == 0) division += 1;
        return (division + 1) / 2;
    }

    private int calculateColumnsForBSsCoverage(double zoneWidth) {
        double division = zoneWidth / bsRange;
        if (division > 5.0) {
            int scale = (int) Math.floor((division - 5.0) / 1.5);
            return 5 + scale;
        }
        if (division > 3.5) return 4;
        if (division > 2.0) return 3;
        if (division > 0.5) return 2;
        return 1;
    }

    public NetworkArchitecture getNetworkArchitecture() {
        return networkArchitecture;
    }

    public List<List<BaseStation>> getBaseStations() {
        return baseStations;
    }
}
