package edu.ucc.core.transport.links;

import edu.ucc.network.devices.Host;

import java.io.Serializable;

/***
 * This class is a wrapper for modeling the simultaneous uplink and downlink available between connected devices.
 */
public class GroupedLink implements Serializable {
    private Host hostA;
    private Host hostB;
    private HalfDuplexLink halfDuplexLinkAB;
    private HalfDuplexLink halfDuplexLinkBA;

    public GroupedLink(Host hostA, Host hostB, double bandwidthAB, double bandwidthBA) {
        this.hostA = hostA;
        this.hostB = hostB;
        this.halfDuplexLinkAB = new HalfDuplexLink(hostA, hostB, bandwidthAB);
        this.halfDuplexLinkBA = new HalfDuplexLink(hostB, hostA, bandwidthBA);
    }

    public GroupedLink(Host hostA, Host hostB) {
        this(hostA, hostB, 0.0, 0.0);
    }

    public HalfDuplexLink getLinkWithOrigin(Host origin) {
        if (origin == hostA) {
            return this.halfDuplexLinkAB;
        } else if (origin == hostB) {
            return this.halfDuplexLinkBA;
        }
        return null;
    }

    /***
     * Finds out whether this link is connecting the specified hosts in any direction
     * @param hostA One host to be on one end of the link.
     * @param hostB One host to be on one end of the link.
     * @return true if this link connects the specified hosts, false otherwise
     */
    public boolean connectionBetween(Host hostA, Host hostB) {
        return (this.hostA == hostA && this.hostB == hostB) || (this.hostA == hostB && this.hostB == hostA);
    }

    /***
     * Gets the bandwidth that is provided by this host (i.e., when the specified host is the father host).
     * @param host The host that provides the bandwidth.
     * @return The available bandwidth.
     */
    public double getAvailableBandwidthProvidedByMe(Host host) {
        HalfDuplexLink halfDuplexLink = validateAndGetHalfDuplex(host);
        return halfDuplexLink.getAvailableBandwidth();
    }

    /***
     * Gets the bandwidth that is provided by this host (i.e., when the specified host is the father host).
     * @param host The host that provides the bandwidth.
     * @return The available bandwidth.
     */
    public double getTotalBandwidthProvidedByMe(Host host) {
        HalfDuplexLink halfDuplexLink = validateAndGetHalfDuplex(host);
        return halfDuplexLink.getTotalBandwidth();
    }

    /***
     * Gets the correct {@link HalfDuplexLink} validating that the specified host is one of the end systems.
     * @param host The host that acts as the father/origin in the HalfDuplexLink
     * @return The HalfDuplexLink in this GroupedLink where the host is the origin/father
     */
    private HalfDuplexLink validateAndGetHalfDuplex(Host host) {
        HalfDuplexLink halfDuplexLink;
        if (host == hostA) {
            halfDuplexLink = getLinkWithOrigin(hostA);
        } else if (host == hostB) {
            halfDuplexLink = getLinkWithOrigin(hostB);
        } else {
            throw new IllegalArgumentException("The specified host is neither the origin nor the destination " +
                    "in this duplex link.");
        }
        return halfDuplexLink;
    }

    /***
     * Increases the bandwidth in the corresponding {@link HalfDuplexLink} where the specified host is the origin/father.
     * @param host The host that acts as the father/origin in the HalfDuplexLink
     * @param increase The amount to increase the bandwidth
     */
    public void increaseBandwidthFromMe(Host host, double increase) {
        HalfDuplexLink halfDuplexLink = validateAndGetHalfDuplex(host);
        halfDuplexLink.increaseAvailableBandwidth(increase);
    }

    /***
     * Decreases the bandwidth in the corresponding {@link HalfDuplexLink} where the specified host is the origin/father.
     * @param host The host that acts as the father/origin in the HalfDuplexLink
     * @param decrease The amount to decrease the bandwidth.
     */
    public void decreaseBandwidthFromMe(Host host, double decrease) {
        HalfDuplexLink halfDuplexLink = validateAndGetHalfDuplex(host);
        halfDuplexLink.reduceAvailableBandwidth(decrease);
    }


    /***
     * Increases the total bandwidth in the corresponding {@link HalfDuplexLink} where the specified host is the origin.
     * @param host The host that is acting as the origin in the HalfDuplexLink.
     * @param increase The amount to increase in the total bandwidth.
     */
    public void increaseTotalBandwidthFromMe(Host host, double increase) {
        HalfDuplexLink halfDuplexLink = validateAndGetHalfDuplex(host);
        halfDuplexLink.increaseTotalBandwidth(increase);
    }

    /***
     * Decreases the total bandwidth in the corresponding {@link HalfDuplexLink} where the specified host is the origin.
     * @param host The host that is acting as the origin in the HalfDuplexLink.
     * @param decrease The amon to decrease in the total bandwidth.
     */
    public void decreaseTotalBandwidthFromMe(Host host, double decrease) {
        HalfDuplexLink halfDuplexLink = validateAndGetHalfDuplex(host);
        halfDuplexLink.reduceTotalBandwidth(decrease);
    }
}
