package edu.ucc.core.simpleimplementations;

import edu.ucc.core.ContentTransmitter;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.entities.Data;
import edu.ucc.network.devices.BaseStation;
import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.UserEquipment;
import edu.ucc.utils.TransmissionUtils;

import static edu.ucc.utils.Logging.printWarning;
import static edu.ucc.utils.TransmissionUtils.*;

/**
 * This class (implementation of {@link ContentTransmitter}) defines how a UE handles content requests and associated
 * chunk transmissions. Notice that an UE under specific circumstances might be the producer of content ({@link edu.ucc.entities.IoTData}.
 * This, it is possible for a UE to serve content requests coming from the cloud or other UEs in the network.
 */
public class UserEquipmentContentTransmitter implements ContentTransmitter {
    @Override
    public void onPullRequestReceived(Host host, PullRequestEvent pullRequestEvent) {
        final UserEquipment userEquipment = (UserEquipment) host;
        Simulation.getInstance().getEventsBroker().onPullRequestReceived(userEquipment, pullRequestEvent);

        final int eventDirection = pullRequestEvent.getEventDirection();
        switch (eventDirection) {
            case UPWARDS_DIRECTION:
                TransmissionUtils.propagatePullRequestToParent(host, pullRequestEvent);
                break;
            case SAME_LEVEL_DIRECTION:
                throw new RuntimeException(String.format("Same level pull request at %s", userEquipment));
            case DOWNWARDS_DIRECTION:
                dispatchDataRequestLocally(userEquipment, (PullDataRequestEvent) pullRequestEvent);
                break;
            default:
                throw new RuntimeException("Unrecognized event direction");
        }
    }

    private void dispatchDataRequestLocally(UserEquipment userEquipment, PullDataRequestEvent requestEvent) {
        final BaseStation nextHopBS = (BaseStation) userEquipment.getParentUnit();
        final Data referredContent = requestEvent.getReferredContent();
        int chunkSize = userEquipment.getMtu();
        int totalChunks = getNumberOfChunksForContent(referredContent, chunkSize);

        userEquipment.addChunksForNewRequest(nextHopBS, requestEvent, totalChunks, chunkSize);
        userEquipment.sendChunksToParent();
    }

    @Override
    public void onAckReceived(Host host, AckReceivedEvent ackReceivedEvent) {
        UserEquipment userEquipment = (UserEquipment) host;
        ContentChunk chunk = ackReceivedEvent.getContentChunk();

        userEquipment.deleteChunkFromUplinkAckBuffer(chunk);
        userEquipment.sendChunksToParentAfterAck();
        Simulation.getInstance().getEventsBroker().onAckReceived(host, ackReceivedEvent);
    }

    @Override
    public void onChunkReceived(Host host, ChunkReceivedEvent chunkReceivedEvent) {
        final UserEquipment userEquipment = (UserEquipment) host;
        ContentChunk chunk = chunkReceivedEvent.getChunk();

        if (!chunk.isThisDestinationHost(userEquipment)) {
            throw new RuntimeException("Chunk received at a UE but this is not the destination");
        }

        final BaseStation baseStation = (BaseStation) chunkReceivedEvent.getEventOriginHost();
        if (!userEquipment.isThisMyParent(baseStation)) {
            printWarning(String.format("At UE %s: Received chunk %s/%s from a previous BS parent %s, @ %s",
                    userEquipment.getId(), chunk.getChunkNumber(), chunk.getTotalChunks() - 1, baseStation.getId(),
                    chunkReceivedEvent.getTimestamp()));
            return;
        }

        Simulation.getInstance().getEventsBroker().onChunkReceived(host, chunkReceivedEvent);
        userEquipment.sendAck(baseStation, chunk);
        userEquipment.increaseBandwidthFromParent(chunk);
    }

    @Override
    public void onPushRequestReceived(Host host, PushRequestEvent pushRequestEvent) {
        UserEquipment userEquipment = (UserEquipment) host;
        final BaseStation nextHopBS = (BaseStation) userEquipment.getParentUnit();
        Simulation.getInstance().getEventsBroker().onPushRequestReceived(host, pushRequestEvent);

        if (pushRequestEvent instanceof PushDataRequestEvent) {
            final Data referredContent = (Data) pushRequestEvent.getReferredContent();
            int chunkSize = userEquipment.getMtu();
            int totalChunks = getNumberOfChunksForContent(referredContent, chunkSize);
            userEquipment.addChunksForNewRequest(nextHopBS, pushRequestEvent, totalChunks, chunkSize);
            userEquipment.sendChunksToParent();
        } else {
            throw new RuntimeException("Not implemented yet, a UE pushing a file");
        }

    }
}
