package edu.ucc.core.simpleimplementations;

import edu.ucc.core.ContentTransmitter;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.core.transport.routing.RoutingTable;
import edu.ucc.network.devices.BaseStation;
import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.UserEquipment;
import edu.ucc.utils.TransmissionUtils;

import static edu.ucc.utils.Logging.printInfo;
import static edu.ucc.utils.TransmissionUtils.*;

/**
 * This class (implementation of {@link ContentTransmitter}) defines how a base station handles content requests and
 * associated chunks transmission. Notice that a BaseStation simply relays requests and chunks, as it does not have any
 * storage, caching, or decision making features.
 */
public class BaseStationContentTransmitter implements ContentTransmitter {
    @Override
    public void onPullRequestReceived(Host host, PullRequestEvent pullRequestEvent) {
        BaseStation baseStation = (BaseStation) host;

        switch (pullRequestEvent.getEventDirection()) {
            case UPWARDS_DIRECTION:
                TransmissionUtils.propagatePullRequestToParent(baseStation, pullRequestEvent);
                break;
            case SAME_LEVEL_DIRECTION:
                throw new RuntimeException(String.format("Same level pull request at %s", baseStation));
            case DOWNWARDS_DIRECTION:
                final Host dataLocationHost = pullRequestEvent.getPullDestinationHost();
                if (!(dataLocationHost instanceof UserEquipment))
                    throw new RuntimeException("Pull Request in the downlink but dataLocationHost is not a UE");

                final RoutingTable routingTable = Simulation.getInstance().getNetworkArchitecture().getRoutingTable();
                final UserEquipment nextChildHop = routingTable.getEntryForUe((UserEquipment) dataLocationHost).getUserEquipmentPartInRoute();
                propagatePullRequestToChild(baseStation, nextChildHop, pullRequestEvent);
                break;
            default:
                throw new RuntimeException("Unrecognized event direction");
        }

        Simulation.getInstance().getEventsBroker().onPullRequestReceived(baseStation, pullRequestEvent);
    }

    @Override
    public void onAckReceived(Host host, AckReceivedEvent ackReceivedEvent) {
        BaseStation baseStation = (BaseStation) host;

        switch (ackReceivedEvent.getEventDirection()) {
            case UPWARDS_DIRECTION:
                processAckUpwards(baseStation, ackReceivedEvent);
                break;
            case SAME_LEVEL_DIRECTION:
                throw new RuntimeException(String.format("Same level ack received at %s", baseStation));
            case DOWNWARDS_DIRECTION:
                processAckDownwards(baseStation, ackReceivedEvent);
                break;
            default:
                throw new RuntimeException("Unrecognized event direction");
        }

        Simulation.getInstance().getEventsBroker().onAckReceived(baseStation, ackReceivedEvent);
    }

    /***
     * Process an ACK that is coming downwards, that is, from an ES
     * @param baseStation The reference to the BaseStation handling the event.
     * @param ackReceivedEvent The received event.
     */
    private void processAckDownwards(BaseStation baseStation, AckReceivedEvent ackReceivedEvent) {
        ContentChunk chunk = ackReceivedEvent.getContentChunk();
        baseStation.deleteChunkFromUplinkAckBuffer(chunk);
        baseStation.sendChunksToParentAfterAck();
    }

    /***
     * Process an ACK that is coming upwards, that is, from a UE.
     * @param baseStation The reference to the BaseStation handling the event.
     * @param ackReceivedEvent The received event.
     */
    private void processAckUpwards(BaseStation baseStation, AckReceivedEvent ackReceivedEvent) {
        ContentChunk chunk = ackReceivedEvent.getContentChunk();
        UserEquipment userEquipment = (UserEquipment) ackReceivedEvent.getEventOriginHost();
        baseStation.deleteChunkFromDownlinkAckBuffer(userEquipment, chunk);
        baseStation.sendChunksDownwardsAfterAck();
    }

    @Override
    public void onChunkReceived(Host host, ChunkReceivedEvent chunkReceivedEvent) {
        BaseStation baseStation = (BaseStation) host;
        final ContentChunk chunk = chunkReceivedEvent.getChunk();
        final RequestEvent requestEvent = chunk.getRequestEvent();
        final Host eventOriginHost = chunkReceivedEvent.getEventOriginHost();
        Simulation.getInstance().getEventsBroker().onChunkReceived(baseStation, chunkReceivedEvent);

        UserEquipment userEquipment;
        switch (chunkReceivedEvent.getEventDirection()) {
            case UPWARDS_DIRECTION:
                userEquipment = (UserEquipment) eventOriginHost;
                // Recall that we might receive a chunk from a UE that is not already in this BS
                if (baseStation.containsUserEquipment(userEquipment))
                    processChunkUpwards(baseStation, userEquipment, chunk);

                break;
            case DOWNWARDS_DIRECTION:
                userEquipment = (UserEquipment) requestEvent.getContentDestinationHost();
                if (baseStation.containsUserEquipment(userEquipment))
                    processChunksDownwards(baseStation, userEquipment, chunk);

                break;
            default:
                throw new RuntimeException("Not valid direction here");
        }

        if (baseStation.containsUserEquipment(userEquipment)) baseStation.sendAck(eventOriginHost, chunk);
        else {
            int requestId = requestEvent.getRequestId();
            final double timestamp = chunkReceivedEvent.getTimestamp();
            printInfo((String.format("At BS %s: DISCARDING Chunk %s for request %s received @ %s, " +
                            "UE %s not here due to HO", baseStation.getId(),
                    chunk.getChunkNumber(), requestId, timestamp, userEquipment.getId())));
        }
    }

    private void processChunkUpwards(BaseStation baseStation, UserEquipment userEquipment, ContentChunk chunk) {
        baseStation.increaseBandwidthFromChild(chunk, userEquipment);
        baseStation.addChunkForTransmissionToParent(chunk);
        baseStation.sendChunksToParent();
    }

    private void processChunksDownwards(BaseStation baseStation, UserEquipment userEquipment, ContentChunk chunk) {
        baseStation.increaseBandwidthFromParent(chunk);
        baseStation.addChunkToDownlinkBuffer(userEquipment, chunk);
        baseStation.sendChunkDownwardsForHost(userEquipment);
    }

    @Override
    public void onPushRequestReceived(Host host, PushRequestEvent pushRequestEvent) {
        throw new RuntimeException("The base stations do not push content in this implementation");
    }
}
