package core.processingchain;

import core.app.App;
import core.control.MessageData;
import core.control.PushMessageRequestEvent;
import core.control.SystemMessageData;
import core.processingchain.actions.BaseAction;
import core.processingchain.actions.exceptions.AppOutOfStorageException;
import core.processingchain.actions.exceptions.HostOutOfStorageException;
import core.processingchain.actions.exceptions.SolutionException;
import core.processingchain.actions.policies.BasePolicy;
import core.processingchain.events.ApplicationEvent;
import core.processingchain.events.SolutionEvent;
import core.processingchain.events.detectors.BaseEventDetector;
import core.processingchain.metadata.MetadataAPI;
import core.processingchain.metadata.MetadataType;
import core.processingchain.metadata.extractor.AppMetadataExtractor;
import core.processingchain.metadata.extractor.MetadataExtractor;
import core.processingchain.metadata.pool.BaseMetadata;
import core.processingchain.metadata.pool.SolutionEventMetadata;
import core.processingchain.preprocessors.PPOutcome;
import core.processingchain.preprocessors.PreProcessor;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static core.SolutionConstants.IMMEDIATE_PROCESSING;
import static edu.ucc.utils.FileConstants.KILOBYTE;
import static edu.ucc.utils.Logging.*;

@SuppressWarnings("rawtypes, unchecked")
public class ProcessingPipeline {
    private final Map<Integer, App> apps;
    private final MetadataAPI metadataAPI;
    private final AppMetadataExtractor appMetadataExtractor;
    private final Host localHost;


    public ProcessingPipeline(Host localHost) {
        this.apps = new HashMap<>();
        this.metadataAPI = new MetadataAPI();
        this.appMetadataExtractor = new AppMetadataExtractor();
        this.localHost = localHost;
    }

    public void registerApp(App app) {
        if (validateApp(app)) {
            apps.put(app.getAppId(), app);
            final BaseMetadata appMetadata = appMetadataExtractor.extractMetadata(app);
            metadataAPI.addMetadata(appMetadata);
        } else {
            printError(String.format("App %s could not be validated so it is rejected", app.getName()));
        }
    }

    private boolean validateApp(App app) {
        return app.getTotalSpace() > 0;
    }

    public PipelineResult executeProcessingChain(Host localHost, ObservableTestbedEvent eventFromTestbed) {
        final int appId = eventFromTestbed.getAppId();
        final App app = obtainApp(appId);
        if (app.getProcessingInterval() == IMMEDIATE_PROCESSING) {
            return processEvent(localHost, eventFromTestbed, Simulation.getInstance().getCurrentTime());
        } else {
            // We don't retry here (delete content) because other pending events would be deleted.
            try {
                app.addEventForLaterProcessing(eventFromTestbed);
                return null;
            } catch (AppOutOfStorageException | HostOutOfStorageException e) {
                printError(String.format("The data for event %s could not be saved for later execution",
                        eventFromTestbed));
                e.printStackTrace();
                return null;
            }
        }
    }

    /**
     * Process an event at a host
     *
     * @param localHost        The host that is processing the event
     * @param eventFromTestbed The event being processed.
     * @param timestamp        The timestamp at which the processing pipeline is invoked. From this time, the timestamps
     *                         for internal produced components are created.
     */
    public PipelineResult processEvent(Host localHost, ObservableTestbedEvent eventFromTestbed, double timestamp) {
        // 1. Extract metadata:
        BaseMetadata metadata = extractMetadata(eventFromTestbed);

        // 2. Execute pre-processing:
        PPOutcome ppOutcome = preprocessData(eventFromTestbed, metadata, timestamp);

        if (ppOutcome == null) return new PipelineResult(null, null);

        // 3. execute event detection:
        SolutionEvent detectedEvent = detectEvent(localHost, ppOutcome);

        if (detectedEvent == null) return new PipelineResult(ppOutcome, null); // we stop if null
        handleMetadataForDetectedEvent(detectedEvent);
        handleContextForDetectedEvent(detectedEvent);

        // 4. Handle the notification of solution event as a control message (distinct from app notification specification)
        handleMessagingOfSolutionEvent(detectedEvent);

        // 5. Execute event handling
        final List<BaseAction> actions = handleEvent(detectedEvent);

        // 6. Execute actions
        executeActions(detectedEvent, actions);
        return new PipelineResult(ppOutcome, detectedEvent);
    }

    private BaseMetadata extractMetadata(ObservableTestbedEvent observableTestbedEvent) {
        // Only for workload.requests...
        if (observableTestbedEvent instanceof RequestEvent) {
            final RequestEvent asRequestEvent = (RequestEvent) observableTestbedEvent;
            final int appId = asRequestEvent.getAppId();
            final App app = obtainApp(appId);
            final MetadataExtractor<RequestEvent> metadataExtractorForType = app.getMetadataExtractorForType(MetadataType.REQUEST_METADATA);
            if (metadataExtractorForType != null) {
                final BaseMetadata metadata = metadataExtractorForType.extractMetadata(asRequestEvent);
                metadataAPI.addMetadata(metadata);
                return metadata;
            }
        }
        // ... and content received
        else if (observableTestbedEvent instanceof ContentReceivedEvent) {
            final ContentReceivedEvent asContentReceivedEvent = (ContentReceivedEvent) observableTestbedEvent;
            final int appId = asContentReceivedEvent.getAppId();
            final App app = obtainApp(appId);
            final MetadataExtractor<ContentReceivedEvent> metadataExtractorForType = app.getMetadataExtractorForType(MetadataType.CONTENT_METADATA);
            if (metadataExtractorForType != null) {
                final BaseMetadata metadata = metadataExtractorForType.extractMetadata(asContentReceivedEvent);
                //                printInfo(String.format("Metadata extracted for ContentTransmitted %s for app %s",
                //                        asContentReceivedEvent.getReferredContent(),
                //                        appId));
                metadataAPI.addMetadata(metadata);
                return metadata;
            }
        }
        return null;
    }

    private PPOutcome preprocessData(ObservableTestbedEvent observableTestbedEvent, BaseMetadata metadata,
                                     double currentTime) {
        PPOutcome ppOutcome = null;
        if (observableTestbedEvent instanceof HandoverEvent) {
            throw new RuntimeException("Not implemented yet. Mobility will be appId=0");
        }

        final int appId = observableTestbedEvent.getAppId();
        final App app = obtainApp(appId);
        final PreProcessor preProcessor = app.getPreProcessorForSimulationEventType(observableTestbedEvent.getClass());
        if (preProcessor != null) {
            ppOutcome = preProcessor.execute(app, observableTestbedEvent, metadata, currentTime);
        }
        return ppOutcome;
    }

    private SolutionEvent detectEvent(Host localHost, PPOutcome ppOutcome) {
        // final App app = ppOutcome.getApp();
        final App app = obtainApp(ppOutcome.getAppId());
        final BaseEventDetector eventDetector = app.getEventDetectorForPreProcessorOutcomeType(ppOutcome.getClass());

        SolutionEvent detectedEvent = null;
        if (eventDetector != null) {
            detectedEvent = eventDetector.execute(app.getContextAPI(), localHost, ppOutcome);
            if (detectedEvent == null) {
                printInfo("No event detected");
            } else {
                app.addContextData(detectedEvent);
            }
        }
        return detectedEvent;
    }

    private void handleMetadataForDetectedEvent(SolutionEvent event) {
        App app = obtainApp(event.getAppId());
        final var metadataExtractorForEvent = app.getMetadataExtractorForType(MetadataType.INTERNAL_EVENT_METADATA);
        if (metadataExtractorForEvent != null) {
            final var eventMetadata = (SolutionEventMetadata) metadataExtractorForEvent.extractMetadata(event);
            event.setMetadata(eventMetadata);
            metadataAPI.addMetadata(eventMetadata);
        }
    }

    private void handleContextForDetectedEvent(SolutionEvent detectedEvent) {
        App app = obtainApp(detectedEvent.getAppId());
        app.addContextData(detectedEvent);
    }

    private void handleMessagingOfSolutionEvent(SolutionEvent detectedEvent) {
        if (!(detectedEvent instanceof ApplicationEvent)) {
            if (localHost instanceof EdgeServer) {
                EdgeServer edgeServer = (EdgeServer) localHost;
                final Host cloud = edgeServer.getParentUnit();
                MessageData messageData = new SystemMessageData(KILOBYTE, detectedEvent, detectedEvent.getTimestamp());
                SimulationEvent controlMessageEvent = new PushMessageRequestEvent(detectedEvent.getAppId(),
                        null, edgeServer, detectedEvent.getTimestamp(), messageData, edgeServer,
                        cloud);
                Simulation.getInstance().postEvent(controlMessageEvent);
            }
        }
    }

    private List<BaseAction> handleEvent(SolutionEvent detectedEvent) {
        App app = obtainApp(detectedEvent.getAppId());
        return app.getActionsForEventForEventType(detectedEvent.getClass());
    }

    private void executeActions(SolutionEvent solutionEvent, List<BaseAction> actions) {
        final App app = obtainApp(solutionEvent.getAppId());

        for (BaseAction action : actions) {
            final BasePolicy policy = app.getPolicyForActionType(action.getClass());
            int tries = 0;
            boolean success = false;
            while (true) {
                if (tries == 2) break;
                try {
                    action.execute(app, localHost, solutionEvent, policy);
                    success = true;
                    break;
                } catch (SolutionException e) {
                    printWarning(e.getDescription() + ". Retrying... ");
                    // The only thing we can do is to try to free some storage to make space for files. I can't see at
                    // the moment another corrective measure in our system.
                    app.makeSpace(solutionEvent.getDetectingHost(), solutionEvent.getReferredContent());
                } finally {
                    tries++;
                }
            }
            if (!success) {
                printWarning(String.format("Aborting execution of actions due to failing action %s", action.getName()));
                break;
            }
            //            else {
            //                printInfo(String.format("Action %s executed in %s trials", action.getName(), tries));
            //            }
        }
    }

    public App obtainApp(int appId) {
        return apps.get(appId);
    }

    public List<App> getApps() {
        return new ArrayList<>(apps.values());
    }
}
