package core.distributed;

import core.Solution;
import core.app.App;
import core.control.PushMessageRequestEvent;
import core.control.StorageChangeEventMessageData;
import core.control.StorageConfigurationEventMessageData;
import core.processingchain.PipelineResult;
import core.processingchain.actions.exceptions.AppOutOfStorageException;
import core.processingchain.actions.exceptions.HostOutOfStorageException;
import core.processingchain.events.SolutionEvent;
import core.processingchain.events.pool.RequestCompletedEvent;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.entities.Content;
import edu.ucc.entities.File;
import edu.ucc.entities.IoTData;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;

import java.util.List;

import static edu.ucc.utils.Logging.*;
import static edu.ucc.utils.TransmissionUtils.*;

public class EdgeSolution extends LocalSolution {
    private EdgeServer edgeServer;

    public EdgeSolution(Solution solution, EdgeServer edgeServer) {
        super(solution, edgeServer);
        this.edgeServer = edgeServer;
    }

    public EdgeServer getEdgeServer() {
        return edgeServer;
    }

    @Override
    public void onContentReceptionCompleted(ContentReceivedEvent contentReceivedEvent) {
        final double timestamp = contentReceivedEvent.getTimestamp();
        final RequestEvent originalRequestEvent = contentReceivedEvent.getOriginalRequestEvent();
        final Content content = contentReceivedEvent.getReferredContent();
        final int direction = contentReceivedEvent.getEventDirection();

        App app = pipeline.obtainApp(contentReceivedEvent.getAppId());
        if (app == null) {
            printWarning(String.format("App %s is not deployed in this ES %s", contentReceivedEvent.getAppId(), localHost.getId()));
            return;
        }
        switch (direction) {
            case UPWARDS_DIRECTION:
                if (originalRequestEvent instanceof PullRequestEvent) {
                    // investigate on popularity as a result of users interacting with architecture.
                    // But we only do it if initiator is the cloud. Why? Because if requester is a UE, the content
                    // will travel down again.
                    handleOpportunisticStorage(app, edgeServer, content, timestamp);
                    app.updateContentPopularity(edgeServer, timestamp);
                    if (((PullRequestEvent) originalRequestEvent).getPullInitiatorHost() instanceof CloudUnit) {
                        addContextDataForAppAndRequest(originalRequestEvent, app);
                        executeProcessingChain(app, originalRequestEvent);
                    }
                } else if (originalRequestEvent instanceof PushRequestEvent) {
                    // Branch for IoT data processing (custom events detection)
                    executeProcessingChain(app, contentReceivedEvent);
                } else {
                    printWarning(String.format("Unsupported type of the event received in the uplink %s",
                            contentReceivedEvent.toString()));
                }
                break;
            case DOWNWARDS_DIRECTION:
                if (originalRequestEvent instanceof PullRequestEvent) {
                    addContextDataForAppAndRequest(originalRequestEvent, app);
                    handleOpportunisticStorage(app, edgeServer, content, timestamp);
                    // HERE: Try to explore on popularity of other files
                    app.updateContentPopularity(edgeServer, timestamp);
                    executeProcessingChain(app, originalRequestEvent);
                    if (!(content instanceof File || content instanceof IoTData)) {
                        printWarning(
                                String.format("Content different than IoTData and File (%s) was received at ES %s in downlink. Verify this",
                                        content.getClass().getSimpleName(), edgeServer.getId()));
                    }
                } else if (originalRequestEvent instanceof PushRequestEvent) {
                    if (content instanceof StorageConfigurationEventMessageData) {
                        handleStorageConfigurationChange((StorageConfigurationEventMessageData) content);
                    } else {
                        final PushRequestEvent asPushRequestEvent = (PushRequestEvent) originalRequestEvent;
                        if (asPushRequestEvent.getPushReason() == PushReason.APP_REQUEST) {
                            printDebug(String.format("Storing prefetched content %s", content.getDescription()));
                            try {
                                app.storeContentFromDirectAppRequest(edgeServer, content, timestamp);
                                checkAndNotifyChangesInStorage(app, timestamp + delayFactor);
                            } catch (AppOutOfStorageException | HostOutOfStorageException e) {
                                printError(e.getMessage());
                            }
                        }
                    }

                }
                break;
            case SAME_LEVEL_DIRECTION:
                // This is for pushed content from other EdgeServers. We might store it depending on its type:
                if (originalRequestEvent instanceof PushRequestEvent) {

                    final PushRequestEvent asPushRequestEvent = (PushRequestEvent) originalRequestEvent;
                    if (asPushRequestEvent.getPushReason() == PushReason.POPULARITY) {
                        if (content instanceof File || content instanceof IoTData) {
                            printDebug(String.format("Storing content %s", content.getDescription()));
                            try {
                                app.storeExternalPopularContent(edgeServer, content, timestamp);
                                checkAndNotifyChangesInStorage(app, timestamp + delayFactor);
                            } catch (AppOutOfStorageException | HostOutOfStorageException e) {
                                printError(e.getMessage());
                            }
                        }
                    } else if (asPushRequestEvent.getPushReason() == PushReason.NORMAL_TRAFFIC) {
                        executeProcessingChain(app, contentReceivedEvent);
                    } else {
                        //                    printInfo(String.format("At ES %s received notification of %s (app %s) @ %s",
                        //                            edgeServer.getId(),
                        //                            content.getDescription(),
                        //                            contentReceivedEvent.getAppId(),
                        //                            timestamp
                        //                    ));
                    }
                }

                break;
        }
    }

    private void handleStorageConfigurationChange(StorageConfigurationEventMessageData storageConfigurationEventMessageData) {
        final App app = pipeline.obtainApp(storageConfigurationEventMessageData.getAppId());
        app.setAvailableSpace(storageConfigurationEventMessageData.getNewAvailableSpace());
        app.resetStorageBalance();
    }

    @Override
    public double executePendingEventsForApp(int appId) {
        double timestamp = super.executePendingEventsForApp(appId);
        App app = pipeline.obtainApp(appId);
        checkAndNotifyChangesInStorage(app, timestamp);
        return 0;
    }

    private void checkAndNotifyChangesInStorage(App app, double timestampForNotification) {
        if (app.isOverStorageBalance()) {
            boolean isSpaceReduced = app.getStorageBalance() < 0;
            StorageChangeEventMessageData data = new StorageChangeEventMessageData(localHost, app.getAppId(), app.getStorageBalance(), isSpaceReduced, timestampForNotification);
            final Host cloud = localHost.getParentUnit();
            PushMessageRequestEvent event = new PushMessageRequestEvent(app.getAppId(), null, localHost, timestampForNotification, data, localHost, cloud);
            Simulation.getInstance().postEvent(event);
            //            if (app.getStorageAllocationType() == StorageAllocationType.FIXED_AND_FULLY_DISTRIBUTED || app.getStorageAllocationType() == StorageAllocationType.FIXED_AND_PARTIALLY_DISTRIBUTED) {
            //                // reset if decisions are taken locally
            //                // Notice that if you don't reset it is possible you transmit wrong used space.
            //                app.resetStorageBalance();
            //            }
            app.resetStorageBalance();
        }
    }

    private void addContextDataForAppAndRequest(RequestEvent originalRequestEvent, App app) {
        app.getContextAPI().addContextData(new RequestCompletedEvent(app.getAppId(), edgeServer, originalRequestEvent));
    }

    private void handleOpportunisticStorage(App app, EdgeServer edgeServer, Content content, double timestamp) {
        if (app.isOpportunisticStorage()) {
            printInfo(String.format("App uses opportunistic storage. Trying to store %s", content.getDescription()));
            try {
                app.storeOpportunisticContent(edgeServer, content, timestamp);
                checkAndNotifyChangesInStorage(app, timestamp + delayFactor);
            } catch (AppOutOfStorageException | HostOutOfStorageException e) {
                printWarning(e.getMessage());
            }
        }
    }

    @Override
    public void onPullRequestServedLocally(PullRequestEvent pullRequestEvent) {
        // final RequestEvent testbedRequestEvent = contentReceivedEvent.getOriginalRequestEvent();
        //        printInfo(String.format("Solution working at %s %s (content served locally)", edgeServer.getTypeAsString(), edgeServer.getId()));
        App app = pipeline.obtainApp(pullRequestEvent.getAppId());
        addContextDataForAppAndRequest(pullRequestEvent, app);
        executeProcessingChain(app, pullRequestEvent);
    }

    private void executeProcessingChain(App app, ObservableTestbedEvent eventFromTestbed) {
        PipelineResult result = pipeline.executeProcessingChain(edgeServer, eventFromTestbed);
        if (result != null) {
            final SolutionEvent detectedEvent = result.getDetectedEvent();
            if (detectedEvent != null) {
                checkAndNotifyChangesInStorage(app, detectedEvent.getTimestamp() + delayFactor);
            }
        }
    }

    @Override
    public void onUserEquipmentHandover(HandoverEvent handoverEvent) {

    }

    @Override
    public List<SimulationEvent> createControlEvents() {
        return null;
    }
}
