package core.distributed;

import core.app.App;
import core.mobility.MobilityAnalyzer;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.PrefetchRequestEvent;
import edu.ucc.core.events.simulationevents.PushFileRequestEvent;
import edu.ucc.core.events.simulationevents.PushReason;
import edu.ucc.entities.File;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

public class ContentPrefetcher {
    private final CloudSolution cloudSolution;

    public ContentPrefetcher(CloudSolution cloudSolution) {
        this.cloudSolution = cloudSolution;
    }

    void performMobilityAwarePrefetching(PrefetchRequestEvent prefetchRequestEvent) {
                double prefetchTimestamp = prefetchRequestEvent.getPrefetchTimestamp();

        final int[] destinationUEs = prefetchRequestEvent.getDestinationHostsIds();
        var predictedEdgeServers = new HashSet<Integer>();
        final MobilityAnalyzer mobilityAnalyzer = cloudSolution.mobilityAnalyzer;
        for (int ueId : destinationUEs) {
            final int edgeServerId = mobilityAnalyzer.predictMobilityForUserEquipment(ueId, prefetchTimestamp);
            if (edgeServerId != -1) {
                predictedEdgeServers.add(edgeServerId);
            }
        }

        int appId = prefetchRequestEvent.getAppId();
        final App app = cloudSolution.getApp(appId);
        final Iterator<Integer> iterator = predictedEdgeServers.iterator();
        int i = 0;
        final double internalProcessingDelay = 0.0000001;
        final File fileToPrefetch = prefetchRequestEvent.getFileToPrefetch();
        List<PushFileRequestEvent> pushRequestEvents = new ArrayList<>();
        CloudUnit cloud = (CloudUnit) cloudSolution.localHost;

        while (iterator.hasNext()) {
            double eventTimestamp = prefetchTimestamp + (++i * internalProcessingDelay);
            final int esId = iterator.next();
            if (app.isDeployedAtEdgeServer(esId)) {
                EdgeServer es = (EdgeServer) cloud.getChildUnit(esId);
                PushFileRequestEvent requestEvent = new PushFileRequestEvent(appId, null, cloud,
                        eventTimestamp, fileToPrefetch, cloud, es, PushReason.APP_REQUEST);
                pushRequestEvents.add(requestEvent);
            }
        }

        for (var event : pushRequestEvents) {
            Simulation.getInstance().postEvent(event);
        }
    }

    public void performDirectEdgeServerPrefetch(PrefetchRequestEvent prefetchRequestEvent) {
        CloudUnit cloud = (CloudUnit) cloudSolution.localHost;
        App app = cloudSolution.getApp(prefetchRequestEvent.getAppId());
        var edgeServers = new HashSet<EdgeServer>();
        for (var esId : prefetchRequestEvent.getDestinationHostsIds()) {
            final Host child = cloud.getChildUnit(esId);
            if (child != null) {
                EdgeServer es = (EdgeServer) child;
                if (app.isDeployedAtEdgeServer(esId)) {
                    edgeServers.add(es);
                }
            }
        }

        List<PushFileRequestEvent> pushRequestEvents = new ArrayList<>();
        final File fileToPrefetch = prefetchRequestEvent.getFileToPrefetch();
        final var iterator = edgeServers.iterator();
        final double internalProcessingDelay = 0.0000001;
        final double prefetchTimestamp = prefetchRequestEvent.getPrefetchTimestamp();
        int i = 0;
        while (iterator.hasNext()) {
            double eventTimestamp = prefetchTimestamp + (++i * internalProcessingDelay);
            EdgeServer es = iterator.next();
            final PushFileRequestEvent pushRequestEvent = new PushFileRequestEvent(app.getAppId(), null,
                    cloud, eventTimestamp, fileToPrefetch, cloud, es, PushReason.APP_REQUEST);
            pushRequestEvents.add(pushRequestEvent);
        }

        for (var event : pushRequestEvents) {
            Simulation.getInstance().postEvent(event);
        }
    }
}
