package core.distributed;

import core.Solution;
import core.app.App;
import core.control.PushMessageRequestEvent;
import core.control.StorageChangeEventMessageData;
import core.control.StorageConfigurationEventMessageData;
import core.control.SystemMessageData;
import core.mobility.MobilityAnalyzer;
import core.processingchain.events.pool.RequestCompletedEvent;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.entities.Content;
import edu.ucc.entities.IoTData;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.Host;
import edu.ucc.utils.Logging;

import java.util.List;

import static edu.ucc.utils.Logging.printInfo;

public class CloudSolution extends LocalSolution {
    final MobilityAnalyzer mobilityAnalyzer;
    final ContentPrefetcher contentPrefetcher;

    public CloudSolution(Solution solution, CloudUnit localHost) {
        super(solution, localHost);
        this.mobilityAnalyzer = new MobilityAnalyzer();
        this.contentPrefetcher = new ContentPrefetcher(this);
    }

    @Override
    public void onContentReceptionCompleted(ContentReceivedEvent contentReceivedEvent) {
        final Content receivedContent = contentReceivedEvent.getReferredContent();
        if (receivedContent instanceof SystemMessageData) {
            // we don't process the events that refer to notifications inside the cloud
            printInfo(String.format("At cloud received %s @%s", receivedContent.getDescription(), contentReceivedEvent.getTimestamp()));
        } else if (receivedContent instanceof StorageChangeEventMessageData) {
            handleStorageChange((StorageChangeEventMessageData) receivedContent);
        } else if (receivedContent instanceof IoTData) {
            handleUploadedIoTData(contentReceivedEvent);
        }
    }

    private void handleUploadedIoTData(ContentReceivedEvent contentReceivedEvent) {
        App app = pipeline.obtainApp(contentReceivedEvent.getAppId());
        final RequestEvent originalRequestEvent = contentReceivedEvent.getOriginalRequestEvent();
        addContextDataForAppAndRequest(originalRequestEvent, app);
        executeProcessingChain(app, contentReceivedEvent);
    }

    private void addContextDataForAppAndRequest(RequestEvent originalRequestEvent, App app) {
        app.getContextAPI().addContextData(new RequestCompletedEvent(app.getAppId(), localHost, originalRequestEvent));
    }

    private void executeProcessingChain(App app, ObservableTestbedEvent eventFromTestbed) {
        pipeline.executeProcessingChain(localHost, eventFromTestbed);
        //        PipelineResult result = pipeline.executeProcessingChain(localHost, eventFromTestbed);
        //        if (result != null) {
        //            final SolutionEvent detectedEvent = result.getDetectedEvent();
        //            if (detectedEvent != null) {
        //                checkAndNotifyChangesInStorage(app, detectedEvent.getTimestamp() + delayFactor);
        //            }
        //        }
    }

    private void handleStorageChange(StorageChangeEventMessageData storageChangeData) {
        // When we process notifications of changes in storage:
        // 1. If the app has a global storage allocation, then we do what we have in our slides (app notifies servers of new storage...).
        // 2. If the app has a per-server way to deal, then pretty much the message is ignored, as the decision to allow content (or not) can be made locally.
        final App app = pipeline.obtainApp(storageChangeData.getAppId());
        switch (app.getStorageAllocationType()) {
            case FLEXIBLE_AND_FULLY_DISTRIBUTED:
            case FLEXIBLE_AND_PARTIALLY_DISTRIBUTED:
                updateSpaceForApp(app, storageChangeData);
                notifyStorageChange(app, storageChangeData.getTimestamp());
                break;
            case FIXED_AND_FULLY_DISTRIBUTED:
                System.out.println(String.format("Change in a fixed fully distributed allocation app %s: %s. Nothing to handle as the storage is always evaluated locally.", app.getAppId(), storageChangeData.getDescription()));
                updateSpaceForApp(app, storageChangeData);
                break;
            // System.out.println(String.format("Change in a flexible and partially distributed allocation app %s: %s", app.getAppId(), storageChangeData.getDescription()));
            case FIXED_AND_PARTIALLY_DISTRIBUTED:
                System.out.println(String.format("Change in a fixed and partially distributed allocation app %s: %s. Nothing to handle as the storage is always evaluated locally.", app.getAppId(), storageChangeData.getDescription()));
                updateSpaceForApp(app, storageChangeData);
                break;
        }
    }

    private void notifyStorageChange(App app, double currentTimestamp) {
        long newAvailableSpace = app.getAvailableSpace();
        final double delay = 0.0001;
        double timestamp = currentTimestamp + delay;
        final List<Host> edgeServers = ((CloudUnit) localHost).getChildrenUnits();
        for (Host edgeServer : edgeServers) {
            StorageConfigurationEventMessageData storageConfigurationEventMessageData = new StorageConfigurationEventMessageData(app.getAppId(), newAvailableSpace, timestamp);
            PushMessageRequestEvent requestEvent = new PushMessageRequestEvent(app.getAppId(), null, localHost, timestamp, storageConfigurationEventMessageData, localHost, edgeServer);
            Simulation.getInstance().postEvent(requestEvent);
            timestamp += delay;
        }
    }

    private void updateSpaceForApp(App app, StorageChangeEventMessageData storageChangeData) {
        long currentAvailableSpace = app.getAvailableSpace();
        long availableSpaceChange = storageChangeData.getDataVolume();
        long newAvailableSpace = currentAvailableSpace + availableSpaceChange;
        app.setAvailableSpace(newAvailableSpace);
    }


    @Override
    public void onPullRequestServedLocally(PullRequestEvent pullRequestEvent) {

    }

    @Override
    public void onUserEquipmentHandover(HandoverEvent handoverEvent) {
        mobilityAnalyzer.onReportedMovement(localHost, handoverEvent);
    }

    @Override
    public List<SimulationEvent> createControlEvents() {
        return null;
    }

    public void onPrefetchRequestReceived(Host cloud, PrefetchRequestEvent prefetchRequestEvent) {
        Logging.logRequest(prefetchRequestEvent);
        if (prefetchRequestEvent.isTargetingEdgeServers()) {
            contentPrefetcher.performDirectEdgeServerPrefetch(prefetchRequestEvent);
        } else {
            contentPrefetcher.performMobilityAwarePrefetching(prefetchRequestEvent);
        }
    }

    public void registerMobilityChange(int ueId, int sourceBSId, int targetBSId, double timestamp) {
        mobilityAnalyzer.registerMobilityChange((CloudUnit) localHost, ueId, sourceBSId, targetBSId, timestamp);
    }
}
