package core;

import com.rits.cloning.Cloner;
import core.app.App;
import core.app.StorageAllocationType;
import core.control.AppPendingProcessingEvent;
import core.control.ContentManagementEvent;
import core.distributed.CloudSolution;
import core.distributed.EdgeSolution;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.*;
import edu.ucc.core.transport.chunks.ContentChunk;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.NetworkArchitecture;
import edu.ucc.testbedinterface.ObservableTestbedEventListener;

import java.util.*;
import java.util.stream.Collectors;

import static core.SolutionConstants.IMMEDIATE_PROCESSING;
import static core.SolutionConstants.WATCHDOG_INTERVAL;
import static edu.ucc.utils.Logging.*;


public class Solution implements ObservableTestbedEventListener {
    private final CloudSolution cloudSolution;
    private final Map<Integer, EdgeSolution> solutionAtEdgeServers;
    private final NetworkArchitecture networkArchitecture;

    public Solution(NetworkArchitecture networkArchitecture) {
        solutionAtEdgeServers = new HashMap<>();
        cloudSolution = new CloudSolution(this, networkArchitecture.getCloudRoot());
        final List<EdgeServer> edgeServers = networkArchitecture.getEdgeServers();
        for (EdgeServer edgeServer : edgeServers) {
            EdgeSolution localSolution = new EdgeSolution(this, edgeServer);
            solutionAtEdgeServers.put(edgeServer.getId(), localSolution);
        }
        this.networkArchitecture = networkArchitecture;
    }

    public NetworkArchitecture getNetworkArchitecture() {
        return networkArchitecture;
    }

    public void registerApp(App app) {
        final Cloner cloner = new Cloner();
        final App cloudApp = cloner.deepClone(app);
        cloudSolution.registerApp(cloudApp);

        validateApp(app);
        switch (app.getStorageAllocationType()) {
            case FLEXIBLE_AND_FULLY_DISTRIBUTED:
            case FIXED_AND_FULLY_DISTRIBUTED:
                deployForAllServers(app);
                break;
            case FLEXIBLE_AND_PARTIALLY_DISTRIBUTED:
            case FIXED_AND_PARTIALLY_DISTRIBUTED:
                deployForSpecificServers(app);
                break;
        }
    }

    public void deployForSpecificServers(App app) {
        if (app.isStorageFixed()) {
            deployWithFixedStorage(app);
        } else {
            final int[] serversDeployed = app.getServersDeployed();
            for (int serverId : serversDeployed) {
                final List<EdgeSolution> edgeSolutions = solutionAtEdgeServers.values().stream().filter(ses -> ses.getEdgeServer().getId() == serverId).collect(Collectors.toList());
                if (edgeSolutions.isEmpty()) {
                    throw new RuntimeException(String.format("Can't deploy app %s, edge server with id %s does not exist in the network architecture", app.getAppId(), serverId));
                }
                final EdgeSolution edgeSolution = edgeSolutions.get(0);
                final Cloner cloner = new Cloner();
                final App edgeServerApp = cloner.deepClone(app);

                edgeSolution.registerApp(edgeServerApp);
            }
        }
    }

    public void deployForAllServers(App app) {
        if (app.isStorageFixed()) {
            deployWithFixedStorage(app);
        } else {
            for (EdgeSolution edgeSolution : solutionAtEdgeServers.values()) {
                final Cloner cloner = new Cloner();
                final App edgeServerApp = cloner.deepClone(app);
                edgeSolution.registerApp(edgeServerApp);
            }
        }
    }

    private void deployWithFixedStorage(App app) {
        final int[] serversDeployed = app.isDeployedAtAllEdgeServers() ? networkArchitecture.getEdgeServers().stream().mapToInt(Host::getId).toArray() : app.getServersDeployed();
        for (int i = 0; i < serversDeployed.length; i++) {
            int serverId = serversDeployed[i];
            final List<EdgeSolution> edgeSolutions = solutionAtEdgeServers.values().stream().filter(ses -> ses.getEdgeServer().getId() == serverId).collect(Collectors.toList());
            if (edgeSolutions.isEmpty()) {
                throw new RuntimeException(String.format("Can't deploy app %s, edge server with id %s does not exist in the network architecture", app.getAppId(), serverId));
            }

            final EdgeSolution edgeSolution = edgeSolutions.get(0);
            final Cloner cloner = new Cloner();
            final App edgeServerApp = cloner.deepClone(app);
            final long storageThisServer = app.getStoragePerServer()[i];

            edgeServerApp.setTotalSpace(storageThisServer);
            edgeServerApp.setAvailableSpace(storageThisServer);
            edgeSolution.registerApp(edgeServerApp);
        }
    }

    private void validateApp(App app) {
        if (app.getStorageAllocationType() == StorageAllocationType.FIXED_AND_FULLY_DISTRIBUTED) {
            final long storageAcrossServers = Arrays.stream(app.getStoragePerServer()).sum();
            if (storageAcrossServers < app.getTotalSpace()) {
                printWarning("Warning app %s is using %s bytes across servers");
            } else if (storageAcrossServers > app.getTotalSpace()) {
                throw new RuntimeException(String.format("App %s requested %s bytes across servers, which is more than the total specified storage %s", app.getAppId(), storageAcrossServers, app.getTotalSpace()));
            }

            if (app.getStoragePerServer().length != networkArchitecture.getEdgeServers().size()) {
                throw new RuntimeException(String.format("The specified list of storage per edge server does not match the number of servers in the architecture (%s specified servers but %s in the architecture)", app.getStoragePerServer().length, networkArchitecture.getEdgeServers().size()));
            }
        }
    }

    @Override
    public void onContentReceptionCompleted(Host host, ContentReceivedEvent contentReceivedEvent) {
        logContentReceived(contentReceivedEvent);
        if (isAppDeployedHere(host, contentReceivedEvent.getAppId())) {
            if (host instanceof EdgeServer) {
                final EdgeSolution edgeSolution = solutionAtEdgeServers.get(host.getId());
                edgeSolution.onContentReceptionCompleted(contentReceivedEvent);
            } else if (host instanceof CloudUnit) {
                cloudSolution.onContentReceptionCompleted(contentReceivedEvent);
            }
        }
    }

    @Override
    public void onPullRequestServedLocally(EdgeServer edgeServer, PullRequestEvent pullRequestEvent) {
        solutionAtEdgeServers.get(edgeServer.getId()).onPullRequestServedLocally(pullRequestEvent);
    }

    @Override
    public void onUserEquipmentHandover(Host host, HandoverEvent handoverEvent) {
        // i copied this to another layer. I believe the simulator should handle this.
        //        if (handoverEvent.isIntraHandover()) {
        //            host.getFileSystem().updateURIForMovingUEData(handoverEvent.getMovingUE());
        //        }
        cloudSolution.onUserEquipmentHandover(handoverEvent);
    }

    @Override
    public void onPrefetchRequestReceived(Host host, PrefetchRequestEvent prefetchRequestEvent) {
        cloudSolution.onPrefetchRequestReceived(host, prefetchRequestEvent);
    }

    private void scheduleContentLifeCycleManagerEvent(double currentTime) {
        if (currentTime < Simulation.getInstance().getSimulationLength()) {
            double eventTime = currentTime + WATCHDOG_INTERVAL;
            Simulation.getInstance().postEvent(new ContentManagementEvent(this, eventTime));
        }
    }

    /**
     * This is for setting an event to do some self-control (delete non popular files, etc.).
     */
    public void handleContentLifeCycle() {
        printDebug("Handling ContentManagementEvent");
        scheduleContentLifeCycleManagerEvent(Simulation.getInstance().getCurrentTime());
    }

    @Override
    public List<SimulationEvent> createInitialControlEvents() {
        // return new ContentManagementEvent(this, WATCHDOG_INTERVAL);
        final ArrayList<SimulationEvent> initialControlEvents = new ArrayList<>();
        // As the apps are also registered in cloud, we can take the list from there.
        var apps = cloudSolution.getApps();
        for (App app : apps) {
            if (app.getProcessingInterval() != IMMEDIATE_PROCESSING) {
                // new ContentManagementEvent()
                final var event = new AppPendingProcessingEvent(this, app, app.getProcessingInterval());
                initialControlEvents.add(event);
            }
        }
        return initialControlEvents;
    }

    @Override
    public boolean isAppDeployedHere(Host host, int appId) {
        if (host instanceof CloudUnit) {
            return true;
        }
        final App appAtCloud = getAppAtCloud(appId);
        return appAtCloud.isDeployedAtEdgeServer(host.getId());
    }

    @Override
    public Host findDestinationForChunkAtServerWithoutApp(ContentChunk chunk, EdgeServer edgeServer, int appId) {
        final App app = getAppAtCloud(appId);

        switch (app.getDataRelayPolicy()) {
            case RELAY_TO_CLOUD:
                return edgeServer.getParentUnit();
            case RELAY_TO_CLOSEST_HOST:
                final Object[] closestSibling = getFastestSibling(edgeServer, chunk.getChunkSize(), appId);
                if (closestSibling[1] == null) {
                    return edgeServer.getParentUnit();
                } else {
                    final double txTimeToParent = edgeServer.calculateTxTimeToParent(chunk.getChunkSize());
                    final double txTimeToSibling = (double) closestSibling[0];
                    if (txTimeToParent < txTimeToSibling) {
                        return edgeServer.getParentUnit();
                    } else {
                        return (EdgeServer) closestSibling[1];
                    }
                }
            case IGNORE:
                return null;
            default:
                throw new IllegalArgumentException("Unknown data relay policy");
        }
    }

    private Object[] getFastestSibling(EdgeServer edgeServer, int chunkSize, int appId) {
        double fastestTime = Double.MAX_VALUE;
        EdgeServer fastestServer = null;
        for (Host siblingHost : edgeServer.getSiblingUnits()) {
            final EdgeServer siblingServer = (EdgeServer) siblingHost;
            if (isAppDeployedHere(siblingHost, appId)) {
                double txTimeToSibling = edgeServer.calculateTxTimeToSibling(chunkSize, siblingServer);
                if (txTimeToSibling < fastestTime) {
                    fastestTime = txTimeToSibling;
                    fastestServer = siblingServer;
                }
            }
        }
        return new Object[]{fastestTime, fastestServer};
    }

    public App getAppAtCloud(int appId) {
        return cloudSolution.getApp(appId);
    }

    public App getAppAtEdgeServer(int appId, int edgeServerId) {
        final EdgeSolution edgeSolution = solutionAtEdgeServers.get(edgeServerId);
        return edgeSolution.getApp(appId);
    }

    public void executePendingDataProcessingForApp(App app) {
        cloudSolution.executePendingEventsForApp(app.getAppId());
        for (EdgeSolution edgeSolution : solutionAtEdgeServers.values()) {
            if (isAppDeployedHere(edgeSolution.getLocalHost(), app.getAppId()))
                edgeSolution.executePendingEventsForApp(app.getAppId());
        }
    }

    /**
     * This temp method allows to inject some mobility information to the system.
     *
     * @param ueId       The id of the UE to move
     * @param sourceBSId the id of the source BS
     * @param targetBSId the if of the target BS
     * @param timestamp  The timestamp of the movement.
     */
    @Deprecated
    public void registerMobilityChange(int ueId, int sourceBSId, int targetBSId, double timestamp) {
        cloudSolution.registerMobilityChange(ueId, sourceBSId, targetBSId, timestamp);
    }

    @Override
    public void registerInitialMobilityEvent(List<HandoverEvent> initialMobilityEvents) {
        for (var event : initialMobilityEvents) {
            cloudSolution.registerMobilityChange(event.getMovingUE().getId(),
                    -1, event.getTargetBS().getId(), event.getTimestamp());
        }
    }
}
