package integrationtests.tests;

import core.Solution;
import core.app.App;
import core.processingchain.actions.policies.pool.PushPolicy;
import core.processingchain.actions.pool.PushAction;
import core.processingchain.actions.pool.StoreAction;
import core.processingchain.events.detectors.pool.popularity.PopularityEventDetector;
import core.processingchain.events.pool.HighPopularityEvent;
import core.processingchain.events.pool.PopularityEvent;
import core.processingchain.metadata.MetadataType;
import core.processingchain.metadata.extractor.MetadataExtractor;
import core.processingchain.metadata.pool.BaseMetadata;
import core.processingchain.metadata.pool.PopularityEventMetadata;
import core.processingchain.preprocessors.pool.PPOutcomePullRequest;
import edu.ucc.core.Simulation;
import edu.ucc.entities.Content;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.StorageReason;
import edu.ucc.utils.Logging;
import edu.ucc.utils.TimeConstants;
import integrationtests.tests.sysloaders.CustomWorkloadSystemLoader;
import integrationtests.tests.sysloaders.ICustomWorkloadGenerator;
import integrationtests.tests.sysloaders.WorkloadGeneratorFactory;
import loader.AbstractSystemLoader;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import workload.mobility.MobilityParameters;
import workload.requests.ContentRequestParameters;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static edu.ucc.utils.FileConstants.MEGABYTE;
import static utils.ScenarioConstants.TRAJECTORY_TIME;
import static utils.Utils.createGenericArchitectureParameters;
import static utils.Utils.createGenericMobilityParameters;

class ScenarioHighThenLowPopularityDownlinkTest {
    private Solution solution;

    @Test
    void executeTestForScenarioPushFreeOnlyRequiredSpace() {
        // What should I see here:
        // We have an app that deletes only required not popular files when we need space [app.setFreeOnlyRequiredSpace(true);]
        // 1. A data File0.html is popular and stored.
        // 2. Then we see many files until the storage for the ES is depleted.
        // 3. Then File99.html is popular, at that moment we should see that the store action fails.
        // 4. Then space is made so that storing File99.html is successful.
        // 5. At the end we should see that :
        //     - File0, File99 are stored in ES0.
        //     - One file in the interval 1..8 is not gonna be there (due to making space).
        //     - File 0 and 99 should be also in ES1.
        //     - In other words, the whole distributed space is user for the app.

        App app = new App(1, "app1", "www.app1.com/", 10 * 10 * MEGABYTE);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));
        app.addMetadataExtractor(MetadataType.INTERNAL_EVENT_METADATA, new MetadataExtractor<PopularityEvent>() {
            @Override
            public BaseMetadata extractMetadata(PopularityEvent hpEvent) {
                boolean highPopularity = hpEvent instanceof HighPopularityEvent;
                return new PopularityEventMetadata(hpEvent.getAppId(), hpEvent.getTestbedEvent(), highPopularity);
            }
        });
        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        // Won't push to study what happens in a local edge
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);
        app.setFreeOnlyRequiredSpace(true);
        Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadHighThenLowPopularityDownlink());

        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();
        final EdgeServer es0 = edgeServers.get(0);
        final EdgeServer es1 = edgeServers.get(1);

        assert es0.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es1.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es0.getFileSystem().contentExists("www.app1.com/File99.html");

        final List<Content> contentForAppES0 = es0.getFileSystem().getContentForApp(1);
        final List<Content> contentForAppES1 = es1.getFileSystem().getContentForApp(1);
        System.out.println("In Es0");
        contentForAppES0.forEach(System.out::println);
        System.out.println("In Es1");
        contentForAppES1.forEach(System.out::println);
        final long spaceUsedByFiles = contentForAppES0.stream().mapToLong(Content::getSize).sum()
                + contentForAppES1.stream().mapToLong(Content::getSize).sum();
        final long distributedSpace = solution.getAppAtCloud(1).getTotalSpace();

        assert spaceUsedByFiles == distributedSpace;
    }

    @Test
    void executeTestForScenarioPushClearSpace() {
        // What should I see here:
        // We have an app that deletes only required not popular files when we need space [app.setFreeOnlyRequiredSpace(true);]
        // 1. A data File0.html is popular and stored.
        // 2. Then we see many files until the storage for the ES is depleted.
        // 3. Then File99.html is popular, at that moment we should see that the store action fails.
        // 4. Then we see that all not popular files are deleted to make space, so that storing File99.html is successful.
        // 5. At the end we should see that :
        //     - File0, File99 are stored in ES0 (storage reason local popularity)
        //     - and also pushed to ES1 (storage reason in ES1 is ExternalPopularity)

        App app = new App(1, "app1", "www.app1.com/", 10 * 10 * MEGABYTE);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));
        app.addMetadataExtractor(MetadataType.INTERNAL_EVENT_METADATA, new MetadataExtractor<PopularityEvent>() {
            @Override
            public BaseMetadata extractMetadata(PopularityEvent hpEvent) {
                boolean highPopularity = hpEvent instanceof HighPopularityEvent;
                return new PopularityEventMetadata(hpEvent.getAppId(), hpEvent.getTestbedEvent(), highPopularity);
            }
        });
        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        // Won't push to study what happens in a local edge
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);
        app.setFreeOnlyRequiredSpace(false);
        Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadHighThenLowPopularityDownlink());

        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();
        final EdgeServer es0 = edgeServers.get(0);
        final EdgeServer es1 = edgeServers.get(1);

        assert es0.getFileSystem().contentExists("www.app1.com/File0.html");
        assert !es0.getFileSystem().contentExists("www.app1.com/File1.html");
        assert !es0.getFileSystem().contentExists("www.app1.com/File2.html");
        assert !es0.getFileSystem().contentExists("www.app1.com/File3.html");
        assert !es0.getFileSystem().contentExists("www.app1.com/File4.html");
        assert !es0.getFileSystem().contentExists("www.app1.com/File5.html");
        assert !es0.getFileSystem().contentExists("www.app1.com/File6.html");
        assert !es0.getFileSystem().contentExists("www.app1.com/File7.html");
        assert !es0.getFileSystem().contentExists("www.app1.com/File8.html");
        assert es0.getFileSystem().contentExists("www.app1.com/File99.html");
        assert es1.getFileSystem().contentExists("www.app1.com/File0.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File1.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File2.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File3.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File4.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File5.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File6.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File7.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File8.html");
        assert es1.getFileSystem().contentExists("www.app1.com/File99.html");

        final List<Content> contentForAppES0 = es0.getFileSystem().getContentForApp(1);
        final List<Content> contentForAppES1 = es1.getFileSystem().getContentForApp(1);
        System.out.println("In Es0");
        contentForAppES0.forEach(System.out::println);
        System.out.println("In Es1");
        contentForAppES1.forEach(System.out::println);
        final long spaceUsedByFiles = contentForAppES0.stream().mapToLong(Content::getSize).sum()
                + contentForAppES1.stream().mapToLong(Content::getSize).sum();
        final long distributedSpace = solution.getAppAtCloud(1).getTotalSpace();

        assert spaceUsedByFiles < distributedSpace;

        StorageReason storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File0.html");
        Assertions.assertTrue(storageReasonEs0 == StorageReason.LOCAL_POPULARITY);
        storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File99.html");
        Assertions.assertTrue(storageReasonEs0 == StorageReason.LOCAL_POPULARITY);

        StorageReason storageReasonEs1 = es1.getFileSystem().getStorageReason("www.app1.com/File0.html");
        Assertions.assertTrue(storageReasonEs1 == StorageReason.EXTERNAL_POPULARITY);
        storageReasonEs1 = es1.getFileSystem().getStorageReason("www.app1.com/File99.html");
        Assertions.assertTrue(storageReasonEs1 == StorageReason.EXTERNAL_POPULARITY);

        final App appAtCloud = solution.getAppAtCloud(1);
        final App appAtEdgeServer0 = solution.getAppAtEdgeServer(1, 0);
        final App appAtEdgeServer1 = solution.getAppAtEdgeServer(1, 1);
        Assertions.assertEquals(appAtCloud.getAvailableSpace(), appAtEdgeServer0.getAvailableSpace());
        Assertions.assertEquals(appAtCloud.getAvailableSpace(), appAtEdgeServer1.getAvailableSpace());
    }

    @Test
    void executeTestForScenarioNoPush() {
        // What should I see here:
        // We have an app that deletes only required not popular files when we need space [app.setFreeOnlyRequiredSpace(true);]
        // 1. A data File0.html is popular and stored.
        // 2. Then we see many files until the storage for the ES is depleted.
        // 3. Then File99.html is popular, so it is stored.
        // 5. At the end we should see that :
        //     - File0, File99 are stored in ES0 as well as other files.
        //     - In other words, the whole distributed space is user for the app.

        App app = new App(1, "app1", "www.app1.com/", 10 * 10 * MEGABYTE);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));
        app.addMetadataExtractor(MetadataType.INTERNAL_EVENT_METADATA, new MetadataExtractor<PopularityEvent>() {
            @Override
            public BaseMetadata extractMetadata(PopularityEvent hpEvent) {
                boolean highPopularity = hpEvent instanceof HighPopularityEvent;
                return new PopularityEventMetadata(hpEvent.getAppId(), hpEvent.getTestbedEvent(), highPopularity);
            }
        });
        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.setOpportunisticStorage(true);
        app.setFreeOnlyRequiredSpace(true);
        Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadHighThenLowPopularityDownlink());

        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();
        final EdgeServer es0 = edgeServers.get(0);
        final EdgeServer es1 = edgeServers.get(1);

        assert es0.getFileSystem().contentExists("www.app1.com/File0.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es0.getFileSystem().contentExists("www.app1.com/File99.html");

        final List<Content> contentForAppES0 = es0.getFileSystem().getContentForApp(1);
        final List<Content> contentForAppES1 = es1.getFileSystem().getContentForApp(1);
        System.out.println("In Es0");
        contentForAppES0.forEach(System.out::println);
        System.out.println("In Es1");
        contentForAppES1.forEach(System.out::println);
        final long spaceUsedByFiles = contentForAppES0.stream().mapToLong(Content::getSize).sum()
                + contentForAppES1.stream().mapToLong(Content::getSize).sum();
        final long distributedSpace = solution.getAppAtCloud(1).getTotalSpace();
        assert spaceUsedByFiles == distributedSpace;
    }

    private Solution executeScenario(App app, ICustomWorkloadGenerator workloadGenerator) {
        ArchitectureParameters architectureParameters = createGenericArchitectureParameters();
        MobilityParameters mobilityParameters = createGenericMobilityParameters();

        final ContentRequestParameters contentRequestParametersForAppDownlink = createContentParametersForAppDownlink(mobilityParameters.getTrajectoryTime());
        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(1, contentRequestParametersForAppDownlink);

        AbstractSystemLoader systemLoader = new CustomWorkloadSystemLoader(architectureParameters,
                mobilityParameters, parametersPerApp, workloadGenerator);

        systemLoader.loadSystem();

        solution = new Solution(systemLoader.getNetworkArchitecture());
        solution.registerApp(app);

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO | Logging.ERROR);
        final Simulation simulation = Simulation.getInstance();

        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, TRAJECTORY_TIME);
        simulation.startSimulation();
        simulation.restart();
        return solution;
    }

    private ContentRequestParameters createContentParametersForAppDownlink(double requestEndTime) {
        // This is for pull workload.requests
        final int numFileRequests = 0;
        final int numOfFiles = 100;
        final long minFileSize = 10 * MEGABYTE;
        final long maxFileSize = 10 * MEGABYTE; // 2 * MEGABYTE;

        final int numPullDataRequests = 0; // it does not matter as we overwrite the loader

        // This is for push workload.requests
        final int numPushDataRequests = 0;
        final long minDataSizeInPushDataRequests = 0;
        final long maxDataSizeInPushDataRequests = 0;
        return new ContentRequestParameters(1,
                "www.app1.com/",
                numFileRequests,
                numOfFiles,
                minFileSize,
                maxFileSize,
                numPullDataRequests,
                0,
                0,
                numPushDataRequests,
                minDataSizeInPushDataRequests,
                maxDataSizeInPushDataRequests,
                0.0, requestEndTime);
    }
}