package integrationtests.tests;


import core.Solution;
import core.app.App;
import core.processingchain.actions.policies.pool.PushPolicy;
import core.processingchain.actions.pool.PushAction;
import core.processingchain.actions.pool.StoreAction;
import core.processingchain.events.detectors.pool.popularity.PopularityEventDetector;
import core.processingchain.events.pool.HighPopularityEvent;
import core.processingchain.preprocessors.pool.PPOutcomePullRequest;
import edu.ucc.core.Simulation;
import edu.ucc.entities.Content;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.Host;
import edu.ucc.network.devices.StorageReason;
import edu.ucc.network.devices.UserEquipment;
import edu.ucc.utils.FileConstants;
import edu.ucc.utils.Logging;
import edu.ucc.utils.TimeConstants;
import integrationtests.tests.sysloaders.CustomWorkloadSystemLoader;
import integrationtests.tests.sysloaders.ICustomWorkloadGenerator;
import integrationtests.tests.sysloaders.WorkloadGeneratorFactory;
import loader.AbstractSystemLoader;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import workload.mobility.MobilityParameters;
import workload.requests.ContentRequestParameters;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static utils.ScenarioConstants.TRAJECTORY_TIME;
import static utils.Utils.createGenericArchitectureParameters;
import static utils.Utils.createGenericMobilityParameters;

class ScenarioHighPopularityDownlinkStorageAllocationTest {
    @Test
    void executeTestForScenarioFixedSomeServers() {
        final int fortyMegas = 4 * 10 * FileConstants.MEGABYTE;
        final long[] storagePerServer = {fortyMegas, fortyMegas, fortyMegas};
        final int[] serversDeployed = {0, 2};
        App app = new App(1, "app1", "www.app1.com/", 10 * 10 * FileConstants.MEGABYTE, serversDeployed, storagePerServer);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);

        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadForStorageAllocation());

        //
        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();
        final EdgeServer es0 = edgeServers.stream().filter(es -> es.getId() == 0).collect(Collectors.toList()).get(0); // edgeServers.get(0);
        final EdgeServer es1 = edgeServers.stream().filter(es -> es.getId() == 1).collect(Collectors.toList()).get(0);// edgeServers.get(1);
        final EdgeServer es2 = edgeServers.stream().filter(es -> es.getId() == 2).collect(Collectors.toList()).get(0);// edgeServers.get(2);
        //
        assert es0.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es0.getFileSystem().contentExists("www.app1.com/File1.html");
        assert es2.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es2.getFileSystem().contentExists("www.app1.com/File1.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File0.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File1.html");

        // Make sure files exist on es0 and 2 because of right reasons
        StorageReason storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File0.html");
        StorageReason storageReasonEs2 = es2.getFileSystem().getStorageReason("www.app1.com/File0.html");
        Assertions.assertSame(storageReasonEs0, StorageReason.LOCAL_POPULARITY);
        Assertions.assertSame(storageReasonEs2, StorageReason.EXTERNAL_POPULARITY);

        storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File1.html");
        storageReasonEs2 = es2.getFileSystem().getStorageReason("www.app1.com/File1.html");
        Assertions.assertSame(storageReasonEs0, StorageReason.EXTERNAL_POPULARITY);
        Assertions.assertSame(storageReasonEs2, StorageReason.LOCAL_POPULARITY);

        // Make sure space is used as expected. Only storing 4 files
        final App appAtCloud = solution.getAppAtCloud(1);
        long usedSpaceSum = 0;
        for (EdgeServer server : edgeServers) {
            final App edgeServerApp = solution.getAppAtEdgeServer(1, server.getId());
            if (edgeServerApp != null) {
                final List<Content> contentForApp = server.getFileSystem().getContentForApp(1);
                usedSpaceSum += contentForApp.stream().mapToLong(Content::getSize).sum();
            }
        }
        Assertions.assertEquals(appAtCloud.getUsedSpace(), usedSpaceSum);
    }

    @Test
    void executeTestForScenarioFlexibleSomeServers() {
        // What should I see here:
        // 1. Content file0 is popular in the first wave after the third download as the threshold is set as 3 in the ES0
        // we should also see that the file downloaded exists in the file system after the execution and that it is pushed
        // to other ess as well (only in es2 as file0 should be rejected from es1).
        // 2. Content popularity for file1 is seen in es2, again file1 is pushed to es0 and rejected for es1 as app is not running there
        // 3. We need to see that the messages regarding the popularity are received by the cloud
        // 4. We need to see that the available storage space for the app is reduced in servers where the app is running.

        final int[] serversDeployed = {0, 2};
        App app = new App(1, "app1", "www.app1.com/", 10 * 10 * FileConstants.MEGABYTE, serversDeployed);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);

        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadForStorageAllocation());
        //

        // Get ESs this way because random generator affects their labeling.
        UserEquipment ue1 = solution.getNetworkArchitecture().getUserEquipments().get(1);
        UserEquipment ue75 = solution.getNetworkArchitecture().getUserEquipments().get(75);
        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();
        final EdgeServer es0 = (EdgeServer) ue1.getParentUnit().getParentUnit(); //edgeServers.get(0);
        final EdgeServer es2 = (EdgeServer) ue75.getParentUnit().getParentUnit(); //edgeServers.get(2);
        // es1 should be the remaining one.

        EdgeServer es1 = null; // edgeServers.get(1);
        if (es0.getId() == 0 && es2.getId() == 1) {
            es1 = edgeServers.get(2);
        } else if (es0.getId() == 0 && es2.getId() == 2) {
            es1 = edgeServers.get(1);
        } else if (es0.getId() == 1 && es2.getId() == 2) {
            es1 = edgeServers.get(0);
        } else if (es2.getId() == 0 && es0.getId() == 1) {
            es1 = edgeServers.get(2);
        } else if (es2.getId() == 0 && es0.getId() == 2) {
            es1 = edgeServers.get(1);
        } else if (es2.getId() == 1 && es0.getId() == 2) {
            es1 = edgeServers.get(0);
        }

        //
        assert es0.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es0.getFileSystem().contentExists("www.app1.com/File1.html");
        assert es2.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es2.getFileSystem().contentExists("www.app1.com/File1.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File0.html");
        assert !es1.getFileSystem().contentExists("www.app1.com/File1.html");

        // Make sure files exist on es0 and 2 because of right reasons
        StorageReason storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File0.html");
        StorageReason storageReasonEs2 = es2.getFileSystem().getStorageReason("www.app1.com/File0.html");
        Assertions.assertSame(storageReasonEs0, StorageReason.LOCAL_POPULARITY);
        Assertions.assertSame(storageReasonEs2, StorageReason.EXTERNAL_POPULARITY);

        storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File1.html");
        storageReasonEs2 = es2.getFileSystem().getStorageReason("www.app1.com/File1.html");
        Assertions.assertSame(storageReasonEs0, StorageReason.EXTERNAL_POPULARITY);
        Assertions.assertSame(storageReasonEs2, StorageReason.LOCAL_POPULARITY);

        // Make sure space is used as expected. Only storing 4 files
        final App appAtCloud = solution.getAppAtCloud(1);
        long usedSpaceSum = 0;
        for (EdgeServer server : edgeServers) {
            final App edgeServerApp = solution.getAppAtEdgeServer(1, server.getId());
            if (edgeServerApp != null) {
                final List<Content> contentForApp = server.getFileSystem().getContentForApp(1);
                usedSpaceSum += contentForApp.stream().mapToLong(Content::getSize).sum();
                // usedSpaceSum += edgeServerApp.getAvailableSpace();
            }
        }
        // final App appAtEdgeServer0 = solution.getAppAtEdgeServer(1, 0);
        Assertions.assertEquals(appAtCloud.getUsedSpace(), usedSpaceSum);

    }

    @Test
    void executeTestForScenarioFlexibleAllServers() {
        // What should I see here:
        // 1. Content file0 is popular in the first wave after the third download as the threshold is set as 3 in the ES0
        // we should also see that the file downloaded exists in the file system after the execution and that it is pushed
        // to other ess as well
        // 2. Content popularity for file1 should be seen in the other ES2
        // 3. We need to see that the messages regarding the popularity are received by the cloud
        // 4. We need to see that the available storage space for the app is reduced in servers where the app is running.

        App app = new App(1, "app1", "www.app1.com/", 10 * 10 * FileConstants.MEGABYTE);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);

        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadForStorageAllocation());
        //
        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();
        final EdgeServer es0 = edgeServers.get(0);
        final EdgeServer es1 = edgeServers.get(1);
        final EdgeServer es2 = edgeServers.get(2);
        //
        assert es0.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es0.getFileSystem().contentExists("www.app1.com/File1.html");
        assert es2.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es2.getFileSystem().contentExists("www.app1.com/File1.html");
        assert es1.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es1.getFileSystem().contentExists("www.app1.com/File1.html");

        // Make sure files exist on es0 and 2 because of right reasons
        StorageReason storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File0.html");
        StorageReason storageReasonEs1 = es1.getFileSystem().getStorageReason("www.app1.com/File0.html");
        StorageReason storageReasonEs2 = es2.getFileSystem().getStorageReason("www.app1.com/File0.html");
        Assertions.assertSame(storageReasonEs0, StorageReason.LOCAL_POPULARITY);
        Assertions.assertSame(storageReasonEs1, StorageReason.EXTERNAL_POPULARITY);
        Assertions.assertSame(storageReasonEs2, StorageReason.EXTERNAL_POPULARITY);

        storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File1.html");
        storageReasonEs1 = es1.getFileSystem().getStorageReason("www.app1.com/File1.html");
        storageReasonEs2 = es2.getFileSystem().getStorageReason("www.app1.com/File1.html");
        Assertions.assertSame(storageReasonEs0, StorageReason.EXTERNAL_POPULARITY);
        Assertions.assertSame(storageReasonEs1, StorageReason.EXTERNAL_POPULARITY);
        Assertions.assertSame(storageReasonEs2, StorageReason.LOCAL_POPULARITY);

        // Make sure space is used as expected. I guess we store 6 files here
        final App appAtCloud = solution.getAppAtCloud(1);
        long usedSpaceSum = 0;
        for (EdgeServer server : edgeServers) {
            final App edgeServerApp = solution.getAppAtEdgeServer(1, server.getId());
            if (edgeServerApp != null) {
                final List<Content> contentForApp = server.getFileSystem().getContentForApp(1);
                usedSpaceSum += contentForApp.stream().mapToLong(Content::getSize).sum();
                // usedSpaceSum += edgeServerApp.getAvailableSpace();
            }
        }
        // final App appAtEdgeServer0 = solution.getAppAtEdgeServer(1, 0);
        Assertions.assertEquals(appAtCloud.getUsedSpace(), usedSpaceSum);

    }

    @Test
    void executeTestForScenarioFixedAllServers() {
        final int fortyMegas = 4 * 10 * FileConstants.MEGABYTE;
        final long[] storagePerServer = {fortyMegas, fortyMegas, fortyMegas};
        App app = new App(1, "app1", "www.app1.com/", 12 * 10 * FileConstants.MEGABYTE, storagePerServer);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);

        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadForStorageAllocation());
        //
        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();
        //        final EdgeServer es0 = edgeServers.get(0);
        //        final EdgeServer es1 = edgeServers.get(1);
        //        final EdgeServer es2 = edgeServers.get(2);

        // Get ESs this way because random generator affects their labeling.
        UserEquipment ue1 = solution.getNetworkArchitecture().getUserEquipments().get(1);
        UserEquipment ue75 = solution.getNetworkArchitecture().getUserEquipments().get(75);
        final EdgeServer es0 = (EdgeServer) ue1.getParentUnit().getParentUnit(); //edgeServers.get(0);
        final EdgeServer es2 = (EdgeServer) ue75.getParentUnit().getParentUnit(); //edgeServers.get(2);
        // es1 should be the remaining one.

        // int[] usedIndexes = new int[]{es0.getId(), es2.getId()};
        List<Integer> usedIndexes = List.of(es0.getId(), es2.getId());

        var serverIds = edgeServers.stream().mapToInt(Host::getId).boxed().collect(Collectors.toList());
        serverIds.removeAll(usedIndexes);
        int unusedServerId = serverIds.get(0);
        EdgeServer es1 = edgeServers.get(unusedServerId);
        // EdgeServer es1 = null; // edgeServers.get(1);
        //        if (es0.getId() == 0 && es2.getId() == 1) {
        //            es1 = edgeServers.get(2);
        //        } else if (es0.getId() == 0 && es2.getId() == 2) {
        //            es1 = edgeServers.get(1);
        //        } else if (es0.getId() == 1 && es2.getId() == 2) {
        //            es1 = edgeServers.get(0);
        //        } else if (es2.getId() == 0 && es0.getId() == 1) {
        //            es1 = edgeServers.get(2);
        //        } else if (es2.getId() == 0 && es0.getId() == 2) {
        //            es1 = edgeServers.get(1);
        //        } else if (es2.getId() == 1 && es0.getId() == 2) {
        //            es1 = edgeServers.get(0);
        //        }


        //
        assert es0.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es0.getFileSystem().contentExists("www.app1.com/File1.html");
        assert es2.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es2.getFileSystem().contentExists("www.app1.com/File1.html");
        assert es1.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es1.getFileSystem().contentExists("www.app1.com/File1.html");

        // Make sure files exist on es0 and 2 because of right reasons
        StorageReason storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File0.html");
        StorageReason storageReasonEs1 = es1.getFileSystem().getStorageReason("www.app1.com/File0.html");
        StorageReason storageReasonEs2 = es2.getFileSystem().getStorageReason("www.app1.com/File0.html");
        Assertions.assertSame(storageReasonEs0, StorageReason.LOCAL_POPULARITY);
        Assertions.assertSame(storageReasonEs1, StorageReason.EXTERNAL_POPULARITY);
        Assertions.assertSame(storageReasonEs2, StorageReason.EXTERNAL_POPULARITY);

        storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File1.html");
        storageReasonEs1 = es1.getFileSystem().getStorageReason("www.app1.com/File1.html");
        storageReasonEs2 = es2.getFileSystem().getStorageReason("www.app1.com/File1.html");
        Assertions.assertSame(storageReasonEs0, StorageReason.EXTERNAL_POPULARITY);
        Assertions.assertSame(storageReasonEs1, StorageReason.EXTERNAL_POPULARITY);
        Assertions.assertSame(storageReasonEs2, StorageReason.LOCAL_POPULARITY);

        // Make sure space is used as expected. I guess we store 6 files here
        final App appAtCloud = solution.getAppAtCloud(1);
        long usedSpaceSum = 0;
        for (EdgeServer server : edgeServers) {
            final App edgeServerApp = solution.getAppAtEdgeServer(1, server.getId());
            if (edgeServerApp != null) {
                final List<Content> contentForApp = server.getFileSystem().getContentForApp(1);
                usedSpaceSum += contentForApp.stream().mapToLong(Content::getSize).sum();
                // usedSpaceSum += edgeServerApp.getAvailableSpace();
            }
        }
        // final App appAtEdgeServer0 = solution.getAppAtEdgeServer(1, 0);
        Assertions.assertEquals(appAtCloud.getUsedSpace(), usedSpaceSum);
    }

    private Solution executeScenario(App app, ICustomWorkloadGenerator workloadGenerator) {
        ArchitectureParameters architectureParameters = createGenericArchitectureParameters();
        architectureParameters.setNumEdgeServers(3);

        MobilityParameters mobilityParameters = createGenericMobilityParameters();

        final ContentRequestParameters contentRequestParametersForAppDownlink = createContentParametersForAppDownlink(mobilityParameters.getTrajectoryTime());

        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(1, contentRequestParametersForAppDownlink);

        AbstractSystemLoader systemLoader = new CustomWorkloadSystemLoader(architectureParameters,
                mobilityParameters,
                parametersPerApp, workloadGenerator);

        systemLoader.loadSystem();
        final Solution solution = new Solution(systemLoader.getNetworkArchitecture());
        solution.registerApp(app);

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO);
        final Simulation simulation = Simulation.getInstance();
        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, TRAJECTORY_TIME);
        simulation.startSimulation();
        simulation.restart();
        return solution;
    }

    private ContentRequestParameters createContentParametersForAppDownlink(double requestEndTime) {
        // This is for pull workload.requests
        final int numFileRequests = 1;
        final int numOfFiles = 2;
        final long minFileSize = 10 * FileConstants.MEGABYTE;
        final long maxFileSize = 10 * FileConstants.MEGABYTE; // 2 * MEGABYTE;

        final int numPullDataRequests = 0;

        // This is for push workload.requests
        final int numPushDataRequests = 0;
        final long minDataSizeInPushDataRequests = 0;
        final long maxDataSizeInPushDataRequests = 0;
        return new ContentRequestParameters(1,
                "www.app1.com/",
                numFileRequests,
                numOfFiles,
                minFileSize,
                maxFileSize,
                numPullDataRequests,
                0,
                0,
                numPushDataRequests,
                minDataSizeInPushDataRequests,
                maxDataSizeInPushDataRequests,
                0.0, requestEndTime);
    }
}