package integrationtests.tests;


import core.Solution;
import core.app.App;
import core.processingchain.actions.policies.pool.PushPolicy;
import core.processingchain.actions.pool.PushAction;
import core.processingchain.actions.pool.StoreAction;
import core.processingchain.events.detectors.pool.popularity.PopularityEventDetector;
import core.processingchain.events.pool.HighPopularityEvent;
import core.processingchain.preprocessors.pool.PPOutcomePullRequest;
import edu.ucc.core.Simulation;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.network.devices.EdgeServer;
import edu.ucc.network.devices.StorageReason;
import edu.ucc.utils.FileConstants;
import edu.ucc.utils.Logging;
import edu.ucc.utils.TimeConstants;
import integrationtests.tests.sysloaders.CustomWorkloadSystemLoader;
import integrationtests.tests.sysloaders.ICustomWorkloadGenerator;
import integrationtests.tests.sysloaders.WorkloadGeneratorFactory;
import loader.AbstractSystemLoader;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import workload.mobility.MobilityParameters;
import workload.requests.ContentRequestParameters;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static utils.ScenarioConstants.TRAJECTORY_TIME;
import static utils.Utils.createGenericArchitectureParameters;
import static utils.Utils.createGenericMobilityParameters;

class ScenarioHighPopularityDownlinkForControlMessagesTest {

    @Test
    void executeTestForScenario() {
        // What should I see here:
        // 1. Content is popular in the first wave after the third download as the threshold is set as 3
        // we should also see that the file downloaded exists in the file system after the execution and that it is pushed
        // to other ess as well.
        // 2. We need to see that the messages regarding the popularity are received by the cloud
        // 3. We need to see that the available storage space for the app is reduced in all edge servers and the cloud,
        // this will happen only if we receive the store messages at the cloud, and if the cloud propagates that info to edge servers.
        App app = new App(1, "app1", "www.app1.com/", 512 * FileConstants.GIGABYTE);
        app.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(TimeConstants.HOUR, 3));

        app.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        app.addActionForEventType(HighPopularityEvent.class, new PushAction());
        app.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        app.setOpportunisticStorage(true);

        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());

        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();
        final EdgeServer es0 = edgeServers.get(0);
        final EdgeServer es1 = edgeServers.get(1);

        assert es0.getFileSystem().contentExists("www.app1.com/File0.html");
        assert es1.getFileSystem().contentExists("www.app1.com/File0.html");

        final StorageReason storageReasonEs0 = es0.getFileSystem().getStorageReason("www.app1.com/File0.html");
        final StorageReason storageReasonEs1 = es1.getFileSystem().getStorageReason("www.app1.com/File0.html");

        Assertions.assertSame(storageReasonEs0, StorageReason.LOCAL_POPULARITY);
        Assertions.assertSame(storageReasonEs1, StorageReason.EXTERNAL_POPULARITY);
        final App appAtCloud = solution.getAppAtCloud(1);
        final App appAtEdgeServer0 = solution.getAppAtEdgeServer(1, 0);
        Assertions.assertEquals(appAtCloud.getAvailableSpace(), appAtEdgeServer0.getAvailableSpace());

    }

    private Solution executeScenario(App app, ICustomWorkloadGenerator workloadGenerator) {
        ArchitectureParameters architectureParameters = createGenericArchitectureParameters();

        MobilityParameters mobilityParameters = createGenericMobilityParameters();

        final ContentRequestParameters contentRequestParametersForAppDownlink = createContentParametersForAppDownlink(mobilityParameters.getTrajectoryTime());

        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(1, contentRequestParametersForAppDownlink);

        AbstractSystemLoader systemLoader = new CustomWorkloadSystemLoader(architectureParameters,
                mobilityParameters,
                parametersPerApp, workloadGenerator);

        systemLoader.loadSystem();
        final Solution solution = new Solution(systemLoader.getNetworkArchitecture());

        solution.registerApp(app);

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO);
        final Simulation simulation = Simulation.getInstance();


        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, TRAJECTORY_TIME);
        simulation.startSimulation();

        simulation.restart();
        return solution;
    }

    private ContentRequestParameters createContentParametersForAppDownlink(double requestEndTime) {
        // This is for pull workload.requests
        final int numFileRequests = 1;
        final int numOfFiles = 2;
        final long minFileSize = 10 * FileConstants.MEGABYTE;
        final long maxFileSize = 10 * FileConstants.MEGABYTE; // 2 * MEGABYTE;

        final int numPullDataRequests = 0;

        // This is for push workload.requests
        final int numPushDataRequests = 0;
        final long minDataSizeInPushDataRequests = 0;
        final long maxDataSizeInPushDataRequests = 0;
        return new ContentRequestParameters(1,
                "www.app1.com/",
                numFileRequests,
                numOfFiles,
                minFileSize,
                maxFileSize,
                numPullDataRequests,
                0,
                0,
                numPushDataRequests,
                minDataSizeInPushDataRequests,
                maxDataSizeInPushDataRequests,
                0.0, requestEndTime);
    }
}