package integrationtests.tests;

import core.Solution;
import core.app.App;
import core.app.DataRelayPolicy;
import core.processingchain.actions.policies.pool.PushPolicy;
import core.processingchain.actions.pool.NotificationAction;
import core.processingchain.actions.pool.PushAction;
import core.processingchain.metadata.MetadataType;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.ContentReceivedEvent;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.network.devices.CloudUnit;
import edu.ucc.network.devices.Location;
import edu.ucc.utils.Logging;
import integrationtests.tests.sysloaders.CustomWorkloadSystemLoader;
import integrationtests.tests.sysloaders.ICustomWorkloadGenerator;
import integrationtests.tests.sysloaders.WorkloadGeneratorFactory;
import loader.AbstractSystemLoader;
import org.junit.jupiter.api.Test;
import sampleapps.platerecognition.*;
import workload.mobility.MobilityParameters;
import workload.requests.ContentRequestParameters;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static edu.ucc.utils.FileConstants.KILOBYTE;
import static edu.ucc.utils.FileConstants.MEGABYTE;
import static edu.ucc.utils.TimeConstants.HOUR;
import static edu.ucc.utils.TransmissionConstants.KILOMETER;
import static utils.ScenarioConstants.TRAJECTORY_TIME;
import static utils.Utils.createGenericArchitectureParameters;
import static utils.Utils.createGenericMobilityParameters;

public class ScenarioAppNotDeployedAtServerTest {

    private App createAppUpLink(int[] serversDeployed, int totalSpace) {

        App appUpLink = new App(1, "app1", "www.app1.com/", totalSpace, serversDeployed);
        appUpLink.addMetadataExtractor(MetadataType.CONTENT_METADATA, new ContentMetadataExtractorPlateRecognition());

        appUpLink.addPreProcessor(ContentReceivedEvent.class, new PreProcessorPlateRecognition());
        appUpLink.addEventDetector(PPOutcomePlateRecognition.class, new EventDetectorPlateRecognition());
        appUpLink.addMetadataExtractor(MetadataType.INTERNAL_EVENT_METADATA, new EventMetadataExtractorPlateRecognition());

        appUpLink.addActionForEventType(PlateRecognitionEvent.class, new NotificationAction());
        appUpLink.setExternalSubscriberPerEventType(PlateRecognitionEvent.class,
                List.of("http://www.policedepartment.com", "http://www.fineticketingsystem.com"));
        appUpLink.addActionForEventType(PlateRecognitionEvent.class, new PushAction());
        appUpLink.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));

        return appUpLink;
    }


    @Test
    void executeTestAppImmediateProcessingDeferredToCloud() {
        final int[] serversDeployed = {0, 2};
        final int totalSpace = 10 * 10 * MEGABYTE;
        App app = createAppUpLink(serversDeployed, totalSpace);
        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadForAppNotDeployedAtServerScenario(), false);
        //        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();

        // As we are deploying all the apps across all of edge servers, we need to see that used space is 0 and that available space is all for all of them.
        App cloudApp = solution.getAppAtCloud(1);
        assert cloudApp.getUsedSpace() == 0;
        // Space used in the cloud is not subtracted from app's storage quota
        assert cloudApp.getTotalSpace() == totalSpace;
    }

    @Test
    void executeTestAppScheduledProcessingDeferredToCloud() {
        final int[] serversDeployed = {0, 2};
        final int totalSpace = 10 * 10 * MEGABYTE;
        App app = createAppUpLink(serversDeployed, totalSpace);
        app.setProcessingInterval(TRAJECTORY_TIME - 1 * HOUR);

        // final Simulation simulation = executeScenario(app, WorkloadGeneratorFactory.workloadHighPopularityDownlinkForControlMessagesScenario());
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadForAppNotDeployedAtServerScenario(), false);
        //        final List<EdgeServer> edgeServers = solution.getNetworkArchitecture().getEdgeServers();

        // As we are deploying all the apps across all of edge servers, we need to see that used space is 0 and that available space is all for all of them.
        App cloudApp = solution.getAppAtCloud(1);
        assert cloudApp.getUsedSpace() == 0;
        assert cloudApp.getTotalSpace() == totalSpace;
    }


    @Test
    void executeTestAppImmediateProcessingDeferredToClosestHost() {
        final int[] serversDeployed = {0, 2};
        final int totalSpace = 10 * 10 * MEGABYTE;
        App app = createAppUpLink(serversDeployed, totalSpace);
        app.setDataRelayPolicy(DataRelayPolicy.RELAY_TO_CLOSEST_HOST);
        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadForAppNotDeployedAtServerScenario(), true);

        // As we are deploying all the apps across all of edge servers, we need to see that used space is 0 and that available space is all for all of them.
        App cloudApp = solution.getAppAtCloud(1);
        assert cloudApp.getUsedSpace() == 0;
        assert cloudApp.getTotalSpace() == totalSpace;
    }

    @Test
    void executeTestAppScheduledProcessingDeferredToClosestHost() {
        final int[] serversDeployed = {0, 2};
        final int totalSpace = 10 * 10 * MEGABYTE;
        App app = createAppUpLink(serversDeployed, totalSpace);
        app.setProcessingInterval(TRAJECTORY_TIME - 1 * HOUR);
        app.setDataRelayPolicy(DataRelayPolicy.RELAY_TO_CLOSEST_HOST);

        final Solution solution = executeScenario(app, WorkloadGeneratorFactory.workloadForAppNotDeployedAtServerScenario(), true);

        App cloudApp = solution.getAppAtCloud(1);
        assert cloudApp.getUsedSpace() == 0;
        assert cloudApp.getTotalSpace() == totalSpace;
    }

    private Solution executeScenario(App app, ICustomWorkloadGenerator workloadGenerator, boolean putCloudFarther) {
        ArchitectureParameters architectureParameters = createGenericArchitectureParameters();
        architectureParameters.setNumEdgeServers(3);

        MobilityParameters mobilityParameters = createGenericMobilityParameters();

        final ContentRequestParameters contentRequestParametersForAppUplink = createContentParametersForAppUplink(mobilityParameters.getTrajectoryTime());

        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(1, contentRequestParametersForAppUplink);

        AbstractSystemLoader systemLoader = new CustomWorkloadSystemLoader(architectureParameters, mobilityParameters, parametersPerApp, workloadGenerator);

        systemLoader.loadSystem();
        if (putCloudFarther)
            moveCloudFarther(systemLoader.getNetworkArchitecture().getCloudRoot());
        final Solution solution = new Solution(systemLoader.getNetworkArchitecture());

        solution.registerApp(app);

        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO | Logging.ERROR);
        final Simulation simulation = Simulation.getInstance();

        simulation.init(systemLoader.getNetworkArchitecture(), systemLoader.getWorkload(), solution, TRAJECTORY_TIME);
        simulation.startSimulation();
        simulation.restart();
        return solution;
    }

    private void moveCloudFarther(CloudUnit cloud) {
        // 250 km (~ Dublin)
        double requiredDistance = 1000 * KILOMETER;
        double coordinate = requiredDistance / Math.sqrt(2);
        final Location cloudLocation = new Location(-coordinate, -coordinate);
        cloud.setLocation(cloudLocation);

    }

    private ContentRequestParameters createContentParametersForAppUplink(double requestEndTime) {
        // This is for pull workload.requests
        final int numFileRequests = 0;
        final int numOfFiles = 0;
        final long minFileSize = MEGABYTE;
        final long maxFileSize = MEGABYTE; // 2 * MEGABYTE;

        final int numPullDataRequests = 0;
        final long minDataSizeInPullDataRequests = 0;
        final long maxDataSizeInPullDataRequests = 0;

        // This is for push workload.requests
        final int numPushDataRequests = 1;
        final long minDataSizeInPushDataRequests = 500 * KILOBYTE;
        final long maxDataSizeInPushDataRequests = 5 * MEGABYTE;
        return new ContentRequestParameters(2, "www.app2.com/", numFileRequests, numOfFiles, minFileSize,
                maxFileSize, numPullDataRequests, minDataSizeInPullDataRequests,
                maxDataSizeInPullDataRequests, numPushDataRequests, minDataSizeInPushDataRequests, maxDataSizeInPushDataRequests,
                0.0, requestEndTime);
    }
}
