import core.Solution;
import core.app.App;
import core.processingchain.actions.policies.pool.PushPolicy;
import core.processingchain.actions.pool.NotificationAction;
import core.processingchain.actions.pool.PushAction;
import core.processingchain.actions.pool.StoreAction;
import core.processingchain.events.detectors.pool.popularity.PopularityEventDetector;
import core.processingchain.events.pool.HighPopularityEvent;
import core.processingchain.events.pool.PopularityEvent;
import core.processingchain.metadata.MetadataType;
import core.processingchain.metadata.extractor.MetadataExtractor;
import core.processingchain.metadata.pool.BaseMetadata;
import core.processingchain.metadata.pool.PopularityEventMetadata;
import core.processingchain.preprocessors.pool.PPOutcomePullRequest;
import edu.ucc.core.Simulation;
import edu.ucc.core.events.simulationevents.ContentReceivedEvent;
import edu.ucc.network.architecture.ArchitectureParameters;
import edu.ucc.statisticscollection.StatisticsCalculator;
import edu.ucc.utils.FileConstants;
import edu.ucc.utils.Logging;
import loader.AbstractSystemLoader;
import loader.SystemLoader;
import sampleapps.platerecognition.*;
import workload.mobility.MobilityParameters;
import workload.requests.ContentRequestParameters;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static edu.ucc.utils.FileConstants.KILOBYTE;
import static edu.ucc.utils.FileConstants.MEGABYTE;
import static edu.ucc.utils.TimeConstants.HOUR;
import static edu.ucc.utils.TimeConstants.MINUTE;
import static utils.ScenarioConstants.NUM_USER_EQUIPMENTS;
import static utils.ScenarioConstants.TRAJECTORY_TIME;
import static utils.Utils.createGenericArchitectureParameters;
import static utils.Utils.createGenericMobilityParameters;

/**
 * Sample class to demonstrate how to create scenarios to simulate with CONTENTO
 */
public class ExperimentLauncher {
    public static void main(String[] args) {
        final String baseDir = System.getProperty("user.home") + File.separator + "desktop\\contento-output" + File.separator;

        final String outputNumberRecognitionExperiment = baseDir + "plate-number-recognition" + File.separator;
        new ExperimentLauncher().executePlateRecognitionScenario(outputNumberRecognitionExperiment);

        final String outputEdgeCDNVaryingReqsExperiment = baseDir + "edge-cdn-varying-requests" + File.separator;
        new ExperimentLauncher().executeEdgeCdnWithVaryingRequests(outputEdgeCDNVaryingReqsExperiment);

        final String outputEdgeCDNVaryingUEsExperiment = baseDir + "edge-cdn-varying-ues" + File.separator;
        new ExperimentLauncher().executeEdgeCDNWithVaryingUEs(outputEdgeCDNVaryingUEsExperiment);
    }

    /**
     * This methods creates an edge CDN app by attaching the corresponding extractors, event detectors, and actions.
     *
     * @return An app that will simulate an edge cdn system that caches popular content and pushes it to neighbour servers.
     */
    private App createEdgeCdnApp() {
        App appDownLink = new App(1, "app1", "www.app1.com/", 512 * FileConstants.GIGABYTE);
        appDownLink.addEventDetector(PPOutcomePullRequest.class, new PopularityEventDetector(HOUR, 3));
        appDownLink.addMetadataExtractor(MetadataType.INTERNAL_EVENT_METADATA, new MetadataExtractor<PopularityEvent>() {
            @Override
            public BaseMetadata extractMetadata(PopularityEvent hpEvent) {
                boolean highPopularity = hpEvent instanceof HighPopularityEvent;
                return new PopularityEventMetadata(hpEvent.getAppId(), hpEvent.getTestbedEvent(), highPopularity);
            }
        });
        appDownLink.addActionForEventType(HighPopularityEvent.class, new StoreAction());
        appDownLink.addActionForEventType(HighPopularityEvent.class, new PushAction());
        appDownLink.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        appDownLink.setOpportunisticStorage(true);
        return appDownLink;
    }

    /**
     * Helper method to create the parameters for the request workload for the edge cdn app. The requests are produced
     * randomly (files are created with random sizes, random files are generated, and requests are scheduled for random times).
     * The {@link integrationtests.tests.sysloaders.CustomWorkloadSystemLoader} class can be used to generate custom workloads.
     * @param numFileRequests Number of file requests to create
     * @param numOfFiles Number of files the generated requests can refer to
     * @param requestEndTime The final mark for the time period in which file requests can be produced (the initial mark is 0.0)
     * @return An object with the parameters to create a workload
     */
    private ContentRequestParameters createContentParametersForEdgeCDNApp(int numFileRequests, int numOfFiles, double requestEndTime) {
        // This is for pull workload.requests
        final long minFileSize = KILOBYTE;
        final long maxFileSize = 2 * MEGABYTE;

        // No requests for UE data in this scenario
        final int numPullDataRequests = 0;
        final long minDataSizeInPullDataRequests = KILOBYTE;
        final long maxDataSizeInPullDataRequests = 10 * KILOBYTE;

        // This is for push workload.requests
        final int numPushDataRequests = 0;
        final long minDataSizeInPushDataRequests = 0;
        final long maxDataSizeInPushDataRequests = 0;
        return new ContentRequestParameters(1, "www.app1.com/",
                numFileRequests, numOfFiles, minFileSize, maxFileSize, numPullDataRequests,
                minDataSizeInPullDataRequests, maxDataSizeInPullDataRequests, numPushDataRequests,
                minDataSizeInPushDataRequests, maxDataSizeInPushDataRequests, 0.0, requestEndTime);
    }

    /**
     * This method creates an App that operates with CameraData (for the car plate number recognition scenario).
     * See that custom preprocessors and event detectors are employed. See also that URIs are defined to simulate a
     * call to external web services after an event is detected.
     * @return An App focused on the Uplink.
     */
    private App createIoTApp() {
        App iotApp = new App(1, "app2", "www.app2.com/", 512 * FileConstants.GIGABYTE);
        iotApp.addMetadataExtractor(MetadataType.CONTENT_METADATA, new ContentMetadataExtractorPlateRecognition());

        iotApp.addPreProcessor(ContentReceivedEvent.class, new PreProcessorPlateRecognition());
        iotApp.addEventDetector(PPOutcomePlateRecognition.class, new EventDetectorPlateRecognition());
        iotApp.addMetadataExtractor(MetadataType.INTERNAL_EVENT_METADATA, new EventMetadataExtractorPlateRecognition());

        iotApp.addActionForEventType(PlateRecognitionEvent.class, new NotificationAction());
        iotApp.setExternalSubscriberPerEventType(PlateRecognitionEvent.class,
                List.of("http://www.policedepartment.com", "http://www.fineticketingsystem.com"));
        iotApp.addActionForEventType(PlateRecognitionEvent.class, new PushAction());
        iotApp.setPolicyForActionType(PushAction.class, new PushPolicy(false, true, false));
        iotApp.setProcessingInterval(TRAJECTORY_TIME);
        return iotApp;
    }

    /**
     * This method defines the parameters to create the content requests (push data requests) for the plate number
     * recognition app.
     * @param requestEndTime The final mark for the time period in which file requests can be produced (the initial mark is 0.0)
     * @return An object with the parameters to create a workload
     */
    private ContentRequestParameters createContentParametersForIoTApp(double requestEndTime) {
        // No pull requests in this workload
        final int numFileRequests = 0;
        final int numOfFiles = 0;
        final long minFileSize = MEGABYTE;
        final long maxFileSize = MEGABYTE; // 2 * MEGABYTE;

        final int numPullDataRequests = 0;
        final long minDataSizeInPullDataRequests = 0;
        final long maxDataSizeInPullDataRequests = 0;

        // This is for push workload.requests
        final int numPushDataRequests = 2;
        final long minDataSizeInPushDataRequests = 500 * KILOBYTE;
        final long maxDataSizeInPushDataRequests = 5 * MEGABYTE;
        return new ContentRequestParameters(2, "www.app2.com/", numFileRequests, numOfFiles, minFileSize,
                maxFileSize, numPullDataRequests, minDataSizeInPullDataRequests,
                maxDataSizeInPullDataRequests, numPushDataRequests, minDataSizeInPushDataRequests, maxDataSizeInPushDataRequests,
                0.0, requestEndTime);
    }

    /**
     * This experiment measures the network performance (speed and traffic) in an Edge CDN scenario when we vary
     * the number of requests from a fixed number of UEs
     *
     * @param outputDirPath The path to write result files
     */
    private void executeEdgeCdnWithVaryingRequests(String outputDirPath) {
        // We disable logging in output to speed up execution
        Logging.configureLogging(0);
        // Get access to simulation (testbed)
        final Simulation simulation = Simulation.getInstance();

        final int appId = 1;

        // Different number of request to be made by UEs
        final int[] requests = new int[]{1000, 2000, 5000, 10000, 20000, 50000};
        final int numOfGeneratedFiles = 1000; // This is the "Universe" of available files to select for requests.

        // 1. Define edu.ucc.network's architecture
        final ArchitectureParameters architectureParameters = createGenericArchitectureParameters();
        final MobilityParameters mobilityParameters = createGenericMobilityParameters();

        for (int requestsThisTrial : requests) {
            // 2. Define workload per trial. See that we are changing the number of requests in each trial.
            final ContentRequestParameters contentRequestParameters =
                    createContentParametersForEdgeCDNApp(requestsThisTrial, numOfGeneratedFiles, mobilityParameters.getTrajectoryTime());
            final Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
            parametersPerApp.put(appId, contentRequestParameters);

            // 3. Create apps, specifying/selecting the preprocessors, event detectors and policies.
            final App cdnApp = createEdgeCdnApp();

            // 4. Create the solution, linking it with the testbed
            AbstractSystemLoader abstractSystemLoader = new SystemLoader(architectureParameters, mobilityParameters, parametersPerApp);
            abstractSystemLoader.loadSystem();
            final Solution solution = new Solution(abstractSystemLoader.getNetworkArchitecture());

            // 5. Register apps with the solution
            solution.registerApp(cdnApp);

            // 6. Initiate the queue of events within the workload (workload.requests + workload.mobility)
            simulation.init(abstractSystemLoader.getNetworkArchitecture(), abstractSystemLoader.getWorkload(), solution, TRAJECTORY_TIME);

            // 7. Launch simulation
            simulation.startSimulation();

            // 8. Write results by the statistics collector / calculator
            StatisticsCalculator statisticsCalculator = new StatisticsCalculator(simulation.getStatisticsCollector(), simulation.getWorkload(), simulation.getRunningTime());
            statisticsCalculator.writeStatistics(outputDirPath, NUM_USER_EQUIPMENTS, requestsThisTrial, numOfGeneratedFiles, 0);
            simulation.restart();
        }
    }

    /**
     * This experiment measures the network performance (speed and traffic) in an Edge CDN scenario when we vary
     * the number of participant UEs
     *
     * @param outputDirPath The path to write result files
     */
    private void executeEdgeCDNWithVaryingUEs(String outputDirPath) {
        // We disable logging in output to speed up execution
        Logging.configureLogging(0);

        // Get access to simulation (testbed)
        final Simulation simulation = Simulation.getInstance();
        final int appId = 1;

        // Different number of UEs to run the experiment
        final int[] ues = new int[]{400, 500, 600};
        final int numFileRequests = 1000 * 12; // This is the number of files requests to be made from UEs.
        final int numOfGeneratedFiles = 10000; // This is the "Universe" of available files to select for requests.

        // 1. Define edu.ucc.network's architecture
        final ArchitectureParameters architectureParameters = createGenericArchitectureParameters();
        final MobilityParameters mobilityParameters = createGenericMobilityParameters();
        // 2. Define workload per individual app
        final ContentRequestParameters contentRequestParameters = createContentParametersForEdgeCDNApp(numFileRequests,
                numOfGeneratedFiles, mobilityParameters.getTrajectoryTime());
        final Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(appId, contentRequestParameters);
        for (int numUes : ues) {
            // we adjust the architecture parameter by setting the number of UEs for each trial
            architectureParameters.setNumUserEquipments(numUes);

            // 3. Create apps, specifying/selecting the preprocessors, event detectors and policies.
            final App cdnApp = createEdgeCdnApp();

            // 4. Create the solution, linking it with the testbed
            AbstractSystemLoader abstractSystemLoader = new SystemLoader(architectureParameters, mobilityParameters, parametersPerApp);
            abstractSystemLoader.loadSystem();
            final Solution solution = new Solution(abstractSystemLoader.getNetworkArchitecture());

            // 5. Register apps with the solution
            solution.registerApp(cdnApp);

            // 6. Initiate the queue of events within the workload (workload.requests + workload.mobility)
            simulation.init(abstractSystemLoader.getNetworkArchitecture(), abstractSystemLoader.getWorkload(), solution, TRAJECTORY_TIME);

            // 7. Launch simulation
            simulation.startSimulation();

            // 8. Write results by the statistics collector / calculator
            StatisticsCalculator statisticsCalculator = new StatisticsCalculator(simulation.getStatisticsCollector(), simulation.getWorkload(), simulation.getRunningTime());
            statisticsCalculator.writeStatistics(outputDirPath, numUes, numFileRequests, numOfGeneratedFiles, 0);
            statisticsCalculator.presentStatistics();
            simulation.restart();
        }
    }

    /**
     * +
     *
     * @param outputDirPath The path to write result files.
     */
    private void executePlateRecognitionScenario(String outputDirPath) {
        // This configures the output shown in the log (standard output, console)
        Logging.configureLogging(Logging.WARNING | Logging.CONTENT_RECEIVED | Logging.INFO | Logging.PUSH_DATA_REQUEST);
        // Get access to simulation (testbed)
        final Simulation simulation = Simulation.getInstance();
        final int appId = 1;

        // 1. Define edu.ucc.network's architecture
        ArchitectureParameters architectureParameters = createGenericArchitectureParameters();
        MobilityParameters mobilityParameters = createGenericMobilityParameters();

        // We can disable mobility with the following line. Devices would still have an initial location to connect
        // with base stations.
        mobilityParameters.setMobilityEnabled(false);

        // 2. Define workload per individual app
        final ContentRequestParameters requestParametersForIoTApp = createContentParametersForIoTApp(mobilityParameters.getTrajectoryTime());
        Map<Integer, ContentRequestParameters> parametersPerApp = new HashMap<>();
        parametersPerApp.put(appId, requestParametersForIoTApp);

        // 3. Create apps, specifying/selecting the preprocessors, event detectors and policies.
        App iotApp = createIoTApp();

        // 4. Create the solution, linking it with the testbed
        AbstractSystemLoader abstractSystemLoader = new SystemLoader(architectureParameters, mobilityParameters, parametersPerApp);

        abstractSystemLoader.loadSystem();
        final Solution solution = new Solution(abstractSystemLoader.getNetworkArchitecture());

        // 5. Register apps with the solution
        solution.registerApp(iotApp);

        // 6. Initiate the queue of events within the workload (workload.requests + workload.mobility)
        simulation.init(abstractSystemLoader.getNetworkArchitecture(), abstractSystemLoader.getWorkload(), solution, TRAJECTORY_TIME + MINUTE);

        // 7. Launch simulation
        simulation.startSimulation();

        // 8. Write results by the statistics collector / calculator
        StatisticsCalculator statisticsCalculator = new StatisticsCalculator(simulation.getStatisticsCollector(), simulation.getWorkload(), simulation.getRunningTime());
        statisticsCalculator.writeStatistics(outputDirPath, architectureParameters.getNumUserEquipments(), 0, 0, requestParametersForIoTApp.getNumPushDataRequests());
        simulation.restart();
    }
}
